// ===========================================================================
// CONTENT  : CLASS JsonUtil
// AUTHOR   : Manfred Duchrow
// VERSION  : 2.0 - 28/07/2019
// HISTORY  :
//  26/08/2014  mdu  CREATED
//  13/07/2019  mdu   changed -> Replaced all StringBuffer by Appendable
//
// Copyright (c) 2014-2019, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text.json;

import static org.pfsw.bif.text.IJSONConvertible.*;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.Map;

import org.pfsw.bif.exceptions.IORuntimeException;
import org.pfsw.bif.text.IJSONConvertible;
import org.pfsw.bif.text.IStringRepresentation;

/**
 * Convenience methods for JavaScript Object Notation (JSON) handling.
 * {@link  <a href="http://www.json.org/" target="blank">JSON Web Site</a>}
 *
 * @author Manfred Duchrow
 * @version 2.0
 */
public class JsonUtil
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  /**
   * The singleton of this class (i.e. JsonUtil.current()) which can easily included
   * by static import and the used like JU.toJSONStringLiteral(object).
   */
  public static final JsonUtil JU = new JsonUtil();

  private static final Class<?>[] JSON_TYPES = { JsonObject.class, JsonArray.class, String.class, Boolean.class, Integer.class, Long.class, BigDecimal.class };

  // =========================================================================
  // CLASS VARIABLES
  // =========================================================================
  private static JsonUtil soleInstance = JU;

  // =========================================================================
  // CLASS METHODS
  // =========================================================================
  /**
   * Returns the only instance this class supports (design pattern "Singleton")
   */
  public static JsonUtil current()
  {
    return soleInstance;
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  protected JsonUtil()
  {
    super();
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Converts the given object to a valid JSON string representation.
   * @param jsonConvertible The object to convert.
   * @return The JSON string representation
   */
  public String convertToJSON(IJSONConvertible jsonConvertible)
  {
    StringBuffer buffer;

    if (jsonConvertible == null)
    {
      return IJSONConvertible.JSON_LITERAL_NULL;
    }
    buffer = new StringBuffer(100);
    jsonConvertible.appendAsJSONString(buffer);
    return buffer.toString();
  }

  /**
   * Converts the given object array to a valid JSON string representation.
   * 
   * @param objects The object array to convert.
   * @return The JSON string representation
   */
  public String arrayToJSON(Object... objects)
  {
    StringBuffer buffer;

    buffer = new StringBuffer(50 * objects.length);
    this.appendJSONArray(buffer, objects);
    return buffer.toString();
  }

  /**
   * Converts the given map to a valid JSON string representation.
   * 
   * @param map The map to convert.
   * @return The JSON string representation
   */
  public String mapToJSON(Map<String, Object> map)
  {
    StringBuffer buffer;

    if (map == null)
    {
      return JSON_LITERAL_NULL;
    }
    buffer = new StringBuffer(50 * map.size());
    this.appendJSONMap(buffer, map);
    return buffer.toString();
  }

  /**
   * Returns the given string as JSON string literal (i.e. enclosed in quotes).
   * 
   * @param str The string to make JSON compatible (might by null)
   * @return The quoted string or "null" string without quotes for str being null.
   */
  public String toJSONStringLiteral(String str)
  {
    if (str == null)
    {
      return JSON_LITERAL_NULL;
    }
    return JSON_STRING_DELIMITER + str + JSON_STRING_DELIMITER;
  }

  /**
   * Appends the given name and value as JSON pair member to the given buffer.
   * It produces a compact output, which means no extra spaces are inserted.
   *  
   * @param output The buffer to append to.
   * @param name The name of the pair.
   * @param value The value of the pair.
   * @throws IORuntimeException if the given output object throws an IOException.
   */
  public void appendJSONPair(Appendable output, String name, Object value)
  {
    appendJsonPair(output, name, value, true);
  }

  /**
   * Appends the given name and value as JSON pair member to the given buffer.
   * For better readability extra spaces are inserted around the name/value separator.
   *  
   * @param output The buffer to append to.
   * @param name The name of the pair.
   * @param value The value of the pair.
   * @throws IORuntimeException if the given output object throws an IOException.
   */
  public void appendJsonPair(Appendable output, String name, Object value)
  {
    appendJsonPair(output, name, value, false);
  }
  
  /**
   * Appends the given name and value as JSON pair member to the given output.
   *  
   * @param output The buffer to append to.
   * @param name The name of the pair.
   * @param value The value of the pair.
   * @param compact if true, no spaces are inserted between name and separator and separator and value.
   * @throws IORuntimeException if the given output object throws an IOException.
   */
  public void appendJsonPair(Appendable output, String name, Object value, boolean compact)
  {
    try
    {
      this.appendJSONString(output, name);
      if (!compact)
      {        
        output.append(" ");
      }
      output.append(JSON_PAIR_SEPARATOR);
      if (!compact)
      {        
        output.append(" ");
      }
      this.appendJSONObject(output, value);      
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }
  
  /**
   * Appends the given string to the buffer as a valid JSON string literal.
   * That is, surrounded by quotes.
   * 
   * @throws IORuntimeException if the given output object throws an IOException.
   */
  public void appendJSONString(Appendable output, String str)
  {
    try
    {
      output.append(JSON_STRING_DELIMITER);
      output.append(str);
      output.append(JSON_STRING_DELIMITER);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Converts the given object to a valid JSON string representation and appends 
   * it to the given buffer.
   * 
   * @param jsonConvertible The object to convert and append.
   * @throws IORuntimeException if the given output object throws an IOException.
   */
  public void appendJSONConvertible(Appendable output, IJSONConvertible jsonConvertible)
  {
    if (jsonConvertible == null)
    {
      try
      {
        output.append(IJSONConvertible.JSON_LITERAL_NULL);
      }
      catch (IOException e)
      {
        throw new IORuntimeException(e);
      }
    }
    else
    {
      jsonConvertible.appendAsJSONString(output);
    }
  }

  /**
   * Appends the given object array to the given output in a valid JSON 
   * string representation.
   * 
   * @param objects The object array to append.
   * @throws IORuntimeException if the given output object throws an IOException.
   */
  public void appendJSONArray(Appendable output, Object... objects)
  {
    boolean isFirst = true;

    try
    {
      output.append(JSON_ARRAY_START);
      for (Object object : objects)
      {
        if (isFirst)
        {
          isFirst = false;
        }
        else
        {
          output.append(JSON_ELEMENT_SEPARATOR);
        }
        this.appendJSONObject(output, object);
      }
      output.append(JSON_ARRAY_END);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Appends the given object to the output as a valid JSON string.
   * @throws IORuntimeException if the given output object throws an IOException.
   */
  public void appendJSONObject(Appendable output, Object object)
  {
    try
    {
      if (object == null)
      {
        output.append(JSON_LITERAL_NULL);
      }
      else if (object instanceof String)
      {
        this.appendJSONString(output, (String)object);
      }
      else if (object instanceof IJSONConvertible)
      {
        IJSONConvertible convertible = (IJSONConvertible)object;
        this.appendJSONConvertible(output, convertible);
      }
      else if (object instanceof Number)
      {
        output.append(object.toString());
      }
      else if (object instanceof Character)
      {
        Character ch = (Character)object;
        this.appendJSONString(output, String.valueOf(ch.charValue()));
      }
      else if (object instanceof Boolean)
      {
        Boolean bool = (Boolean)object;
        output.append(bool.booleanValue() ? JSON_LITERAL_TRUE : JSON_LITERAL_FALSE);
      }
      else if (object instanceof IStringRepresentation)
      {
        this.appendJSONString(output, ((IStringRepresentation)object).asString());
      }
      else
      {
        this.appendJSONString(output, object.toString());
      }
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Adds the given map to the buffer as JSON representation.
   * <p/>
   * Example:<br/>
   * {"key1":"value1","key2":"value2","key3":"value3"}
   * 
   * @throws IORuntimeException if the given output object throws an IOException.
   */
  public void appendJSONMap(Appendable output, Map<String, Object> map)
  {
    boolean isFirst = true;
    
    try
    {
      if (map == null)
      {
        output.append(JSON_LITERAL_NULL);
        return;
      }
      output.append(JSON_OBJECT_START);
      for (Map.Entry<String, Object> entry : map.entrySet())
      {
        if (isFirst)
        {
          isFirst = false;
        }
        else
        {
          output.append(JSON_ELEMENT_SEPARATOR);
        }
        this.appendJSONPair(output, entry.getKey(), entry.getValue());
      }
      output.append(JSON_OBJECT_END);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Tries to convert the given object to a JSON string.
   * If the object is not any of the following then its toString() result will
   * be returned.
   * <ul>
   * <li>null</li>
   * <li>IJSONConvertible</li>
   * <li>String</li>
   * <li>Character</li>
   * <li>Boolean</li>
   * <li>Number</li>
   * </ul>
   * 
   * @param object Any object (even null).
   * @return A JSON value representing the given object
   */
  public String objectToJSONValue(Object object)
  {
    StringBuffer buffer;

    buffer = new StringBuffer(100);
    this.appendJSONObject(buffer, object);
    return buffer.toString();
  }

  /**
   * Returns true if the given object is null or an instance of one of the 
   * supported JSON object types.
   * <p>
   * Valid types are:
   * <ul>
   * <li>JsonObject</li>
   * <li>JsonArray</li>
   * <li>String</li>
   * <li>Boolean</li>
   * <li>Integer</li>
   * <li>Long</li>
   * <li>BigDecimal</li>
   * </ul> 
   */
  public boolean isValidJsonTypeInstance(Object object)
  {
    if (object == null)
    {
      return true;
    }
    return this.isValidJsonType(object.getClass());
  }

  /**
   * Returns true if the given class one of the supported JSON object types.
   * <p>
   * Valid types are:
   * <ul>
   * <li>JsonObject</li>
   * <li>JsonArray</li>
   * <li>String</li>
   * <li>Boolean</li>
   * <li>Integer</li>
   * <li>Long</li>
   * <li>BigDecimal</li>
   * </ul> 
   */
  public boolean isValidJsonType(Class<?> type)
  {
    if (type == null)
    {
      return false;
    }
    for (Class<?> jsonType : JSON_TYPES)
    {
      if (type == jsonType)
      {
        return true;
      }
    }
    return false;
  }
}
