// ===========================================================================
// CONTENT  : CLASS JsonObjectAccessor
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.2 - 08/03/2020
// HISTORY  :
//  08/01/2017  mdu  CREATED
//  16/02/2017  mdu   added --> getRequiredStringArrayValue(), getOptionalStringArrayValue()
//  08/03/2020  mdu   added --> support IStringConstant for field names
//
// Copyright (c) 2017-2020, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text.json;

import java.math.BigDecimal;

import org.pfsw.bif.text.IStringConstant;
import org.pfsw.text.NaturalNumber;
import org.pfsw.text.TimeUnit;
import org.pfsw.text.TimeValue;

/**
 * This is a wrapper around a single JsonObject. It supports access to the 
 * objects's field values in a convenient way and supports optional and required
 * field access as well as implicit type conversion.
 * All access problems are signaled by {@link JsonAccessException} which is a RuntimeException.
 *
 * @author Manfred Duchrow
 * @version 1.2
 */
public class JsonObjectAccessor
{
  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private final JsonObject jsonObject;

  // =========================================================================
  // STATIC METHODS
  // =========================================================================
  public static JsonObjectAccessor create(final JsonObject jsonObject)
  {
    return new JsonObjectAccessor(jsonObject);
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  public JsonObjectAccessor(final JsonObject jsonObject)
  {
    super();
    this.jsonObject = jsonObject;
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  public JsonObject getJsonObject()
  {
    return this.jsonObject;
  }

  public JsonObjectAccessor getRequiredJsonObjectAccessorValue(final IStringConstant name)
  {
    return getRequiredJsonObjectAccessorValue(name.asString());
  }

  public JsonObjectAccessor getRequiredJsonObjectAccessorValue(final String name)
  {
    JsonObject object;

    object = this.getRequiredJsonObjectValue(name);
    return new JsonObjectAccessor(object);
  }

  public JsonObject getRequiredJsonObjectValue(final IStringConstant name)
  {
    return getRequiredJsonObjectValue(name.asString());
  }

  public JsonObject getRequiredJsonObjectValue(final String name)
  {
    return this.getRequiredTypedValue(name, JsonObject.class);
  }

  public JsonArray getRequiredJsonArrayValue(final IStringConstant name)
  {
    return getRequiredJsonArrayValue(name.asString());
  }

  public JsonArray getRequiredJsonArrayValue(final String name)
  {
    return this.getRequiredTypedValue(name, JsonArray.class);
  }

  public String getRequiredStringValue(final IStringConstant name)
  {
    return getRequiredStringValue(name.asString());
  }

  public String getRequiredStringValue(final String name)
  {
    return this.getRequiredTypedValue(name, String.class);
  }

  public String[] getRequiredStringArrayValue(final IStringConstant name)
  {
    return getRequiredStringArrayValue(name.asString());
  }

  public String[] getRequiredStringArrayValue(final String name)
  {
    return this.getStringArray(name, true);
  }

  public Boolean getRequiredBooleanValue(final IStringConstant name)
  {
    return getRequiredBooleanValue(name.asString());
  }

  public Boolean getRequiredBooleanValue(final String name)
  {
    return this.getRequiredTypedValue(name, Boolean.class);
  }

  public Integer getRequiredIntegerValue(final IStringConstant name)
  {
    return getRequiredIntegerValue(name.asString());
  }

  public Integer getRequiredIntegerValue(final String name)
  {
    return this.getRequiredTypedValue(name, Integer.class);
  }

  public Long getRequiredLongValue(final IStringConstant name)
  {
    return getRequiredLongValue(name.asString());
  }

  public Long getRequiredLongValue(final String name)
  {
    return this.getRequiredTypedValue(name, Long.class);
  }

  public BigDecimal getRequiredBigDecimalValue(final IStringConstant name)
  {
    return getRequiredBigDecimalValue(name.asString());
  }

  public BigDecimal getRequiredBigDecimalValue(final String name)
  {
    return this.getRequiredTypedValue(name, BigDecimal.class);
  }

  public NaturalNumber getRequiredNaturalNumberValue(final IStringConstant name)
  {
    return getRequiredNaturalNumberValue(name.asString());
  }

  public NaturalNumber getRequiredNaturalNumberValue(final String name)
  {
    return getRequiredTypedValue(name, NaturalNumber.class);
  }

  public TimeValue getRequiredTimeValue(IStringConstant name, TimeUnit timeUnit)
  {
    return getRequiredTimeValue(name.asString(), timeUnit);
  }

  public TimeValue getRequiredTimeValue(String name, TimeUnit timeUnit)
  {
    return asTimeValue(getRequiredLongValue(name), timeUnit);
  }

  public Object getRequiredRawValue(final IStringConstant name)
  {
    return getRequiredRawValue(name.asString());
  }

  public Object getRequiredRawValue(final String name)
  {
    return this.getRawValueOf(name, true);
  }

  public JsonObjectAccessor getOptionalJsonObjectAccessorValue(final IStringConstant name)
  {
    return getOptionalJsonObjectAccessorValue(name.asString());
  }

  public JsonObjectAccessor getOptionalJsonObjectAccessorValue(final String name)
  {
    JsonObject object;

    object = this.getOptionalJsonObjectValue(name);
    if (object == null)
    {
      return null;
    }
    return new JsonObjectAccessor(object);
  }

  public JsonObject getOptionalJsonObjectValue(final IStringConstant name)
  {
    return getOptionalJsonObjectValue(name.asString());
  }

  public JsonObject getOptionalJsonObjectValue(final String name)
  {
    return this.getOptionalTypedValue(name, JsonObject.class);
  }

  public JsonArray getOptionalJsonArrayValue(final IStringConstant name)
  {
    return getOptionalJsonArrayValue(name.asString());
  }

  public JsonArray getOptionalJsonArrayValue(final String name)
  {
    return this.getOptionalTypedValue(name, JsonArray.class);
  }

  public String getOptionalStringValue(final IStringConstant name)
  {
    return getOptionalStringValue(name.asString());
  }

  public String getOptionalStringValue(final String name)
  {
    return this.getOptionalTypedValue(name, String.class);
  }

  public String[] getOptionalStringArrayValue(final IStringConstant name)
  {
    return getOptionalStringArrayValue(name.asString());
  }

  public String[] getOptionalStringArrayValue(final String name)
  {
    return getStringArray(name, false);
  }

  public Boolean getOptionalBooleanValue(final IStringConstant name)
  {
    return getOptionalBooleanValue(name.asString());
  }

  public Boolean getOptionalBooleanValue(final String name)
  {
    return getOptionalTypedValue(name, Boolean.class);
  }

  public Integer getOptionalIntegerValue(final IStringConstant name)
  {
    return getOptionalIntegerValue(name.asString());
  }

  public Integer getOptionalIntegerValue(final String name)
  {
    return getOptionalTypedValue(name, Integer.class);
  }

  public Long getOptionalLongValue(final IStringConstant name)
  {
    return getOptionalLongValue(name.asString());
  }

  public Long getOptionalLongValue(final String name)
  {
    return getOptionalTypedValue(name, Long.class);
  }

  public BigDecimal getOptionalBigDecimalValue(final IStringConstant name)
  {
    return getOptionalBigDecimalValue(name.asString());
  }

  public BigDecimal getOptionalBigDecimalValue(final String name)
  {
    return getOptionalTypedValue(name, BigDecimal.class);
  }

  public NaturalNumber getOptionalNaturalNumberValue(final IStringConstant name)
  {
    return getOptionalNaturalNumberValue(name.asString());
  }

  public NaturalNumber getOptionalNaturalNumberValue(final String name)
  {
    return getOptionalTypedValue(name, NaturalNumber.class);
  }

  /**
   * Returns the value at the specified field name as a {@link TimeValue}
   * with the specified {@link TimeUnit}. 
   * 
   * @param name The name of the field (must not be null).
   * @param timeUnit The time unit of the value found in the object (must not be null).
   * @return Returns either the TimeUnit or null if field not present.
   */
  public TimeValue getOptionalTimeValue(IStringConstant name, TimeUnit timeUnit)
  {
    return getOptionalTimeValue(name.asString(), timeUnit);
  }

  /**
   * Returns the value at the specified field name as a {@link TimeValue}
   * with the specified {@link TimeUnit}. 
   * 
   * @param name The name of the field (must not be null).
   * @param timeUnit The time unit of the value found in the object (must not be null).
   * @return Returns either the TimeUnit or null if field not present.
   */
  public TimeValue getOptionalTimeValue(String name, TimeUnit timeUnit)
  {
    return asTimeValue(getOptionalLongValue(name), timeUnit);
  }

  public Object getOptionalRawValue(final IStringConstant name)
  {
    return getOptionalRawValue(name.asString());
  }

  public Object getOptionalRawValue(final String name)
  {
    return getRawValueOf(name, false);
  }

  public <T> T getOptionalTypedValue(final IStringConstant name, Class<T> type)
  {
    return getOptionalTypedValue(name.asString(), type);
  }

  public <T> T getOptionalTypedValue(final String name, Class<T> type)
  {
    return getTypedValueOf(name, type, false);
  }

  public <T> T getRequiredTypedValue(final IStringConstant name, Class<T> type)
  {
    return getRequiredTypedValue(name.asString(), type);
  }

  public <T> T getRequiredTypedValue(final String name, Class<T> type)
  {
    return getTypedValueOf(name, type, true);
  }

  public <T> T getTypedValueOf(final IStringConstant name, final Class<T> type, final boolean isRequired)
  {
    return getTypedValueOf(name.asString(), type, isRequired);
  }

  public <T> T getTypedValueOf(final String name, final Class<T> type, final boolean isRequired)
  {
    T value;

    try
    {
      if (type == Long.class)
      {
        value = type.cast(getLongValueOf(name));
      }
      else if (type == NaturalNumber.class)
      {
        value = type.cast(getNaturalNumberValueOf(name));
      }
      else
      {
        value = getJsonObject().getValueOfType(name, type);
      }
    }
    catch (Exception ex)
    {
      throw new JsonAccessException(ex, "Type conversion problem for value of \"%s\". The value is not %s", name, type.getSimpleName());
    }
    if ((value == null) && isRequired)
    {
      signalRequiredFieldMissing(name);
    }
    return value;
  }

  public Object getRawValueOf(final IStringConstant name, final boolean isRequired)
  {
    return getRawValueOf(name.asString(), isRequired);
  }

  public Object getRawValueOf(final String name, final boolean isRequired)
  {
    Object value;

    value = getJsonObject().get(name);
    if ((value == null) && isRequired)
    {
      signalRequiredFieldMissing(name);
    }
    return value;
  }

  public void signalRequiredFieldMissing(String fieldName) throws JsonAccessException
  {
    throw new JsonAccessException("Required field missing: \"%s\"", fieldName);
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  protected String[] getStringArray(final String name, boolean isRequired)
  {
    JsonArray jsonArray;
    String[] strings;

    jsonArray = getTypedValueOf(name, JsonArray.class, isRequired);
    if (jsonArray == null)
    {
      return null;
    }
    strings = new String[jsonArray.size()];
    for (int i = 0; i < strings.length; i++)
    {
      strings[i] = jsonArray.getString(i);
    }
    return strings;
  }

  protected NaturalNumber getNaturalNumberValueOf(final String name)
  {
    Long longValue;

    longValue = getLongValueOf(name, "NaturalNumber");
    if (longValue == null)
    {
      return null;
    }
    return NaturalNumber.valueOf(longValue.longValue());
  }

  protected Long getLongValueOf(final String name)
  {
    return getLongValueOf(name, "Long");
  }

  protected Long getLongValueOf(final String name, String targetTypeName)
  {
    Object value;
    Integer intValue;

    value = getJsonObject().get(name);
    if (value == null)
    {
      return null;
    }
    if (value instanceof Long)
    {
      return (Long)value;
    }
    if (value instanceof Integer)
    {
      intValue = (Integer)value;
      return Long.valueOf(intValue.longValue());
    }
    throw new ClassCastException("Field named <" + name + "> is not of type " + targetTypeName + ". It is a " + value.getClass().getName());
  }

  protected TimeValue asTimeValue(Long value, TimeUnit timeUnit)
  {
    if (value == null)
    {
      return null;
    }
    return new TimeValue(value, timeUnit);
  }
}
