// ===========================================================================
// CONTENT  : INTERFACE JsonObject
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.2 - 08/03/2020
// HISTORY  :
//  25/08/2014  mdu  CREATED
//  14/07/2019  mdu   added -> asString()
//  08/03/2020  mdu   added -> method with IStringConstant for field name
//
// Copyright (c) 2014-2020, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text.json;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

import org.pfsw.bif.facet.IReadOnlyNamedValues;
import org.pfsw.bif.text.IStringConstant;

/**
 * The Java representation of a JSON object. A typed map that has string keys. 
 *
 * @author Manfred Duchrow
 * @version 1.2
 */
public class JsonObject extends LinkedHashMap<String, Object> implements JsonType, IReadOnlyNamedValues<Object>
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  private static final long serialVersionUID = 4396195499191118203L;

  private static final JsonUtil JU = JsonUtil.current();

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  public JsonObject()
  {
    super();
  }

  public JsonObject(int initialCapacity, float loadFactor)
  {
    super(initialCapacity, loadFactor);
  }

  public JsonObject(int initialCapacity)
  {
    super(initialCapacity);
  }

  public JsonObject(Map<? extends String, ? extends Object> map)
  {
    super(map);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Adds the given element if it is a valid JSON object type.
   * <p>
   * Valid types are:
   * <ul>
   * <li>JsonObject</li>
   * <li>JsonArray</li>
   * <li>String</li>
   * <li>Boolean</li>
   * <li>Integer</li>
   * <li>Long</li>
   * <li>BigDecimal</li>
   * <li>null</li>
   * </ul>
   * Objects of other types will cause an exception.
   * @return <tt>true</tt> if the object was added.
   * @throws IllegalArgumentException If the given object is not of a valid JSON type.  
   */
  @Override
  public Object put(String key, Object value)
  {
    if (JsonUtil.current().isValidJsonTypeInstance(value))
    {
      return super.put(key, value);
    }
    throw new IllegalArgumentException("Cannot add an object of invalid JSON type " + value.getClass().getName());
  }

  /**
   * Adds the given element if it is a valid JSON object type.
   * <p>
   * Valid types are:
   * <ul>
   * <li>JsonObject</li>
   * <li>JsonArray</li>
   * <li>String</li>
   * <li>Boolean</li>
   * <li>Integer</li>
   * <li>Long</li>
   * <li>BigDecimal</li>
   * <li>null</li>
   * </ul>
   * Objects of other types will cause an exception.
   * @return this JSON object.
   * @throws IllegalArgumentException If the given object is not of a valid JSON type.  
   */
  public JsonObject setField(IStringConstant name, Object value)
  {
    put(name.asString(), value);
    return this;
  }

  /**
   * Returns the value for the given name as JsonObject (if it is one). 
   * 
   * @param name The name of the object to look for (must not be null).
   * @return The found JsonObject or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no JsonObject.
   */
  public JsonObject getJsonObject(String name)
  {
    return getValueOfType(name, JsonObject.class);
  }

  /**
   * Returns the value for the given name as JsonObject (if it is one). 
   * 
   * @param name The name of the object to look for (must not be null).
   * @return The found JsonObject or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no JsonObject.
   */
  public JsonObject getJsonObject(IStringConstant name)
  {
    return getJsonObject(name.asString());
  }

  /**
   * Returns the value for the given name as JsonArray (if it is one). 
   * 
   * @param name The name of the array to look for (must not be null).
   * @return The found JsonArray or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no JsonArray.
   */
  public JsonArray getJsonArray(String name)
  {
    return getValueOfType(name, JsonArray.class);
  }

  /**
   * Returns the value for the given name as JsonArray (if it is one). 
   * 
   * @param name The name of the array to look for (must not be null).
   * @return The found JsonArray or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no JsonArray.
   */
  public JsonArray getJsonArray(IStringConstant name)
  {
    return getJsonArray(name.asString());
  }

  /**
   * Returns the value for the given name as String (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found String or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no String.
   */
  public String getString(String name)
  {
    return getValueOfType(name, String.class);
  }

  /**
   * Returns the value for the given name as String (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found String or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no String.
   */
  public String getString(IStringConstant name)
  {
    return getString(name.asString());
  }

  /**
   * Returns the value for the given name as Boolean (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found Boolean or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no Boolean.
   */
  public Boolean getBoolean(String name)
  {
    return getValueOfType(name, Boolean.class);
  }

  /**
   * Returns the value for the given name as Boolean (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found Boolean or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no Boolean.
   */
  public Boolean getBoolean(IStringConstant name)
  {
    return getBoolean(name.asString());
  }

  /**
   * Returns the value for the given name as Number (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found Number or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no Number.
   */
  public Number getNumber(String name)
  {
    return getTypedElement(name, Number.class);
  }

  /**
   * Returns the value for the given name as Number (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found Number or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no Number.
   */
  public Number getNumber(IStringConstant name)
  {
    return getNumber(name.asString());
  }

  /**
   * Returns the value for the given name as Integer (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found Integer or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no Integer.
   */
  public Integer getInteger(String name)
  {
    return getValueOfType(name, Integer.class);
  }

  /**
   * Returns the value for the given name as Integer (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found Integer or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no Integer.
   */
  public Integer getInteger(IStringConstant name)
  {
    return getInteger(name.asString());
  }

  /**
   * Returns the value for the given name as Long (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found Long or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no Long.
   */
  public Long getLong(String name)
  {
    return getValueOfType(name, Long.class);
  }

  /**
   * Returns the value for the given name as Long (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found Long or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no Long.
   */
  public Long getLong(IStringConstant name)
  {
    return getLong(name.asString());
  }

  /**
   * Returns the value for the given name as BigDecimal (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found BigDecimal or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no BigDecimal.
   */
  public BigDecimal getBigDecimal(String name)
  {
    return getValueOfType(name, BigDecimal.class);
  }

  /**
   * Returns the value for the given name as BigDecimal (if it is one). 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found BigDecimal or null if not existent.
   * @throws ClassCastException If the object associated to the given name is no BigDecimal.
   */
  public BigDecimal getBigDecimal(IStringConstant name)
  {
    return getBigDecimal(name.asString());
  }

  /**
   * Returns the value for the given name as type of the given class. 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found object or null if not existent.
   * @throws IllegalArgumentException If the specified type is no valid JSON type.
   * @throws ClassCastException If the object associated to the given name is not of the specified type.
   */
  public <T> T getValueOfType(String name, Class<T> type)
  {
    if (JsonUtil.current().isValidJsonType(type))
    {
      return getTypedElement(name, type);
    }
    throw new IllegalArgumentException("The specified type " + type.getName() + " of \"" + name + "\" is not supported!");
  }

  /**
   * Returns the value for the given name as type of the given class. 
   * 
   * @param name The name of the value to look for (must not be null).
   * @return The found object or null if not existent.
   * @throws IllegalArgumentException If the specified type is no valid JSON type.
   * @throws ClassCastException If the object associated to the given name is not of the specified type.
   */
  public <T> T getValueOfType(IStringConstant name, Class<T> type)
  {
    return getValueOfType(name.asString(), type);
  }

  /**
   * Returns true because this is a JSON object representation (in Java).
   */
  @Override
  public boolean isObject()
  {
    return true;
  }

  /**
   * Returns false because this is no JSON array representation (in Java).
   */
  @Override
  public boolean isArray()
  {
    return false;
  }

  @Override
  public Collection<String> getNames()
  {
    return keySet();
  }

  @Override
  public Object getValue(String name)
  {
    return get(name);
  }

  public Object getValue(IStringConstant name)
  {
    return getValue(name.asString());
  }

  /**
   * Returns the JSON string representation of this object.
   */
  @Override
  public String asString()
  {
    return toJSON();
  }

  @Override
  public void appendAsJSONString(Appendable output)
  {
    JU.appendJSONMap(output, this);
  }

  @Override
  public String toJSON()
  {
    StringBuffer buffer;

    buffer = new StringBuffer(500);
    appendAsJSONString(buffer);
    return buffer.toString();
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  @SuppressWarnings("unchecked")
  protected <T> T getTypedElement(String name, Class<T> type)
  {
    Object object;

    object = get(name);
    if ((object == null) || (type.isInstance(object)))
    {
      return (T)object;
    }
    throw new ClassCastException("Object named <" + name + "> is no " + type.getName() + ". It is a " + object.getClass().getName());
  }
}