// ===========================================================================
// CONTENT  : CLASS JsonArray
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.1 - 14/07/2019
// HISTORY  :
//  25/08/2014  mdu  CREATED
//  14/07/2019  mdu   added -> asString()
//
// Copyright (c) 2014-2019, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text.json;

import java.io.IOException;
// ===========================================================================
// IMPORTS
// ===========================================================================
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.pfsw.bif.exceptions.IORuntimeException;

/**
 * The Java internal representation of a JSON array.
 *
 * @author Manfred Duchrow
 * @version 1.1
 */
public class JsonArray extends ArrayList<Object> implements JsonType
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  private static final long serialVersionUID = -6724887777835446777L;
  
  private static final JsonUtil JU = JsonUtil.current();

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  public JsonArray(Object... objects)
  {
    super();
    for (Object object : objects)
    {
      this.add(object);
    }
  } 
  
  public JsonArray(Collection<? extends Object> collection)
  {
    super(collection);
  }

  public JsonArray(int initialCapacity)
  {
    super(initialCapacity);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns the value at the given index as JsonObject (if it is one). 
   * 
   * @param index The index of the object to look for.
   * @return The found JsonObject or null.
   * @throws ClassCastException If the object at the given index is no JsonObject.
   */
  public JsonObject getJsonObject(int index)
  {
    return this.getTypedElement(index, JsonObject.class);
  }

  /**
   * Returns the value at the given index as JsonArray (if it is one). 
   * 
   * @param index The index of the array to look for.
   * @return The found JsonArray or null.
   * @throws ClassCastException If the object at the given index is no JsonArray.
   */
  public JsonArray getJsonArray(int index)
  {
    return this.getTypedElement(index, JsonArray.class);
  }

  /**
   * Returns the value for the given index as String (if it is one). 
   * 
   * @param index The index of the value to look for.
   * @return The found String or null if not existent.
   * @throws ClassCastException If the object at the given index is no String.
   */
  public String getString(int index)
  {
    return this.getValueOfType(index, String.class);
  }
  
  /**
   * Returns the value for the given index as Boolean (if it is one). 
   * 
   * @param index The index of the value to look for.
   * @return The found Boolean or null if not existent.
   * @throws ClassCastException If the object at the given index is no Boolean.
   */
  public Boolean getBoolean(int index)
  {
    return this.getValueOfType(index, Boolean.class);
  }
  
  /**
   * Returns the value for the given index as Number (if it is one). 
   * 
   * @param index The index of the value to look for.
   * @return The found Number or null if not existent.
   * @throws ClassCastException If the object at the given index is no Number.
   */
  public Number getNumber(int index)
  {
    return this.getTypedElement(index, Number.class);
  }
  
  /**
   * Returns the value for the given index as Integer (if it is one). 
   * 
   * @param index The index of the value to look for.
   * @return The found Integer or null if not existent.
   * @throws ClassCastException If the object at the given index is no Integer.
   */
  public Integer getInteger(int index)
  {
    return this.getValueOfType(index, Integer.class);
  }
  
  /**
   * Returns the value for the given index as Long (if it is one). 
   * 
   * @param index The index of the value to look for.
   * @return The found Long or null if not existent.
   * @throws ClassCastException If the object at the given index is no Long.
   */
  public Long getLong(int index)
  {
    return this.getValueOfType(index, Long.class);
  }
  
  /**
   * Returns the value for the given index as BigDecimal (if it is one). 
   * 
   * @param index The index of the value to look for.
   * @return The found BigDecimal or null if not existent.
   * @throws ClassCastException If the object at the given index is no BigDecimal.
   */
  public BigDecimal getBigDecimal(int index)
  {
    return this.getValueOfType(index, BigDecimal.class);
  }
  
  /**
   * Returns the value for the given index as type of the given class. 
   * 
   * @param index The index of the value to look for.
   * @return The found object or null if not existent.
   * @throws IllegalArgumentException If the specified type is no valid JSON type.
   * @throws ClassCastException If the object at the given index is not of the specified type.
   */
  public <T> T getValueOfType(int index, Class<T> type)
  {
    if (JsonUtil.current().isValidJsonType(type))
    {      
      return this.getTypedElement(index, type);
    }
    throw new IllegalArgumentException("The specified type " + type.getName() + " at index " + index + " is not supported!");
  }
  
  /**
   * Returns false because this is no JSON object representation (in Java).
   */
  @Override
  public boolean isObject()
  {
    return false;
  }

  /**
   * Returns true because this is a JSON array representation (in Java).
   */
  @Override
  public boolean isArray()
  {
    return true;
  }

  /**
   * Adds the given element if it is a valid JSON object type.
   * <p>
   * Valid types are:
   * <ul>
   * <li>JsonObject</li>
   * <li>JsonArray</li>
   * <li>String</li>
   * <li>Boolean</li>
   * <li>Integer</li>
   * <li>Long</li>
   * <li>BigDecimal</li>
   * <li>null</li>
   * </ul>
   * Objects of other types will cause an exception.
   * @return <tt>true</tt> if the object was added.
   * @throws IllegalArgumentException If the given object is not of a valid JSON type.  
   */
  @Override
  public boolean add(Object object)
  {
    if (JsonUtil.current().isValidJsonTypeInstance(object))
    {
      super.add(object);
    }
    else
    {
      throw new IllegalArgumentException("Cannot add object of invalid JSON type " + object.getClass().getName());
    }
    return true;
  }

  /**
   * Adds all given elements if they are valid JSON object types, which are:
   * <ul>
   * <li>JsonObject</li>
   * <li>JsonArray</li>
   * <li>String</li>
   * <li>Boolean</li>
   * <li>Integer</li>
   * <li>Long</li>
   * <li>BigDecimal</li>
   * <li>null</li>
   * </ul>
   * Objects of other types will cause an exception.
   * @throws IllegalArgumentException If a given element is not of a valid JSON type.  
   */
  public void addElements(Object... elements)
  {
    for (Object object : elements)
    {
      this.add(object);
    }
  }

  /**
   * Returns a copy of this JsonArray as list of elements with the same type.
   * 
   * @param elementType The type all elements must have (must not be null).
   * @throws ClassCastException If any of the elements in this JsonArray is not of the specified type.
   */
  @SuppressWarnings("unchecked")
  public <T> List<T> asList(Class<T> elementType)
  {
    List<T> elements;

    elements = new ArrayList<T>();
    for (Object object : this)
    {
      if (elementType.isInstance(object))
      {
        elements.add((T)object);
      }
      else
      {
        throw new ClassCastException("At least one elemnt is not of type " + elementType.getName() + "; It is a " + (object == null ? "null" : object.getClass().getName()));
      }
    }
    return elements;
  }
  
  /**
   * Returns the JSON string representation of this object.
   */
  @Override
  public String asString()
  {
    return toJSON();
  }
  
  @Override
  public void appendAsJSONString(Appendable output)
  {
    boolean first = true;
    
    try
    {
      output.append(JSON_ARRAY_START);
      for (Object element : this)
      {
        if (first)
        {
          first = false;
        }
        else
        {
          output.append(JSON_ELEMENT_SEPARATOR);
        }
        JU.appendJSONObject(output, element);
      }
      output.append(JSON_ARRAY_END);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }
  
  @Override
  public String toJSON()
  {
    StringBuffer buffer;
    
    buffer = new StringBuffer(500);
    this.appendAsJSONString(buffer);
    return buffer.toString();
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  @SuppressWarnings("unchecked")
  protected <T> T getTypedElement(int index, Class<T> type)
  {
    Object object;

    object = this.get(index);
    if ((object == null) || (type.isInstance(object)))
    {
      return (T)object;
    }
    throw new ClassCastException("Object at <" + index + "> is no " + type.getName() + ". It is a " + object.getClass().getName());
  }

} // class JsonArray
