// ===========================================================================
// CONTENT  : CLASS Version
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.4 - 03/01/2020
// HISTORY  :
//  06/02/2004  mdu  CREATED
//	04/06/2006	mdu		changed	-->	to support letters in version numbers
//	01/06/2008	mdu		changed	-->	extended with more compare methods
//  09/08/2014  mdu   changed --> Versions that contain chars in one of first two elements are invalid
//  03/01/2020  mdu   changed --> implements IStringRepresentation
//
// Copyright (c) 2004-2020, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.pfsw.bif.text.IStringPair;
import org.pfsw.bif.text.IStringRepresentation;

/**
 * This class provides a representation of version identifiers of pattern
 * "x.y.z" in a way that they can be compared and sorted according to their
 * real meaning and not just by simple string comparison.
 * <p>
 * The last element (i.e. most right) may contain non-numeric characters and
 * will be compared as String. Such characters are limited to ascii
 * letters, digits and '-' and '_'.
 * <p>
 * Examples:
 * <br>3.1 &gt; 3.0
 * <br>3.1.1 &gt; 3.1
 * <br>2.4 &lt; 10.0
 * <br>0.11.2 &gt; 0.1.22
 * <br>1.4.7b &gt; 1.4.7a
 * <br>1.5.0_02 &lt; 1.5.0_17
 * <br>1.4.2_02 &lt; 1.4.10_01
 *
 * @author Manfred Duchrow
 * @version 1.4
 */
public class Version implements IStringRepresentation
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  private static final int LESS = -1;
  private static final int EQUAL = 0;
  private static final int GREATER = 1;

  private static final String JAVA_VERSION_PROPERTY = "java.version";
  private static final String JAVA_PATCH_SEPARATOR = "_";

  private static final int MUST_BE_NUMERIC = 2;

  static final Integer NOT_SET = new Integer(-1);

  /**
   * Contains the seperator between the version elements. (".")
   */
  public static final String SEPARATOR = ".";

  /**
   * A definition of characters that are allowed in a version element.
   */
  public static final String SPECIAL_CHARACTERS = "-_";

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private List<VersionElement> elements = null;

  // =========================================================================
  // CLASS METHODS
  // =========================================================================
  /**
   * Returns true if the given string represents a valid version.
   * 
   * @param str The string to be checked if it is a valid version
   */
  public static boolean isValidVersion(String str)
  {
    Version version;

    if (str == null)
    {
      return false;
    }
    version = new Version(str);
    return version.isValid();
  }

  /**
   * Returns the version of the current JVM.
   */
  public static Version getJavaVersion()
  {
    return new Version(System.getProperty(JAVA_VERSION_PROPERTY));
  }

  /**
   * Returns the version of the current JVM with out the patch level.
   * That is, for Java version "1.5.0_12" the version "1.5.0" will be returned.
   */
  public static Version getJavaBaseVersion()
  {
    String versionStr;

    versionStr = System.getProperty(JAVA_VERSION_PROPERTY);
    versionStr = StringUtil.current().cutTail(versionStr, JAVA_PATCH_SEPARATOR);
    return new Version(versionStr);
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance. 
   */
  protected Version()
  {
    super();
    setElements(new ArrayList<VersionElement>());
  }

  /**
   * Initialize the new instance with a version string of type "x.y.z".
   * The elements in the string separated by dots is not limited! 
   */
  public Version(String versionString)
  {
    this();
    parse(versionString);
  }

  /**
   * Initialize the new instance with the values from another a version.
   */
  public Version(Version version)
  {
    this();

    Iterator<VersionElement> otherElements = version.getElements().iterator();
    while (otherElements.hasNext())
    {
      VersionElement vElement = otherElements.next();
      getElements().add(vElement.copy());
    }
  }

  /**
   * Initialize the new instance with a major, minor, micro version
   * and a qualifier.
   * For any negative parameter value zero will be used instead! 
   */
  public Version(int major, int minor, int micro, String qualifier)
  {
    this(major, minor, micro);
    if (qualifier.startsWith(JAVA_PATCH_SEPARATOR))
    {
      getElements().add(new VersionElement(qualifier));
    }
    else
    {
      getLastElement().setStringPart(qualifier);
    }
  }

  /**
   * Initialize the new instance with a major, minor and micro version.
   * For any negative parameter value zero will be used instead! 
   */
  public Version(int major, int minor, int micro)
  {
    this(major, minor);
    addElement(micro);
  }

  /**
   * Initialize the new instance with a major and minor version.
   * For any negative parameter value zero will be used instead! 
   */
  public Version(int major, int minor)
  {
    this(major);
    addElement(minor);
  }

  /**
   * Initialize the new instance with a major version.
   * For a negative parameter value zero will be used instead! 
   */
  public Version(int major)
  {
    this();
    addElement(major);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns if this version is greater than the specified version.
   */
  public boolean isGreaterThan(Version version)
  {
    return compareTo(version) == GREATER;
  }

  /**
   * Returns if this version is less than the given version.
   */
  public boolean isLessThan(Version version)
  {
    return compareTo(version) == LESS;
  }

  /**
   * Returns if this version is greater than the given version.
   */
  public boolean isGreaterThan(String version)
  {
    return isGreaterThan(new Version(version));
  }

  /**
   * Returns if this version is less than the specified version.
   */
  public boolean isLessThan(String version)
  {
    return isLessThan(new Version(version));
  }

  /**
   * Returns if this version is greater or equal compared to the given version.
   */
  public boolean isGreaterOrEqual(Version version)
  {
    int result;

    result = compareTo(version);
    return (result == GREATER) || (result == EQUAL);
  }

  /**
   * Returns if this version is less or equal compared to the given version.
   */
  public boolean isLessOrEqual(Version version)
  {
    int result;

    result = compareTo(version);
    return (result == LESS) || (result == EQUAL);
  }

  /**
   * Returns if this version is greater or equal compared to the given version.
   */
  public boolean isGreaterOrEqual(String strVersion)
  {
    return isGreaterOrEqual(new Version(strVersion));
  }

  /**
   * Returns if this version is less or equal compared to the given version.
   */
  public boolean isLessOrEqual(String strVersion)
  {
    return isLessOrEqual(new Version(strVersion));
  }

  /**
   * Returns true if this version is equal to the version
   * represented by the given string.
   * If the strVersion is no valid version false will be returned. 
   */
  public boolean isEqualTo(String strVersion)
  {
    if (!isValidVersion(strVersion))
    {
      return false;
    }
    Version otherVersion;

    otherVersion = new Version(strVersion);
    return equals(otherVersion);
  }

  /**
   * Returns true if this version is equal to the given object.
   * The object must be of type <b>Version</b> or <b>String</b>.
   */
  @Override
  public boolean equals(Object obj)
  {
    if (obj instanceof Version)
    {
      return compareTo(obj) == EQUAL;
    }

    if (obj instanceof String)
    {
      return compareTo(new Version((String)obj)) == EQUAL;
    }

    return false;
  }

  /**
   * Returns a hash code
   */
  @Override
  public int hashCode()
  {
    int hash = 0;

    for (int i = 0; i < getElements().size(); i++)
    {
      hash = hash ^ getElements().get(i).hashCode();
    }
    return hash;
  }
  
  /**
   * Returns the version as string.
   * Use this method to get the version string rather than {@link #toString()}.
   * The {@link #toString()} might be changed in a future release to return
   * a debug string of this object.
   */
  @Override
  public String asString()
  {
    StringBuffer buffer;
    VersionElement element;
    
    buffer = new StringBuffer(20);
    
    for (int i = 0; i < getElements().size(); i++)
    {
      element = getElements().get(i);
      if (needsSeparator(i, element))
      {
        buffer.append(SEPARATOR);
      }
      buffer.append(element);
    }
    
    return buffer.toString();
  }

  /**
   * Returns the string for this object.
   * Currently this is the same as {@link #asString()}, but might change 
   * in the future. So use {@link #asString()} instead!
   */
  @Override
  public String toString()
  {
    StringBuffer buffer;
    VersionElement element;

    buffer = new StringBuffer(20);

    for (int i = 0; i < getElements().size(); i++)
    {
      element = getElements().get(i);
      if (needsSeparator(i, element))
      {
        buffer.append(SEPARATOR);
      }
      buffer.append(element);
    }

    return buffer.toString();
  }

  /**
   * Returns a new version object with the same value as this one.
   */
  public Version copy()
  {
    return new Version(this);
  }

  /**
   * Returns an array where all sub-elements are contained.
   */
  public String[] getVersionElements()
  {
    String[] subElements;
    VersionElement elem;

    subElements = new String[getElements().size()];
    for (int i = 0; i < getElements().size(); i++)
    {
      elem = getElements().get(i);
      subElements[i] = elem.toString();
    }
    return subElements;
  }

  /**
   * Compares this object with the specified object for order.  Returns a
   * negative integer, zero, or a positive integer as this object is less
   * than, equal to, or greater than the specified object.<p>
   *
   * @throws IllegalArgumentException if the specified object is not a Version
   */
  public int compareTo(Object obj)
  {
    Version otherVersion;
    List<VersionElement> otherElemets;
    int compResult;
    VersionElement element;
    VersionElement otherElement;
    int i;

    if (obj instanceof Version)
    {
      otherVersion = (Version)obj;
      otherElemets = otherVersion.getElements();
      for (i = 0; i < otherElemets.size(); i++)
      {
        if (i >= getElements().size())
          return LESS;

        element = getElements().get(i);
        otherElement = otherElemets.get(i);
        compResult = element.compare(otherElement);

        if (compResult == 0)
        {
          // Still equal - go to next element to compare
        }
        else
        {
          return (compResult < 0) ? LESS : GREATER;
        }
      }
      return (i == getElements().size()) ? EQUAL : GREATER;
    }
    throw new IllegalArgumentException("The object to compare is not a Version");
  }

  /**
   * Returns true if this version contains only positive numeric sub parts.
   */
  public boolean isNumeric()
  {
    Iterator<VersionElement> iter;
    VersionElement element;

    if (getElements().isEmpty())
    {
      return false;
    }
    iter = getElements().iterator();
    while (iter.hasNext())
    {
      element = iter.next();
      if (!element.isNumeric())
      {
        return false;
      }
    }
    return true;
  }

  /**
   * Returns true if this version contains only valid sub parts.
   */
  public boolean isValid()
  {
    Iterator<VersionElement> iter;
    VersionElement element;
    VersionElement lastElement;

    if (getElements().isEmpty())
    {
      return false;
    }
    lastElement = getLastElement();
    iter = getElements().iterator();
    while (iter.hasNext())
    {
      element = iter.next();
      if (!element.isValid())
      {
        return false;
      }
      if ((element != lastElement) && element.hasStringPart())
      {
        return false;
      }
    }
    return true;
  }

  /**
   * Returns the major version (the most left part) or -1 if this element is not
   * available or is not numeric.
   */
  public int getMajorVersion()
  {
    return getIntValueOfElement(0);
  }

  /**
   * Returns the minor version (the second from left part) or -1 if this element 
   * is not available or is not numeric.
   */
  public int getMinorVersion()
  {
    return getIntValueOfElement(1);
  }

  /**
   * Returns the micro version (the third from left part) or -1 if this element 
   * is not available or is not numeric.
   */
  public int getMicroVersion()
  {
    return getIntValueOfElement(2);
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  protected void parse(String versionString)
  {
    String[] parts;
    VersionElement element;
    String fragment;

    parts = str().parts(versionString, SEPARATOR);
    if (str().notNullOrEmpty(parts))
    {
      for (int i = 0; i < parts.length; i++)
      {
        fragment = parts[i];
        if (parts[i].contains(JAVA_PATCH_SEPARATOR))
        {
          IStringPair pair = str().splitStringPair(parts[i], JAVA_PATCH_SEPARATOR);
          if (str().isZeroOrPositiveInteger(pair.getString1()))
          {
            element = new VersionElement(pair.getString1());
            getElements().add(element);
            fragment = JAVA_PATCH_SEPARATOR + pair.getString2();
          }
        }
        element = new VersionElement(fragment);
        getElements().add(element);
        if ((i < MUST_BE_NUMERIC) && element.isPureString())
        {
          element.makeInvalid();
        }
      }
    }
  }

  protected boolean needsSeparator(int elementIndex, VersionElement element)
  {
    if (elementIndex == 0)
    {
      return false;
    }
    if (element.isPureString() && element.strPart.startsWith(JAVA_PATCH_SEPARATOR))
    {
      return false;
    }
    return true;
  }

  protected int getIntValueOfElement(int index)
  {
    VersionElement element;

    element = getElement(index);
    if (element == null)
    {
      return NOT_SET.intValue();
    }
    if (!element.isNumeric())
    {
      return NOT_SET.intValue();
    }
    return element.getNumeric().intValue();
  }

  protected void addElement(int value)
  {
    int num;

    num = (value < 0) ? 0 : value;
    getElements().add(new VersionElement(num));
  }

  protected VersionElement getLastElement()
  {
    return getElements().get(getElements().size() - 1);
  }

  protected VersionElement getElement(int index)
  {
    if ((index >= 0) && (index < getElements().size()))
    {
      return getElements().get(index);
    }
    return null;
  }

  protected List<VersionElement> getElements()
  {
    return elements;
  }

  protected void setElements(List<VersionElement> newValue)
  {
    elements = newValue;
  }

  protected StringUtil str()
  {
    return StringUtil.current();
  }

  // =========================================================================
  // INNER CLASSES
  // =========================================================================
  private class VersionElement
  {
    private boolean isValid = true;
    private Integer intPart = Version.NOT_SET;
    String strPart = "";

    VersionElement(String versionString)
    {
      super();
      parse(versionString);
    }

    VersionElement(int intPart)
    {
      this.intPart = Integer.valueOf(intPart);
    }

    private VersionElement(int intPart, String strPart)
    {
      this.intPart = Integer.valueOf(intPart);
      this.strPart = strPart;
    }

    @Override
    public String toString()
    {
      if (intPart.equals(NOT_SET))
      {
        return strPart;
      }
      return intPart.toString() + strPart;
    }

    protected void parse(String versionPart)
    {
      StringBuffer numBuffer;
      StringBuffer strBuffer;
      boolean isNumber = true;
      int value;
      char ch;

      numBuffer = new StringBuffer(20);
      strBuffer = new StringBuffer(20);
      for (int i = 0; i < versionPart.length(); i++)
      {
        ch = versionPart.charAt(i);
        isNumber = isNumber && Character.isDigit(ch);
        if (isNumber)
        {
          numBuffer.append(ch);
        }
        else
        {
          strBuffer.append(ch);
        }
      }

      try
      {
        if (numBuffer.length() > 0)
        {
          value = Integer.parseInt(numBuffer.toString());
          intPart = Integer.valueOf(value);
        }
      }
      catch (@SuppressWarnings("unused") NumberFormatException e)
      {
        intPart = NOT_SET;
      }
      strPart = strBuffer.toString();
    }

    protected int compare(VersionElement otherElement)
    {
      int compResult;

      compResult = intPart.compareTo(otherElement.intPart);
      if (compResult == EQUAL)
      {
        compResult = strPart.compareTo(otherElement.strPart);
      }
      return compResult;
    }

    protected void makeInvalid()
    {
      //			intPart = NOT_SET;
      isValid = false;
    }

    protected Integer getNumeric()
    {
      return intPart;
    }

    protected void setStringPart(String qualifier)
    {
      strPart = qualifier;
    }

    @Override
    public boolean equals(Object object)
    {
      if (object instanceof VersionElement)
      {
        return compare((VersionElement)object) == EQUAL;
      }
      return false;
    }

    @Override
    public int hashCode()
    {
      if (isNumeric())
      {
        return intPart.intValue();
      }
      return strPart.hashCode();
    }

    protected VersionElement copy()
    {
      return new VersionElement(intPart, strPart);
    }

    protected boolean isValid()
    {
      if (isValid)
      {
        if (intPart.equals(NOT_SET) && str().isNullOrEmpty(strPart))
        {
          isValid = false;
        }
      }
      return isValid;
    }

    protected boolean hasStringPart()
    {
      return str().notNullOrEmpty(strPart);
    }

    protected boolean isNumeric()
    {
      return (!intPart.equals(NOT_SET)) && str().isNullOrEmpty(strPart);
    }

    protected boolean isNumericWithString()
    {
      return (!intPart.equals(NOT_SET)) && hasStringPart();
    }

    protected boolean isPureString()
    {
      return !(isNumeric() || isNumericWithString());
    }
  }
}
