// ===========================================================================
// CONTENT  : CLASS TextFileWriter
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.0 - 04/08/2019
// HISTORY  :
//  04/08/2019  mdu  CREATED
//
// Copyright (c) 2019, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;

import org.pfsw.bif.exceptions.IORuntimeException;
import org.pfsw.bif.text.IStringRepresentation;

/**
 * Convenience class for writing text files with throwing {@link IORuntimeException}
 * rather than {@link IOException} in its methods that extend {@link Writer} functionality.
 * <br>
 * Sometimes (often with lambda expressions) it is very annoying having to explicitly catch
 * checked exceptions. It often "pollutes" the source code so heavily that the actual program logic
 * is hard to be recognized. In such cases it is often much better to write the code straight 
 * forward and put one try-catch for {@link RuntimeException} around it on a much higher level.
 * <br>
 * All {@link Writer} methods still throw {@link IOException}.
 * <p>
 * In contrary to {@link FileWriter} the constructors of this class support defining 
 * character encoding.
 * Furthermore it allows specifying the new line character(s) to be used.
 *
 * <p><code>UncheckedTextFileWriter</code> is meant for writing streams of characters.
 * For writing streams of raw bytes, consider using a <code>FileOutputStream</code>.
 *
 * @see OutputStreamWriter
 * @see FileOutputStream
 * @see FileWriter
 * @see TextFileWriter
 *
 * @author Manfred Duchrow
 * @version 1.0
 */
public class UncheckedTextFileWriter extends TextFileWriter
{
  /**
   * Creates a new <tt>UncheckedTextFileWriter</tt>, given the <tt>File</tt>
   * to write to with the encoding specified by the given charset.
   *
   * @param file the <tt>File</tt> to write to (must not be null).
   * @param charset The encoding to be used (must not be null).
   * @exception  IORuntimeException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public static UncheckedTextFileWriter create(File file, Charset charset)
  {
    try
    {
      return new UncheckedTextFileWriter(file, charset);
    }
    catch (FileNotFoundException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Creates a new <tt>UncheckedTextFileWriter</tt>, given the <tt>File</tt>
   * to write to with the default character encoding (<strong>UTF-8</strong>).
   * <p>
   * If you want to have the platform encoding instead, use {@link FileWriter} instead.
   * <br>
   * If you want to have another encoding, use {@link #create(File, Charset)}.
   *
   * @param file the <tt>File</tt> to write to (must not be null).
   * @exception  IORuntimeException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public static UncheckedTextFileWriter create(File file)
  {
    try
    {
      return new UncheckedTextFileWriter(file);
    }
    catch (FileNotFoundException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Creates a new <tt>UncheckedTextFileWriter</tt> on the file with the given filename
   * to write to with the encoding specified by the given charset.
   *
   * @param filename The name of the file to write to (must not be null).
   * @param charset The encoding to be used (must not be null).
   * @exception  IORuntimeException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public static UncheckedTextFileWriter create(String filename, Charset charset)
  {
    try
    {
      return new UncheckedTextFileWriter(filename, charset);
    }
    catch (FileNotFoundException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Creates a new <tt>UncheckedTextFileWriter</tt> on the file with the given filename
   * to write to with the default character encoding (<strong>UTF-8</strong>).
   * <p>
   * If you want to have the platform encoding instead, use {@link FileWriter} instead.
   * <br>
   * If you want to have another encoding, use {@link #create(String, Charset)}.
   *
   * @param filename The name of the file to write to (must not be null).
   * @exception  IORuntimeException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public static UncheckedTextFileWriter create(String filename)
  {
    try
    {
      return new UncheckedTextFileWriter(filename);
    }
    catch (FileNotFoundException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Creates a new <tt>UncheckedTextFileWriter</tt>, given the <tt>File</tt>
   * to write to with the encoding specified by the given charset.
   *
   * @param file the <tt>File</tt> to write to (must not be null).
   * @param charset The encoding to be used (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public UncheckedTextFileWriter(File file, Charset charset) throws FileNotFoundException
  {
    super(file, charset);
  }

  /**
   * Creates a new <tt>UncheckedTextFileWriter</tt>, given the <tt>File</tt>
   * to write to with the default character encoding (<strong>UTF-8</strong>).
   * <p>
   * If you want to have the platform encoding instead, use {@link FileWriter} instead.
   * <br>
   * If you want to have another encoding, use {@link #UncheckedTextFileWriter(File, Charset)}.
   *
   * @param file the <tt>File</tt> to write to (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public UncheckedTextFileWriter(File file) throws FileNotFoundException
  {
    this(file, DEFAULT_CHARSET);
  }

  /**
   * Creates a new <tt>UncheckedTextFileWriter</tt> on the file with the given filename
   * to write to with the encoding specified by the given charset.
   *
   * @param filename The name of the file to write to (must not be null).
   * @param charset The encoding to be used (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public UncheckedTextFileWriter(String filename, Charset charset) throws FileNotFoundException
  {
    this(new File(filename), charset);
  }

  /**
   * Creates a new <tt>UncheckedTextFileWriter</tt> on the file with the given filename
   * to write to with the default character encoding (<strong>UTF-8</strong>).
   * <p>
   * If you want to have the platform encoding instead, use {@link FileWriter} instead.
   * <br>
   * If you want to have another encoding, use {@link #UncheckedTextFileWriter(String, Charset)}.
   *
   * @param filename The name of the file to write to (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public UncheckedTextFileWriter(String filename) throws FileNotFoundException
  {
    this(filename, DEFAULT_CHARSET);
  }

  /**
   * Allows writing with placeholder replacement as provided by {@link String#format(String, Object...)}.
   * Use this method instead if {@link #write(String)} if {@link IOException} checking should be avoided.
   * 
   * @param text The text message that might contain placeholders (must not be null).
   * @param args Optional arguments to replace placeholders in the given text.
   * @return The writer itself to allow fluent API usage.
   * @throws IORuntimeException In any case of writing problem. 
   */
  @Override
  public UncheckedTextFileWriter writef(String text, Object... args)
  {
    try
    {
      super.writef(text, args);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
    return this;
  }

  /**
   * Writes the given object's {@link IStringRepresentation#asString()}.
   *
   * @param object The object of which to write its string representation (must not be null).
   * @return The writer itself to allow fluent API usage.
   * @throws IORuntimeException In any case of writing problem. 
   */
  @Override
  public UncheckedTextFileWriter write(IStringRepresentation object)
  {
    try
    {
      super.write(object);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
    return this;
  }

  /**
   * Writes the text with placeholder replacement as provided by {@link String#format(String, Object...)}.
   * followed by newline character(s).
   * 
   * @param text The text message that might contain placeholders (must not be null).
   * @param args Optional arguments to replace placeholders in the given text.
   * @return The writer itself to allow fluent API usage.
   * @throws IORuntimeException In any case of writing problem. 
   */
  @Override
  public UncheckedTextFileWriter writeln(String text, Object... args)
  {
    try
    {
      super.writeln(text, args);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
    return this;
  }

  /**
   * Writes the given object's {@link IStringRepresentation#asString()} followed 
   * by newline character(s).
   *
   * @param object The object of which to write its string representation (must not be null).
   * @return The writer itself to allow fluent API usage.
   * @throws IORuntimeException In any case of writing problem. 
   */
  @Override
  public UncheckedTextFileWriter writeln(IStringRepresentation object)
  {
    try
    {
      super.writeln(object);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
    return this;
  }

  /**
   * Writes the newline character(s).
   * 
   * @return The writer itself to allow fluent API usage.
   * @throws IORuntimeException In any case of writing problem. 
   */
  @Override
  public UncheckedTextFileWriter newLine()
  {
    try
    {
      super.newLine();
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
    return this;
  }
  
  @Override
  public UncheckedTextFileWriter setNewline(NewLine newline)
  {
    super.setNewline(newline);
    return this;
  }
}
