// ===========================================================================
// CONTENT  : CLASS TimeValue
// AUTHOR   : M.Duchrow
// VERSION  : 1.2 - 13/03/2020
// HISTORY  :
//  13/09/2013  mdu  CREATED
//  14/09/2014  mdu   added   --> toString(), equals(), hashCode(), compareTo()
//  13/03/2020  mdu   added   --> static create() methods as alternative to constructors, support NaturalNumber
//
// Copyright (c) 2013-2020, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

/**
 * Represents a time value and provides parsing of strings and 
 * various different getters for various time units.
 *
 * @author M.Duchrow
 * @version 1.2
 */
public class TimeValue implements Comparable<TimeValue>
{
  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private long milliseconds;

  // =========================================================================
  // STATIC METHODS
  // =========================================================================
  /**
   * Creates a new instance with the given milliseconds value.
   */
  public static TimeValue create(long millisecondsValue)
  {
    return new TimeValue(millisecondsValue);
  }

  /**
   * Creates a new instance with the given value and time unit.
   */
  public static TimeValue create(long value, TimeUnit unit)
  {
    return new TimeValue(value, unit);
  }

  /**
   * Creates a new instance with the given value and time unit.
   */
  public static TimeValue create(NaturalNumber value, TimeUnit unit)
  {
    return new TimeValue(value, unit);
  }
  
  /**
   * Creates a new instance with a string that must contain digits,
   * optionally followed by the short name of a unit (@see {@link TimeUnit}.
   * <br>
   * The given string will be parsed, taking any optional unit string at the end into account.
   * Allowed unit strings are the short names defined in TimeUnit (i.e. "h", "m", "s", "ms").
   * 
   * @param strValue The string to parse (must not be null).
   * @param defaultUnit The unit to use if the string does not contain an explicit short name (must not be null).
   * @throws NumberFormatException If the string does not contain a valid long value.
   */
  public static TimeValue create(String strValue, TimeUnit defaultUnit)
  {
    return new TimeValue(strValue, defaultUnit);
  }

  /**
   * Creates a the new instance with a string that must contain digits,
   * optionally followed by the short name of a unit (@see {@link TimeUnit}.
   * <br>
   * The default unit is "ms" if not explicitly set in the string.
   * @throws NumberFormatException If the string does not contain a valid long value.
   */
  public static TimeValue create(String strValue)
  {
    return new TimeValue(strValue);
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with a milliseconds value.
   */
  public TimeValue(long millisecondsValue)
  {
    super();
    setMilliseconds(millisecondsValue);
  }

  /**
   * Initialize the new instance with a value and unit.
   */
  public TimeValue(long value, TimeUnit unit)
  {
    this(value * unit.getMsFactor());
  }

  /**
   * Initialize the new instance with a value and unit.
   */
  public TimeValue(NaturalNumber value, TimeUnit unit)
  {
    this(value.longValue(), unit);
  }
  
  /**
   * Initialize the new instance with a string that must contain digits,
   * optionally followed by the short name of a unit (@see {@link TimeUnit}.
   * <br>
   * The given string will be parsed, taking any optional unit string at the end into account.
   * Allowed unit strings are the short names defined in TimeUnit (i.e. "h", "m", "s", "ms").
   * 
   * @param strValue The string to parse (must not be null).
   * @param defaultUnit The unit to use if the string does not contain an explicit short name (must not be null).
   * @throws NumberFormatException If the string does not contain a valid long value.
   */
  public TimeValue(String strValue, TimeUnit defaultUnit)
  {
    this(0L);
    setMilliseconds(parseToMilliseconds(strValue, defaultUnit));
  }

  /**
   * Initialize the new instance with a string that must contain digits,
   * optionally followed by the short name of a unit (@see {@link TimeUnit}.
   * <br>
   * The default unit is "ms" if not explicitly set in the string.
   * @throws NumberFormatException If the string does not contain a valid long value.
   */
  public TimeValue(String strValue)
  {
    this(strValue, TimeUnit.MILLISECONDS);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  public long asMilliseconds()
  {
    return getMilliseconds();
  }

  public long asSeconds()
  {
    return convertTo(TimeUnit.SECONDS);
  }

  public long asMinutes()
  {
    return convertTo(TimeUnit.MINUTES);
  }

  /**
   * Returns the underlying time in hours.
   */
  public long asHours()
  {
    return convertTo(TimeUnit.HOURS);
  }

  /**
   * Returns the value converted to the specified unit.
   * @param unit The unit of the result value (must not be null).
   */
  public long convertTo(TimeUnit unit)
  {
    return getMilliseconds() / unit.getMsFactor();
  }

  @Override
  public String toString()
  {
    return "TimeValue(" + asMilliseconds() + "ms)";
  }

  @Override
  public int hashCode()
  {
    return Long.toString(asMilliseconds()).hashCode();
  }

  @Override
  public boolean equals(Object obj)
  {
    if (obj instanceof TimeValue)
    {
      TimeValue other = (TimeValue)obj;
      return asMilliseconds() == other.asMilliseconds();
    }
    return false;
  }

  @Override
  public int compareTo(TimeValue other)
  {
    return Long.valueOf(asMilliseconds()).compareTo(Long.valueOf(other.asMilliseconds()));
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  /**
   * Parse the given string, taking any optional unit string at the end into account.
   * Allowed unit strings are the short names defined in TimeUnit (i.e. "h", "m", "s", "ms").
   * 
   * @param strValue The string to parse
   * @param defaultUnit The unit to use if the string does not contain an explicit short name.
   * @return The parsed value as milliseconds
   * @throws NumberFormatException If the string does not contain a valid long value or an invalid unit.
   */
  protected long parseToMilliseconds(String strValue, TimeUnit defaultUnit)
  {
    long millis;
    TimeUnit unit = null;
    StringPair stringPair;

    stringPair = splitDigitsAndUnit(strValue.trim());
    for (TimeUnit timeUnit : TimeUnit.values())
    {
      if (timeUnit.getShortName().equals(stringPair.getString2()))
      {
        unit = timeUnit;
        break;
      }
    }
    if (unit == null) // Needing default unit?
    {
      if (StringUtil.current().notNullOrEmpty(stringPair.getString2()))
      {
        throw new NumberFormatException("Invalid time unit specified: " + strValue);
      }
      unit = defaultUnit;
    }
    millis = Long.parseLong(stringPair.getString1()) * unit.getMsFactor();
    return millis;
  }

  protected StringPair splitDigitsAndUnit(String str)
  {
    StringPair pair;
    int noneDigitIndex = 0;

    pair = new StringPair();
    for (char ch : str.toCharArray())
    {
      if (!Character.isDigit(ch))
      {
        break;
      }
      noneDigitIndex++;
    }
    if (noneDigitIndex >= str.length())
    {
      pair.setString1(str);
      pair.setString2("");
    }
    else
    {
      pair.setString1(str.substring(0, noneDigitIndex));
      pair.setString2(str.substring(noneDigitIndex).trim());
    }
    return pair;
  }

  protected long getMilliseconds()
  {
    return milliseconds;
  }

  protected void setMilliseconds(long milliseconds)
  {
    this.milliseconds = milliseconds;
  }
}
