// ===========================================================================
// CONTENT  : CLASS TextFileWriter
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.0 - 04/08/2019
// HISTORY  :
//  04/08/2019  mdu  CREATED
//
// Copyright (c) 2019, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;

import org.pfsw.bif.exceptions.IORuntimeException;
import org.pfsw.bif.text.ICharsets;
import org.pfsw.bif.text.IStringRepresentation;

/**
 * Convenience class for writing text files.  
 * In contrary to {@link FileWriter} the constructors of this class support defining 
 * character encoding.
 * Furthermore it allows specifying the new line character(s) to be used.
 *
 * <p><code>TextFileWriter</code> is meant for writing streams of characters.
 * For writing streams of raw bytes, consider using a <code>FileOutputStream</code>.
 *
 * @see OutputStreamWriter
 * @see FileOutputStream
 * @see FileWriter
 *
 * @author Manfred Duchrow
 * @version 1.0
 */
public class TextFileWriter extends OutputStreamWriter
{
  /**
   * The default character encoding used by this class. 
   */
  public static final Charset DEFAULT_CHARSET = ICharsets.UTF_8;
  /**
   * The default newline definition used by this class. 
   */
  public static final NewLine DEFAULT_NEWLINE = NewLine.CURRENT_OS;
  
  private NewLine newline = DEFAULT_NEWLINE;

  /**
   * Creates a new <tt>TextFileWriter</tt>, given the <tt>File</tt>
   * to write to with the encoding specified by the given charset.
   *
   * @param file the <tt>File</tt> to write to (must not be null).
   * @param charset The encoding to be used (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public TextFileWriter(File file, Charset charset) throws FileNotFoundException
  {
    super(new FileOutputStream(file), charset);
  }

  /**
   * Creates a new <tt>TextFileWriter</tt>, given the <tt>File</tt>
   * to write to with the default character encoding (<strong>UTF-8</strong>).
   * <p>
   * If you want to have the platform encoding instead, use {@link FileWriter} instead.
   * <br>
   * If you want to have another encoding, use {@link #TextFileWriter(File, Charset)}.
   *
   * @param file the <tt>File</tt> to write to (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public TextFileWriter(File file) throws FileNotFoundException
  {
    this(file, DEFAULT_CHARSET);
  }

  /**
   * Creates a new <tt>TextFileWriter</tt> on the file with the given filename
   * to write to with the encoding specified by the given charset.
   *
   * @param filename The name of the file to write to (must not be null).
   * @param charset The encoding to be used (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public TextFileWriter(String filename, Charset charset) throws FileNotFoundException
  {
    this(new File(filename), charset);
  }
  
  /**
   * Creates a new <tt>TextFileWriter</tt> on the file with the given filename
   * to write to with the default character encoding (<strong>UTF-8</strong>).
   * <p>
   * If you want to have the platform encoding instead, use {@link FileWriter} instead.
   * <br>
   * If you want to have another encoding, use {@link #TextFileWriter(String, Charset)}.
   *
   * @param filename The name of the file to write to (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   writing.
   */
  public TextFileWriter(String filename) throws FileNotFoundException
  {
    this(filename, DEFAULT_CHARSET);
  }

  /**
   * Allows writing with placeholder replacement as provided by {@link String#format(String, Object...)}.
   * 
   * @param text The text message that might contain placeholders (must not be null).
   * @param args Optional arguments to replace placeholders in the given text.
   * @return The writer itself to allow fluent API usage.
   * @throws IOException In any case of writing problem. 
   */
  public TextFileWriter writef(String text, Object... args) throws IOException
  {
    write(String.format(text, args));
    return this;
  }

  /**
   * Writes the given object's {@link IStringRepresentation#asString()}.
   *
   * @param object The object of which to write its string representation (must not be null).
   * @return The writer itself to allow fluent API usage.
   * @throws IOException In any case of writing problem. 
   */
  public TextFileWriter write(IStringRepresentation object) throws IOException
  {
    write(object.asString());
    return this;
  }

  /**
   * Writes the text with placeholder replacement as provided by {@link String#format(String, Object...)}.
   * followed by newline character(s).
   * 
   * @param text The text message that might contain placeholders (must not be null).
   * @param args Optional arguments to replace placeholders in the given text.
   * @return The writer itself to allow fluent API usage.
   * @throws IOException In any case of writing problem. 
   */
  public TextFileWriter writeln(String text, Object... args) throws IOException
  {
    writef(text, args);
    return newLine();
  }
  
  /**
   * Writes the given object's {@link IStringRepresentation#asString()} followed 
   * by newline character(s).
   *
   * @param object The object of which to write its string representation (must not be null).
   * @return The writer itself to allow fluent API usage.
   * @throws IOException In any case of writing problem. 
   */
  public TextFileWriter writeln(IStringRepresentation object) throws IOException
  {
    write(object);
    return newLine();
  }
  
  /**
   * Writes the newline character(s).
   * 
   * @return The writer itself to allow fluent API usage.
   * @throws IOException In any case of writing problem. 
   */
  public TextFileWriter newLine() throws IOException
  {
    write(getNewline().asString());
    return this;
  }

  /**
   * Returns the new line definition used by this writer.
   * Use {@link NewLine#asString()} to get the actual new line character(s).
   */
  public NewLine getNewline()
  {
    return this.newline;
  }
  
  /**
   * Sets the new line definition to be used by this writer.
   * @return The writer itself to allow fluent API usage.
   */
  public TextFileWriter setNewline(NewLine newline)
  {
    this.newline = (newline == null) ? DEFAULT_NEWLINE : newline;
    return this;
  }

  /**
   * Closes this writer and swallows any {@link IOException} that might occur.
   */
  public void closeQuietly() 
  {
    try
    {
      close();
    }
    catch (@SuppressWarnings("unused") IOException e)
    {
      // ignore exception
    }
  }
  
  /**
   * Closes this writer and throws an {@link IORuntimeException} wrapping 
   * any {@link IOException} that might occur.
   * 
   * @throws IORuntimeException A wrapped {@link IOException}.
   */
  public void closeUnchecked() 
  {
    try
    {
      close();
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }
}
