// ===========================================================================
// CONTENT  : CLASS TextFileReader
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.0 - 27/07/2019
// HISTORY  :
//  27/07/2019  mdu  CREATED
//
// Copyright (c) 2019, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.Charset;

import org.pfsw.bif.exceptions.IORuntimeException;
import org.pfsw.bif.text.ICharsets;
import org.pfsw.bif.text.ILineProcessor;

/**
 * Convenience class for reading text files.  
 * In contrary to {@link FileReader} the constructors of this class support defining 
 * character encoding.
 *
 * <p><code>TextFileReader</code> is meant for reading streams of characters.
 * For reading streams of raw bytes, consider using a <code>FileInputStream</code>.
 *
 * @see InputStreamReader
 * @see FileInputStream
 * @see FileReader
 *
 * @author Manfred Duchrow
 * @version 1.0
 */
public class TextFileReader extends InputStreamReader
{
  /**
   * The default character encoding used by this class. 
   */
  public static final Charset DEFAULT_CHARSET = ICharsets.UTF_8;

  /**
   * Creates a new <tt>TextFileReader</tt>, given the <tt>File</tt>
   * to read from with the encoding specified by the given charset.
   *
   * @param file the <tt>File</tt> to read from (must not be null).
   * @param charset The encoding to be used (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   reading.
   */
  public TextFileReader(File file, Charset charset) throws FileNotFoundException
  {
    super(new FileInputStream(file), charset);
  }

  /**
   * Creates a new <tt>TextFileReader</tt>, given the <tt>File</tt>
   * to read from with the default character encoding (<strong>UTF-8</strong>)
   * of this class.
   * <p>
   * If you want to have the platform encoding instead, use {@link FileReader} instead.
   * <br>
   * If you want to have another encoding instead, use {@link #TextFileReader(File, Charset)}.
   *
   * @param file the <tt>File</tt> to read from (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   reading.
   */
  public TextFileReader(File file) throws FileNotFoundException
  {
    this(file, DEFAULT_CHARSET);
  }
  
  /**
   * Creates a new <tt>TextFileReader</tt> on the file with the given filename
   * to read from with the encoding specified by the given charset.
   *
   * @param filename The name of the file to read from (must not be null).
   * @param charset The encoding to be used (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   reading.
   */
  public TextFileReader(String filename, Charset charset) throws FileNotFoundException
  {
    this(new File(filename), charset);
  }
  
  /**
   * Creates a new <tt>TextFileReader</tt> on the file with the given filename
   * to read from with the default character encoding (<strong>UTF-8</strong>)
   * of this class.
   * <p>
   * If you want to have the platform encoding instead, use {@link FileReader} instead.
   * <br>
   * If you want to have another encoding instead, use {@link #TextFileReader(String, Charset)}.
   *
   * @param filename The name of the file to read from (must not be null).
   * @exception  FileNotFoundException  if the file does not exist,
   *                   is a directory rather than a regular file,
   *                   or for some other reason cannot be opened for
   *                   reading.
   */
  public TextFileReader(String filename) throws FileNotFoundException
  {
    this(filename, DEFAULT_CHARSET);
  }
  
  /**
   * Processes all lines of the underlying text file by reading 
   * each line and sending it to the given line processor.
   * If the line processor returns false reading further lines is stopped.
   * 
   * @param lineProcessor The processor to receive line by line (must not be null).
   * @return The number of lines read and processed.
   * @throws IORuntimeException In any case of IO problems.
   */
  public int processLines(ILineProcessor lineProcessor) 
  {
    BufferedReader bufferedReader;
    String line;
    boolean goOn = true;
    int counter = 0;
    
    bufferedReader = new BufferedReader(this);
    try
    {
      while (goOn && bufferedReader.ready())
      {
        line = bufferedReader.readLine();
        if (line == null)
        {
          break;
        }
        counter++;
        goOn = lineProcessor.processLine(line, counter);
      }
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
    return counter;
  }
  
  /**
   * Closes this reader and swallows any {@link IOException} that might occur.
   */
  public void closeQuietly() 
  {
    try
    {
      close();
    }
    catch (@SuppressWarnings("unused") IOException e)
    {
      // ignore exception
    }
  }
  
  /**
   * Closes this reader and throws an {@link IORuntimeException} wrapping 
   * any {@link IOException} that might occur.
   * 
   * @throws IORuntimeException A wrapped {@link IOException}.
   */
  public void closeUnchecked() 
  {
    try
    {
      close();
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }
}
