// ===========================================================================
// CONTENT  : CLASS StringPair
// AUTHOR   : M.Duchrow
// VERSION  : 1.2 - 13/07/2019
// HISTORY  :
//  24/03/2008  mdu  CREATED
//	21/01/2012	mdu		changed	-->	Added impl IJSONConvertible
//  13/07/2019  mdu   changed ->  appendAsJSONString(StringBuffer) t appendAsJSONString(Appendable output)
//
// Copyright (c) 2008-2019, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.io.IOException;

import org.pfsw.bif.exceptions.IORuntimeException;
// ===========================================================================
// IMPORTS
// ===========================================================================
import org.pfsw.bif.text.IJSONConvertible;
import org.pfsw.bif.text.IMutableStringPair;
import org.pfsw.bif.text.IStringPair;
import org.pfsw.bif.text.IStringRepresentation;
import org.pfsw.text.json.JsonUtil;

/**
 * A simple pair of strings.
 *
 * @author M.Duchrow
 * @version 1.2
 */
public class StringPair implements IMutableStringPair, IJSONConvertible, IStringRepresentation
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  private static final JsonUtil JU = JsonUtil.current();

  public static final StringPair[] EMPTY_ARRAY = new StringPair[0];

  /**
   * Defines the hash code of instances that contain two null strings.
   */
  public static final int NULL_HASHCODE = Void.TYPE.hashCode();

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private String string1 = null;
  private String string2 = null;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with default values.
   */
  public StringPair()
  {
    super();
  }

  /**
   * Initialize the new instance with one string.
   * That is, both strings get initialized with the given string.
   */
  public StringPair(String string)
  {
    super();
    this.setString1(string);
    this.setString2(string);
  }

  /**
   * Initialize the new instance with two strings.
   */
  public StringPair(String s1, String s2)
  {
    super();
    this.setString1(s1);
    this.setString2(s2);
  }

  /**
   * Initialize the new instance from the given string pair.
   */
  public StringPair(IStringPair pair)
  {
    super();
    this.setString1(pair.getString1());
    this.setString2(pair.getString2());
  }

  /**
   * Initialize the new instance with two strings in an array.
   * 
   * @throws IllegalArgumentException If the array does not contain exactly 2 elements. 
   */
  public StringPair(String[] strings)
  {
    super();
    if ((strings == null) || (strings.length != 2))
    {
      throw new IllegalArgumentException("The given strings array does not exactly consist of 2 string elements!");
    }
    this.setString1(strings[0]);
    this.setString2(strings[1]);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns true if the given object is an IStringPair and its both strings
   * are equal to the corresponding two strings of this object.
   * That applies also to null values in the string variables.
   */
  @Override
  public boolean equals(Object obj)
  {
    IStringPair pair;

    if (obj instanceof IStringPair)
    {
      pair = (IStringPair)obj;
      return this.isEqual(this.getString1(), pair.getString1()) && this.isEqual(this.getString2(), pair.getString2());
    }
    return false;
  }

  /**
   * Returns a hash built over the two contained strings.
   */
  @Override
  public int hashCode()
  {
    int hash = NULL_HASHCODE;

    hash += 1;
    if (this.getString1() != null)
    {
      hash = hash ^ this.getString1().hashCode();
    }
    hash += 2;
    if (this.getString2() != null)
    {
      hash = hash ^ this.getString2().hashCode();
    }
    return hash;
  }

  /**
   * Appends the internal state as JSON string representation to the given buffer.
   * 
   * @param output The buffer to which to append the JSON string (must not be null).
   */
  @Override
  public void appendAsJSONString(Appendable output)
  {
    try
    {
      output.append(JSON_OBJECT_START);
      JU.appendJSONPair(output, this.getString1(), this.getString2());
      output.append(JSON_OBJECT_END);
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
  }

  /**
   * Returns a JSON string representation of this object.
   * @return JSON object: {"string1":"string2"}
   */
  @Override
  public String toJSON()
  {
    return JU.convertToJSON(this);
  }

  /**
   * Returns both strings of this pair in a String array.
   */
  @Override
  public String[] asArray()
  {
    String[] strings;

    strings = new String[2];
    strings[0] = this.getString1();
    strings[1] = this.getString2();
    return strings;
  }

  /**
   * Returns the string representation of the pair with the default separator.
   */
  @Override
  public String asString()
  {
    return this.asString(this.getDefaultSeparator());
  }

  /**
   * Returns the two strings as one string separated by the given separator.
   * If the provided separator is null, no separator should be put between
   * the strings.
   * 
   * @param separator A separator to be placed between the two strings.
   */
  @Override
  public String asString(final String separator)
  {
    String sep = separator;
    StringBuffer buffer;
    int len = 0;
    String s1;
    String s2;

    s1 = this.getString1();
    if (s1 == null)
    {
      s1 = StringUtil.EMPTY_STRING;
    }
    else
    {
      len += s1.length();
    }
    s2 = this.getString2();
    if (s2 == null)
    {
      s2 = StringUtil.EMPTY_STRING;
    }
    else
    {
      len += s2.length();
    }
    if (sep == null)
    {
      sep = StringUtil.EMPTY_STRING;
    }
    else
    {
      len += sep.length();
    }
    buffer = new StringBuffer(len);
    buffer.append(s1);
    buffer.append(sep);
    buffer.append(s2);
    return buffer.toString();
  }

  /**
   * Returns the two strings with the default separator in between.
   */
  @Override
  public String toString()
  {
    return this.asString(this.getDefaultSeparator());
  }

  /**
   * Returns the default separator (here ":").
   */
  public String getDefaultSeparator()
  {
    return ":";
  }

  @Override
  public String getString1()
  {
    return this.string1;
  }

  @Override
  public void setString1(String newValue)
  {
    string1 = newValue;
  }

  @Override
  public String getString2()
  {
    return this.string2;
  }

  @Override
  public void setString2(String newValue)
  {
    string2 = newValue;
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  protected boolean isEqual(String s1, String s2)
  {
    if (s1 == null)
    {
      return (s2 == null);
    }
    if (s2 == null)
    {
      return false;
    }
    return s1.equals(s2);
  }
}
