// ===========================================================================
// CONTENT  : CLASS StringFilterCollection
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.1 - 20/03/2016
// HISTORY  :
//  30/11/2014  mdu  CREATED
//  20/03/2016  mdu   changed --> from StringFilter to IStringFilter0
//
// Copyright (c) 2014-2016, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.util.ArrayList;
import java.util.List;

import org.pfsw.bif.logic.BooleanOperation;
import org.pfsw.bif.text.IStringFilter;

/**
 * A collection of {@link IStringFilter} objects that can be matched against
 * strings according to a defined boolean operator (AND | OR | XOR).
 *
 * @author Manfred Duchrow
 * @version 1.1
 */
public class StringFilterCollection extends AStringFilter
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  public static final BooleanOperation DEFAULT_OPERATION = BooleanOperation.OR;

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private final List<IStringFilter> filterList = new ArrayList<IStringFilter>();
  private BooleanOperation operation = DEFAULT_OPERATION;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Creates a new empty instance with default boolean operation {@link BooleanOperation#OR}.
   */
  public StringFilterCollection()
  {
    super();
  }

  /**
   * Creates a new empty instance with specified boolean operation {@link BooleanOperation#OR}.
   */
  public StringFilterCollection(BooleanOperation operation)
  {
    super();
    setOperation(operation);
  }

  /**
   * Creates a new instance with default boolean operation {@link BooleanOperation#OR}
   * that will be filled with the given filters.
   */
  public StringFilterCollection(IStringFilter... filters)
  {
    this();
    addFilters(filters);
  }

  /**
   * Creates a new instance with the specified boolean operation
   * that will be filled with the given filters.
   */
  public StringFilterCollection(BooleanOperation operation, IStringFilter... filters)
  {
    this(filters);
    setOperation(operation);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  @Override
  public boolean matches(String aString)
  {
    switch (getOperation())
    {
      case OR :
        return matchesAnyFilter(aString);
      case AND :
        return matchesAllFilters(aString);
      case XOR :
        return matchesOneFilter(aString);
    }
    return false;
  }

  /**
   * Adds all given filters to this collection.
   */
  public void addFilters(IStringFilter... filters)
  {
    if (filters != null)
    {
      for (IStringFilter stringFilter : filters)
      {
        getFilterList().add(stringFilter);
      }
    }
  }

  public int size()
  {
    return getFilterList().size();
  }

  public boolean isEmpty()
  {
    return getFilterList().isEmpty();
  }

  public void clear()
  {
    getFilterList().clear();
  }

  public BooleanOperation getOperation()
  {
    return operation;
  }

  /**
   * Set the boolean operation the filter results must combined with.
   * 
   * @param operation The operation (must not be null).
   * @throws IllegalArgumentException If the given operation is null.
   */
  public void setOperation(BooleanOperation operation)
  {
    if (operation == null)
    {
      throw new IllegalArgumentException("operation must not be null!");
    }
    this.operation = operation;
  }

  /**
   * Returns a list with all currently held filters.
   */
  public List<IStringFilter> asList()
  {
    return new ArrayList<IStringFilter>(getFilterList());
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  protected boolean matchesAnyFilter(String aString)
  {
    for (IStringFilter filter : getFilterList())
    {
      if (filter.matches(aString))
      {
        return true;
      }
    }
    return false;
  }

  protected boolean matchesAllFilters(String aString)
  {
    for (IStringFilter filter : getFilterList())
    {
      if (!filter.matches(aString))
      {
        return false;
      }
    }
    return true;
  }

  protected boolean matchesOneFilter(String aString)
  {
    int counter = 0;

    for (IStringFilter filter : getFilterList())
    {
      if (filter.matches(aString))
      {
        counter++;
        if (counter > 1)
        {
          return false;
        }
      }
    }
    return (counter == 1);
  }

  protected List<IStringFilter> getFilterList()
  {
    return filterList;
  }
}
