// ===========================================================================
// CONTENT  : CLASS StringExaminer
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.1 - 01/06/2008
// HISTORY  :
//  29/09/2002  duma  CREATED
//	01/06/2008	mdu		added		-->	findPositionOf(), upToPosition()
//
// Copyright (c) 2002-2008, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

/**
 * As a subclass of StringScanner this class allows more advanced navigation 
 * over the underlying string.    <br>
 * That includes moving to positions of specific substrings etc.
 *
 * @author Manfred Duchrow
 * @version 1.1
 */
public class StringExaminer extends StringScanner
{
  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private boolean ignoreCase = false;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with the string to examine.   <br>
   * The string will be treated case-sensitive.
   * 
   * @param stringToExamine The string that should be examined
   */
  public StringExaminer(String stringToExamine)
  {
    this(stringToExamine, false);
  }

  /**
   * Initialize the new instance with the string to examine.
   * 
   * @param stringToExamine The string that should be examined
   * @param ignoreCase Specified whether or not treating the string case insensitive
   */
  public StringExaminer(String stringToExamine, boolean ignoreCase)
  {
    super(stringToExamine);
    ignoreCase(ignoreCase);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Increments the position pointer up to the last character that matched
   * the character sequence in the given matchString.
   * Returns true, if the matchString was found, otherwise false.
   * <p>
   * If the matchString was found, the next invocation of method nextChar()
   * returns the first character after that matchString.
   * 
   * @param matchString The string to look up
   */
  public boolean skipAfter(String matchString)
  {
    int index = 0;

    if ((matchString == null) || (matchString.length() == 0))
    {
      return false;
    }

    index = findPositionOf(matchString);
    if (index < 0)
    {
      return false;
    }
    setPosition(index + matchString.length());
    return true;
  }

  /**
   * Returns the position index of the given match string starting to look
   * for it from the current position.
   * If the given string cannot be found -1 will be returned.
   * This method does not change the position counter.
   */
  public int findPositionOf(String matchString)
  {
    char ch = '-';
    char matchChar = ' ';
    int startPos = -1;
    int foundPos = -1;
    int index = 0;
    int savedPosition;

    if (matchString == null)
    {
      return -1;
    }
    if (matchString.length() == 0)
    {
      return getPosition();
    }

    savedPosition = getPosition();
    try
    {
      ch = nextChar();
      while ((endNotReached(ch)) && (foundPos < 0))
      {
        matchChar = matchString.charAt(index);
        if (charsAreEqual(ch, matchChar))
        {
          if (startPos < 0)
          {
            startPos = getPosition() - 1;
          }
          index++;
          if (index >= matchString.length()) // whole matchString checked ?
          {
            foundPos = startPos;
          }
          else
          {
            ch = nextChar();
          }
        }
        else
        {
          startPos = -1;
          if (index == 0)
          {
            ch = nextChar();
          }
          else
          {
            index = 0;
          }
        }
      }
      return foundPos;
    }
    finally
    {
      setPosition(savedPosition);
    }
  }

  /**
   * Increments the position pointer up to the first character before
   * the character sequence in the given matchString.
   * Returns true, if the matchString was found, otherwise false.
   * <p>
   * If the matchString was found, the next invocation of method nextChar()
   * returns the first character of that matchString from the position where
   * it was found inside the examined string.
   * 
   * @param matchString The string to look up
   */
  public boolean skipBefore(String matchString)
  {
    boolean found;

    found = skipAfter(matchString);
    if (found)
      skip(0 - matchString.length());

    return found;
  }

  /**
   * Returns the a string containing all characters from the current position
   * up to the end of the examined string.   <br>
   * The character position of the examiner is not changed by this
   * method.
   */
  public String peekUpToEnd()
  {
    return upToEnd(true);
  }

  /**
   * Returns the a string containing all characters from the current position
   * up to the end of the examined string.   <br>
   * The character position is put to the end by this method.
   * That means the next invocation of nextChar() returns END_REACHED.
   */
  public String upToEnd()
  {
    return upToEnd(false);
  }

  /**
   * Returns the a string containing all characters from the current position
   * up to the specified position of the examined string.   <br>
   * The character position of the examiner after calling this method is pos.
   */
  public String peekUpToPosition(int pos)
  {
    StringBuffer strBuffer;

    strBuffer = new StringBuffer(length() - getPosition() + 1);
    appendUpToPosition(strBuffer, pos, true);
    return strBuffer.toString();
  }

  /**
   * Returns the a string containing all characters from the current position
   * up to the specified position of the examined string.   <br>
   * The character position of the examiner after calling this method is 
   * unchanged.
   */
  public String upToPosition(int pos)
  {
    StringBuffer strBuffer;

    strBuffer = new StringBuffer(length() - getPosition() + 1);
    appendUpToPosition(strBuffer, pos);
    return strBuffer.toString();
  }

  /**
   * Appends all characters from the current position
   * up to the specified position of the examined string to the given buffer.   
   * <br>
   * The character position of the examiner after calling this method is pos.
   */
  public void appendUpToPosition(final StringBuffer strBuffer, final int pos)
  {
    appendUpToPosition(strBuffer, pos, false);
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================

  protected boolean charsAreEqual(char char1, char char2)
  {
    return (ignoreCase()) ? (Character.toUpperCase(char1) == Character.toUpperCase(char2)) : (char1 == char2);
  }

  /**
   * Returns the a string containing all characters from the current position
   * up to the end of the examined string.   <br>
   * Depending on the peek flag the character position of the examiner 
   * is unchanged (true) after calling this method or points behind the strings
   * last character.
   */
  protected String upToEnd(boolean peek)
  {
    char result = '-';
    int lastPosition = 0;
    StringBuffer tempBuffer = new StringBuffer(100);

    lastPosition = getPosition();
    try
    {
      result = nextChar();
      while (endNotReached(result))
      {
        tempBuffer.append(result);
        result = nextChar();
      }
    }
    finally
    {
      if (peek)
      {
        setPosition(lastPosition);
      }
    }
    return tempBuffer.toString();
  }

  /**
   * Returns the a string containing all characters from the current position
   * up to the specified position of the examined string.   <br>
   * Depending on the peek flag the character position of the examiner 
   * is unchanged (true) after calling this method or points to pos.
   */
  protected void appendUpToPosition(final StringBuffer strBuffer, final int pos, final boolean peek)
  {
    char result;
    int lastPosition;

    if (pos <= getPosition())
    {
      return;
    }
    lastPosition = getPosition();
    try
    {
      do
      {
        result = nextChar();
        strBuffer.append(result);
      }
      while (getPosition() < pos);
    }
    finally
    {
      if (peek)
      {
        setPosition(lastPosition);
      }
    }
  }

  protected boolean ignoreCase()
  {
    return ignoreCase;
  }

  protected void ignoreCase(boolean newValue)
  {
    ignoreCase = newValue;
  }
}
