// ===========================================================================
// CONTENT  : CLASS StringCollectionFilter
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.2 - 301/11/2014
// HISTORY  :
//  28/05/2005  mdu  CREATED
//	24/02/2006	mdu		changed	-> to extend AStringFilter rather than implementing StringFilter
//  30/11/2014  mdu   changed -> support varargs
//
// Copyright (c) 2005-2014, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.util.Collection;

/**
 * This filter is based on a collection of strings. It matches all
 * strings that are in the underlying collection. 
 * By default it compares case-insensitive. Case sensitivity must be switched on
 * explicitly.
 *
 * @author Manfred Duchrow
 * @version 1.2
 */
public class StringCollectionFilter extends AStringFilter
{
  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private String[] internalStrings = null;
  private boolean unchanged = true;
  private boolean ignoreCase = true;

  // =========================================================================
  // CLASS METHODS
  // =========================================================================
  /**
   * A factory method that supports a variable number of strings.
   */
  public static StringCollectionFilter create(String... strings)
  {
    return new StringCollectionFilter(strings);
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with an array of strings.
   * 
   * @param strings The strings that define the set that matches this filter
   */
  public StringCollectionFilter(String[] strings)
  {
    super();
    setInternalStrings(strings);
  }

  /**
   * Initialize the new instance with a collection of strings.
   * 
   * @param strings A list that must only contain String objects
   */
  public StringCollectionFilter(Collection<String> strings)
  {
    super();
    setInternalStrings(str().asStrings(strings));
  }

  /**
   * Initialize the new instance with a List of strings provided as one
   * string where the values are separated by the specified separators.
   * 
   * @param strings The strings to add (separated values)
   * @param separators Each character is treated as a separator between two string values
   */
  public StringCollectionFilter(String strings, String separators)
  {
    super();
    setInternalStrings(str().parts(strings, separators));
  }

  /**
   * Initialize the new instance with a List of strings provided as one
   * string where the values are separated by comma (",").
   * 
   * @param strings The strings to add (comma separated values)
   */
  public StringCollectionFilter(String strings)
  {
    this(strings, ",");
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns true if the given string is in the internal collection of strings.
   * If the given string is null, false will be returned.
   * 
   * @param string The string to look up in the collection
   */
  @Override
  public boolean matches(String string)
  {
    boolean inArray;

    if ((string == null) || (getInternalStrings() == null))
    {
      return false;
    }
    inArray = str().contains(getInternalStrings(), string, getIgnoreCase());
    return unchanged ? inArray : !inArray;
  }

  /**
   * Inverts the (match) logic of this filter
   */
  public void negate()
  {
    unchanged = !unchanged;
  }

  /**
   * Returns true if this filter matches a string if it is found in the 
   * internal collection.
   * Returns false if this filter matches a string if it is NOT found in the 
   * internal collection.
   */
  public boolean matchesIfInCollection()
  {
    return unchanged;
  }

  /**
   * Add the given strings to the internal string collection.
   * If the argument is null it will be ignored.
   */
  public void add(String... strings)
  {
    if (strings != null)
    {
      for (String string : strings)
      {
        addSingleString(string);
      }
    }
  }

  /**
   * Returns true if the string comparison is done case-insensitive 
   */
  public boolean getIgnoreCase()
  {
    return ignoreCase;
  }

  /**
   * Sets whether or not the string comparison is done case-insensitive 
   */
  public void setIgnoreCase(boolean newValue)
  {
    ignoreCase = newValue;
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  /**
   * Add the given string to the internal string collection.
   * If the argument is null it will be ignored.
   */
  protected void addSingleString(String string)
  {
    String[] newArray;

    if (string != null)
    {
      if (getInternalStrings() == null)
      {
        newArray = new String[] { string };
      }
      else
      {
        newArray = str().append(getInternalStrings(), string);
      }
      setInternalStrings(newArray);
    }
  }

  protected String[] getInternalStrings()
  {
    return internalStrings;
  }

  protected void setInternalStrings(String[] newValue)
  {
    internalStrings = newValue;
  }

  protected StringUtil str()
  {
    return StringUtil.current();
  }
}
