// ===========================================================================
// CONTENT  : CLASS SqlMatchRuleVisitor
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.3 - 24/10/2003
// HISTORY  :
//  17/08/2001  duma  CREATED
//  12/11/2001  duma  changed ->  Redesign with inner class GroupInfo
//	27/12/2002	duma	changed	->	Support of new operators <,<=,>,>=
//	24/10/2003	duma	bugfix	->	Removed trailing space char in generated string
//							duma	added		->	Operator constants and IN() generation 
//
// Copyright (c) 2001-2003, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

/**
 * Walks over a MatchRule to create a SQL WHERE clause out of it.
 *
 * @author Manfred Duchrow
 * @version 1.3
 */
public class SqlMatchRuleVisitor implements MatchRuleVisitor
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  protected static final char MULTICHAR_WILDCARD = '%';

  protected static final String SQL_OPERATOR_NOT = "NOT ";
  protected static final String SQL_OPERATOR_AND = "AND ";
  protected static final String SQL_OPERATOR_OR = "OR ";
  protected static final String SQL_OPERATOR_IN = "IN ";
  protected static final String SQL_OPERATOR_LIKE = "LIKE ";
  protected static final String SQL_OPERATOR_EQUAL = "= ";
  protected static final String SQL_OPERATOR_GE = ">= ";
  protected static final String SQL_OPERATOR_LE = "<= ";
  protected static final String SQL_OPERATOR_GREATER = "> ";
  protected static final String SQL_OPERATOR_LESS = "< ";
  protected static final String OPEN_PARENTHESIS = "( ";
  protected static final String CLOSE_PARENTHESIS = ") ";
  protected static final String COMMA = ", ";
  protected static final String START_QUOTE = "'";
  protected static final String END_QUOTE = "' ";

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private StringBuffer buffer = null;
  private boolean groupStarted = true;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with default values.
   */
  public SqlMatchRuleVisitor()
  {
    super();
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * This method will be called right before the MatchRule walks
   * through its elements.
   */
  @Override
  public void walkThroughInit()
  {
    setBuffer(new StringBuffer(100));
    groupStarted(true);
  }

  /**
   * This method will be called when the MatchRule has finished to walk
   * through its elements.
   */
  @Override
  public void walkThroughFinished()
  {
    // NOP
  }

  /**
   * This method will be called for each start of a new group.
   *
   * @param andOperator If true it is an AND combination otherwise it is OR
   * @param notOperator Is only true for a NOT operation
   */
  @Override
  public void startGroup(boolean andOperator, boolean notOperator)
  {
    appendOperators(andOperator, notOperator);
    getBuffer().append(OPEN_PARENTHESIS);
    groupStarted(true);
  }

  /**
   * This method will be called for each group end occurrence.
   */
  @Override
  public void endGroup()
  {
    getBuffer().append(CLOSE_PARENTHESIS);
    groupStarted(false);
  }

  /**
   * This method will be called for each attribute.
   *
   * @param name The attribute's name
   * @param compareOperator The operator used to compare values
   * @param values All values the attrubute my match (implicit OR combination !)
   * @param andOperator If true it is an AND combination otherwise it is OR
   * @param notOperator Is only true for a NOT operation
   */
  @Override
  public void attribute(String name, MatchRuleCompareOperator compareOperator, String[] values, boolean andOperator, boolean notOperator)
  {
    appendOperators(andOperator, notOperator);

    if (canBeInOperator(compareOperator, values))
    {
      append_IN(name, values);
    }
    else
    {
      appendAttributeValues(name, compareOperator, values);
    }
    groupStarted(false);
  }

  /**
   * Converts the given match rule into a SQL conditional clause.
   *
   * @param matchRule The rule to be converted
   */
  public String asSqlClause(MatchRule matchRule)
  {
    matchRule.apply(this);
    return getBuffer().toString().trim();
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================

  protected void appendOperators(boolean andOperator, boolean notOperator)
  {
    if (!groupStarted())
    {
      getBuffer().append(andOperator ? SQL_OPERATOR_AND : SQL_OPERATOR_OR);
    }

    if (notOperator)
    {
      getBuffer().append(SQL_OPERATOR_NOT);
    }
  }

  protected void appendAttribute(final String name, final MatchRuleCompareOperator compareOperator, final String attrValue)
  {
    String value = attrValue;

    getBuffer().append(name);
    getBuffer().append(' ');
    switch (compareOperator)
    {
      case OPERATOR_EQUALS :
        if (value.indexOf(StringPattern.DEFAULT_MULTICHAR_WILDCARD) >= 0)
        {
          value = value.replace(StringPattern.DEFAULT_MULTICHAR_WILDCARD, MULTICHAR_WILDCARD);
          getBuffer().append(SQL_OPERATOR_LIKE);
        }
        else
        {
          getBuffer().append(SQL_OPERATOR_EQUAL);
        }
        break;
      case OPERATOR_GREATER :
        getBuffer().append(SQL_OPERATOR_GREATER);
        break;
      case OPERATOR_LESS :
        getBuffer().append(SQL_OPERATOR_LESS);
        break;
      case OPERATOR_GREATER_OR_EQUAL :
        getBuffer().append(SQL_OPERATOR_GE);
        break;
      case OPERATOR_LESS_OR_EQUAL :
        getBuffer().append(SQL_OPERATOR_LE);
        break;
    }
    appendValue(value);
  }

  protected void appendValue(String value)
  {
    getBuffer().append(START_QUOTE);
    getBuffer().append(value);
    getBuffer().append(END_QUOTE);
  }

  protected void appendAttributeValues(String name, MatchRuleCompareOperator compareOperator, String[] values)
  {
    boolean manyValues;

    manyValues = values.length > 1;

    if (manyValues)
    {
      getBuffer().append(OPEN_PARENTHESIS);
    }

    appendAttribute(name, compareOperator, values[0]);
    for (int i = 1; i < values.length; i++)
    {
      getBuffer().append(SQL_OPERATOR_OR);
      appendAttribute(name, compareOperator, values[i]);
    }
    if (manyValues)
    {
      getBuffer().append(CLOSE_PARENTHESIS);
    }
  }

  protected void append_IN(String attrName, String[] values)
  {
    getBuffer().append(attrName);
    getBuffer().append(' ');
    getBuffer().append(SQL_OPERATOR_IN);
    getBuffer().append(OPEN_PARENTHESIS);
    for (int i = 0; i < values.length; i++)
    {
      if (i > 0)
      {
        getBuffer().append(COMMA);
      }

      appendValue(values[i]);
    }
    getBuffer().append(CLOSE_PARENTHESIS);
  }

  protected boolean canBeInOperator(MatchRuleCompareOperator compareOperator, String[] values)
  {
    if (compareOperator != MatchRuleCompareOperator.OPERATOR_EQUALS)
    {
      return false;
    }

    if (values.length < 2)
    {
      return false;
    }

    for (int i = 0; i < values.length; i++)
    {
      if (values[i].indexOf(StringPattern.DEFAULT_MULTICHAR_WILDCARD) >= 0)
      {
        return false;
      }

      if (values[i].indexOf(StringPattern.DEFAULT_SINGLECHAR_WILDCARD) >= 0)
      {
        return false;
      }
    }

    return true;
  }

  protected StringBuffer getBuffer()
  {
    return this.buffer;
  }

  protected void setBuffer(StringBuffer newValue)
  {
    this.buffer = newValue;
  }

  protected boolean groupStarted()
  {
    return this.groupStarted;
  }

  protected void groupStarted(boolean newValue)
  {
    this.groupStarted = newValue;
  }

}