// ===========================================================================
// CONTENT  : CLASS RegexPattern
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.1 - 20/03/2016
// HISTORY  :
//  30/11/2014  mdu  CREATED
//  20/03/2016  mdu   changed --> from StringFilter to IStringFilter
//
// Copyright (c) 2014-2016, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

// ===========================================================================
// IMPORTS
// ===========================================================================
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.pfsw.bif.filter.IObjectFilter;
import org.pfsw.bif.text.IStringFilter;

/**
 * A regular expression pattern that can be used multiple times to be matched 
 * against strings. It conforms to interfaces {@link IStringFilter} and
 * {@link IObjectFilter}.
 * <p>
 * The supported regex syntax is the same as in <em>java.util.regex</em> package. 
 *
 * @author Manfred Duchrow
 * @version 1.1
 */
public class RegexPattern extends AStringFilter
{
  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private Pattern pattern;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Creates a new pattern based on the given regular expression.
   * 
   * @param regularExpression The regular expression (must not be null).
   * @throws IllegalArgumentException if the given expression is null.
   */
  public RegexPattern(String regularExpression)
  {
    super();
    if (regularExpression == null)
    {
      throw new IllegalArgumentException("The regularExpression argument must not be null!");
    }
    this.setPattern(Pattern.compile(regularExpression));
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  @Override
  public boolean matches(String aString)
  {
    Matcher matcher;

    if (aString == null)
    {
      return false;
    }
    matcher = this.getPattern().matcher(aString);
    return matcher.matches();
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  protected Pattern getPattern()
  {
    return this.pattern;
  }

  protected void setPattern(Pattern pattern)
  {
    this.pattern = pattern;
  }
}
