// ===========================================================================
// CONTENT  : CLASS RandomStringGenerator
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.0 - 11/10/2014
// HISTORY  :
//  11/10/2014  mdu  CREATED
//
// Copyright (c) 2014, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.security.SecureRandom;
import java.util.Random;

/**
 * A string generator that produces a new random string for each invocation.
 * It can be configured regarding the string length and the allowed characters
 * in the generated string.
 *
 * @author Manfred Duchrow
 * @version 1.0
 */
public class RandomStringGenerator implements StringGenerator
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  private static final char[] DEFAULT_ALLOWED_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789".toCharArray();

  /**
   * The default length (10) being used if not specified differently.
   */
  public static final int DEFAULT_LENGTH = 10;

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private char[] allowedCharsArray = DEFAULT_ALLOWED_CHARS;
  private int defaultLength = DEFAULT_LENGTH;
  private final Random random;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Creates a generator with all default values set (length=10, chars=[A-Z,a-z,0-9]).
   */
  public RandomStringGenerator()
  {
    this(new SecureRandom());
  }

  /**
   * Creates a generator with all default values set (length=10, chars=[A-Z,a-z,0-9]).
   */
  public RandomStringGenerator(int defaultLength)
  {
    this();
    setDefaultLength(defaultLength);
  }

  /**
   * Creates a generator with all default values set (length=10, chars=[A-Z,a-z,0-9])
   * and a custom randomizer.
   */
  public RandomStringGenerator(Random random)
  {
    super();
    this.random = random;
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  @Override
  public String generateString(int length)
  {
    StringBuilder buffer;

    buffer = new StringBuilder(length);
    for (int i = 0; i < length; i++)
    {
      buffer.append(nextChar());
    }
    return buffer.toString();
  }

  @Override
  public String generateString()
  {
    return generateString(getDefaultLength());
  }

  public int getDefaultLength()
  {
    return defaultLength;
  }

  /**
   * Sets the length to be used as default for generating new strings.
   * 
   * @param length The new length value which must be positive.
   * @throws IllegalArgumentException If the given length is not positive.
   */
  public void setDefaultLength(int length)
  {
    if (length <= 0)
    {
      throw new IllegalArgumentException("The default length cannot be set to a negative of zero value: " + length);
    }
    this.defaultLength = length;
  }

  /**
   * Returns the set of characters that are used as base for the string generation.
   */
  public String getAllowedCharacters()
  {
    return new String(getAllowedCharsArray());
  }

  /**
   * Sets the set of characters that are used as base for the string generation.
   * 
   * @param allowedChars A string containing all characters that are possible in
   * generated strings (must not be null).
   * @throws IllegalArgumentException If the given string is null.
   */
  public void setAllowedCharacters(String allowedChars)
  {
    if (allowedChars == null)
    {
      throw new IllegalArgumentException("The set of allowed characters must not be null.");
    }
    setAllowedCharsArray(allowedChars.toCharArray());
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  protected Object nextChar()
  {
    int index;

    index = nextIndex();
    return getAllowedCharsArray()[index];
  }

  protected int nextIndex()
  {
    return getRandom().nextInt(getAllowedCharsSize());
  }

  protected Random getRandom()
  {
    return this.random;
  }

  protected char[] getAllowedCharsArray()
  {
    return this.allowedCharsArray;
  }

  protected void setAllowedCharsArray(char[] charArray)
  {
    this.allowedCharsArray = charArray;
  }

  private int getAllowedCharsSize()
  {
    return getAllowedCharsArray().length;
  }
}
