// ===========================================================================
// CONTENT  : CLASS ObjectIdGenerator
// AUTHOR   : M.Duchrow
// VERSION  : 3.0 - 12/04/2020
// HISTORY  :
//  22/02/2008  mdu  CREATED
//  11/10/2014  mdu  added   -> implements StringGenerator
//  12/04/2020  mdu  changed -> Supports now no padding with length <= 0 
//
// Copyright (c) 2008-2020, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.util.concurrent.atomic.AtomicLong;

import org.pfsw.bif.identifier.IObjectIdGenerator;

/**
 * Generates numeric IDs that are left padded with zeros.
 *
 * @author M.Duchrow
 * @version 3.0
 */
public class ObjectIdGenerator implements IObjectIdGenerator, StringGenerator
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  protected final static char DEFAULT_PAD_CHAR = '0';
  protected final static long DEFAULT_START_ID = 1;
  protected final static int DEFAULT_LENGTH = 10;

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private int length = DEFAULT_LENGTH;
  private char padChar = DEFAULT_PAD_CHAR;
  private String prefix = null;
  private final AtomicLong nextIdHolder = new AtomicLong(DEFAULT_START_ID);

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with default values.
   * That is an ID length of 10 and a start ID of 1 with fill character '0'.
   */
  public ObjectIdGenerator()
  {
    super();
    setLength(getDefaultLength());
    setNextId(getDefaultStartId());
    setPadChar(getDefaultPadChar());
  }

  /**
   * Initialize the new instance with the length for the generated identifiers.
   * 
   * @param idLength The length to which Ids are filled up with leading zeros (must be > 0)
   */
  public ObjectIdGenerator(int idLength)
  {
    this();
    setLength(idLength);
  }

  /**
   * Initialize the new instance with the length for the generated identifiers
   * and the id to start with.
   * 
   * @param startId The first id to be generated
   * @param idLength The length to which Ids are filled up with leading zeros
   */
  public ObjectIdGenerator(long startId, int idLength)
  {
    this(idLength);
    setNextId(startId);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns a new unique identifier as string.
   */
  @Override
  public String newIdentifier()
  {
    return leftPad(nextIdentifier());
  }

  /**
   * Returns a new identifier which is different to the last one
   * and automatically increments the internal value for the next id.
   */
  public synchronized long nextIdentifier()
  {
    return getNextIdHolder().getAndIncrement();
  }

  /**
   * Returns whether or not padding for the identifiers is activated.
   */
  public boolean hasPadding() 
  {
    return getLength() > 0;
  }
  
  /**
   * Returns the full length of the generated identifiers. That includes the 
   * length of the prefix if one is defined.
   * 
   * @see #getPrefix()
   * @see ObjectIdGenerator#getLength()
   */
  public int getFullLength()
  {
    if (hasPadding())
    {
      return getPrefixLength() + getLength();      
    }
    return getPrefixLength() + Long.toString(getNextId()).length();      
  }

  /** 
   * Returns the length to which the IDs (the numeric part without prefix)) 
   * are filled up (left padded).
   */
  public int getLength()
  {
    return length;
  }

  /**
   * Sets the length to which the IDs (numeric part without prefix) must be 
   * filled-up with the defined padding character.
   * Since all IDs are based on data type long a value greater than 20 is
   * not really reasonable, however it is allowed here. 
   * 
   * @param len The new length. If len <= 0 no padding will be done!
   * @see #getPadChar()
   * @see #setPadChar(char)
   */
  public void setLength(int len)
  {
    length = len;
  }

  /**
   * Set the character that is used to fill up the IDs to the full length.
   * 
   * @param fillChar The fill character
   */
  public void setPadChar(char fillChar)
  {
    padChar = fillChar;
  }

  /**
   * Return the character that is used to fill up the IDs to the full length.
   */
  public char getPadChar()
  {
    return padChar;
  }

  /**
   * Returns the prefix that will be prepended to every generated ID.
   */
  public String getPrefix()
  {
    return prefix;
  }

  /**
   * Set the prefix that will be prepended to every generated ID.
   */
  public void setPrefix(String newValue)
  {
    prefix = newValue;
  }

  /**
   * Returns whether or not a prefix is defined.
   */
  public boolean hasPrefix()
  {
    return getPrefix() != null;
  }

  // Interface StringGenerator

  /**
   * Returns a new string. It is a new identifier as returned by {@link #newIdentifier()}.
   * 
   * @return A new string containing a left padded number with the currently defined length, plus the optional prefix.
   */
  @Override
  public String generateString()
  {
    return newIdentifier();
  }

  /**
   * Generates a new string (i.e. identifier) with the specified length. 
   * The contents of the string is a number left padded with the currently
   * defined padding character and prefixed with the defined prefix (if any is set).
   * <p>
   * The specified length will be used only for this single string generation.
   * That is, the defined length (see {@link #getLength()} and {@link #setLength(int)})
   * of this object will not be used or changed by this method.
   * 
   * @param len The length of the string to generate.
   * @return A new string with the given length.
   * @throws IllegalArgumentException if the given length is negative.
   */
  @Override
  public String generateString(int len)
  {
    int lengthWithoutPrefix;
    String string;

    if (len < 0)
    {
      throw new IllegalArgumentException("The length for a newly generated string must not be negative: " + len);
    }
    lengthWithoutPrefix = len - getPrefixLength();
    if (lengthWithoutPrefix <= 0)
    {
      throw new IllegalArgumentException("The underlying prefix is too long for the given length. No space left for any generated string part. Prefix: " + getPrefix());
    }
    string = leftPad(nextIdentifier(), lengthWithoutPrefix);
    return string;
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  protected int getPrefixLength()
  {
    if (hasPrefix())
    {
      return getPrefix().length();
    }
    return 0;
  }

  protected String leftPad(long id)
  {
    return leftPad(id, getLength());
  }

  protected String leftPad(long id, int stringLength)
  {
    StringBuffer buffer;
    
    buffer = new StringBuffer(100);
    if (hasPrefix())
    {
      buffer.append(getPrefix());
    }
    if (stringLength > 0)
    {
      str().leftPadCh(buffer, id, stringLength, getPadChar());      
    }
    else
    {
      buffer.append(id);
    }
    return buffer.toString();
  }
  
  /**
   * Returns the next identifier value without incrementing.
   */
  protected synchronized long getNextId()
  {
    return getNextIdHolder().get();
  }

  /**
   * Sets the next identifier to the given value.
   */
  protected synchronized void setNextId(long id)
  {
    getNextIdHolder().set(id);
  }

  protected long getDefaultStartId()
  {
    return DEFAULT_START_ID;
  }

  protected int getDefaultLength()
  {
    return DEFAULT_LENGTH;
  }

  protected char getDefaultPadChar()
  {
    return DEFAULT_PAD_CHAR;
  }
  
  protected AtomicLong getNextIdHolder()
  {
    return this.nextIdHolder;
  }

  protected StringUtil str()
  {
    return StringUtil.current();
  }
}
