// ===========================================================================
// CONTENT  : CLASS NaturalNumber
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.0 - 13/03/2020
// HISTORY  :
//  13/03/2020  mdu  CREATED
//
// Copyright (c) 2020, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import org.pfsw.bif.conversion.IIntegerRepresentation;
import org.pfsw.bif.conversion.ILongRepresentation;
import org.pfsw.bif.text.IStringRepresentation;

/**
 * A natural number is an abstraction of the two types {@link Integer} and {@link Long}.
 * It allows to treat such values in a more neutral way without being fixed on a specific 
 * java standard type.
 *
 * @author Manfred Duchrow
 * @version 1.0
 */
public class NaturalNumber extends Number implements Comparable<NaturalNumber>, IStringRepresentation, ILongRepresentation, IIntegerRepresentation
{
  private final Long numberValue;

  // =========================================================================
  // STATIC METHODS
  // =========================================================================
  public static NaturalNumber valueOf(long value)
  {
    return new NaturalNumber(value);
  }

  public static NaturalNumber valueOf(int value)
  {
    return new NaturalNumber(value);
  }

  /**
   * Parses the string argument as a signed decimal
   * <code>long</code>.  The characters in the string must all be
   * decimal digits, except that the first character may be an ASCII
   * minus sign <code>'-'</code> (<code>&#92;u002D'</code>) to
   * indicate a negative value. 
   * <p>
   * Note that neither the character <code>L</code>
   * (<code>'&#92;u004C'</code>) nor <code>l</code>
   * (<code>'&#92;u006C'</code>) is permitted to appear at the end
   * of the string as a type indicator, as would be permitted in
   * Java programming language source code.
   *
   * @param      value   a <code>String</code> containing the <code>long</code>
   *             representation to be parsed
   * @return     the <code>long</code> represented by the argument in 
   *       decimal.
   * @exception  NumberFormatException  if the string does not contain a
   *               parsable <code>long</code>.
   */
  public static NaturalNumber parse(String value)
  {
    return new NaturalNumber(value);
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  public NaturalNumber(long value)
  {
    super();
    this.numberValue = Long.valueOf(value);
  }

  public NaturalNumber(int value)
  {
    this((long)value);
  }

  /**
   * Parses the string argument as a signed decimal
   * <code>long</code>.  The characters in the string must all be
   * decimal digits, except that the first character may be an ASCII
   * minus sign <code>'-'</code> (<code>&#92;u002D'</code>) to
   * indicate a negative value. 
   * <p>
   * Note that neither the character <code>L</code>
   * (<code>'&#92;u004C'</code>) nor <code>l</code>
   * (<code>'&#92;u006C'</code>) is permitted to appear at the end
   * of the string as a type indicator, as would be permitted in
   * Java programming language source code.
   *
   * @param      value   a <code>String</code> containing the <code>long</code>
   *             representation to be parsed
   * @exception  NumberFormatException  if the string does not contain a
   *               parsable <code>long</code>.
   */
  public NaturalNumber(String value)
  {
    this(Long.parseLong(value));
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  @Override
  public double doubleValue()
  {
    return getNumberValue().doubleValue();
  }

  @Override
  public float floatValue()
  {
    return getNumberValue().floatValue();
  }

  @Override
  public int intValue()
  {
    return getNumberValue().intValue();
  }

  @Override
  public long longValue()
  {
    return getNumberValue().longValue();
  }

  @Override
  public String asString()
  {
    return getNumberValue().toString();
  }

  @Override
  public Long asLong()
  {
    return getNumberValue();
  }

  @Override
  public Integer asInteger()
  {
    return Integer.valueOf(intValue());
  }

  /**
   * Returns true if this number is greater than 0. 
   */
  public boolean isPositive()
  {
    return longValue() > 0L;
  }

  /**
   * Returns true if this number is less than 0. 
   */
  public boolean isNegative()
  {
    return longValue() < 0L;
  }

  @Override
  public int hashCode()
  {
    return getNumberValue().hashCode();
  }

  @Override
  public boolean equals(Object obj)
  {
    if ((obj != null) && (obj.getClass() == NaturalNumber.class))
    {
      NaturalNumber other = (NaturalNumber)obj;
      return getNumberValue().equals(other.getNumberValue());
    }
    return false;
  }

  /**
   * Compares two <code>NaturalNumber</code> objects numerically.
   *
   * @param   other   the <code>NaturalNumber</code> to be compared.
   * @return  the value <code>0</code> if this <code>NaturalNumber</code> is
   *    equal to the argument <code>NaturalNumber</code>; a value less than
   *    <code>0</code> if this <code>NaturalNumber</code> is numerically less
   *    than the argument <code>NaturalNumber</code>; and a value greater 
   *    than <code>0</code> if this <code>NaturalNumber</code> is numerically
   *    greater than the argument <code>NaturalNumber</code> (signed
   *    comparison).
   */
  @Override
  public int compareTo(NaturalNumber other)
  {
    return getNumberValue().compareTo(other.getNumberValue());
  }

  @Override
  public String toString()
  {
    return String.format("%s(%d)", getClass().getSimpleName(), getNumberValue());
  }

  protected Long getNumberValue()
  {
    return this.numberValue;
  }
}
