// ===========================================================================
// CONTENT  : CLASS MatchGroup
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.7 - 20/12/2004
// HISTORY  :
//  11/07/2001  duma  CREATED
//  09/10/2001  duma  changed -> Made class and constructor public
//  08/01/2002  duma  changed -> Made serializable
//	14/08/2002	duma	changed	-> New constructor with no arguments
//	23/08/2002	duma	re-designed	-> Moved parsing and printing to other classes
//	24/10/2002	duma	bugfix  -> Removed short circuit in doMatch()
//	24/10/2003	duma	added		-> multiCharWildcardMatchesEmptyString()
//	04/12/2003	duma	added		-> optimize(), optimizeAttribute()
//	20/12/2004	duma	added		-> applyDatatypes()
//
// Copyright (c) 2001-2004, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Represents a group of MatchAttributes and/or MatchGroups
 *
 * @author Manfred Duchrow
 * @version 1.7
 */
public class MatchGroup extends MatchElement
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  private static final long serialVersionUID = -4978135628713548031L;

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private List<MatchElement> elements = null;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with default values.
   */
  public MatchGroup()
  {
    super();
    setElements(newElementList());
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Add the specified element to the group.
   * The element will be added at the end of the element list.
   */
  public void addElement(MatchElement element)
  {
    getElements().add(element);
  }

  /**
   * Returns always true, because this is a group.
   */
  @Override
  public boolean isGroup()
  {
    return true;
  }

  /**
   * Returns the current number of elements in this group.
   */
  public int elementCount()
  {
    return getElements().size();
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================

  protected MatchElement getElement(int index)
  {
    return getElements().get(index);
  }

  @Override
  protected boolean doMatch(Map<String, ?> dictionary)
  {
    Iterator<MatchElement> iterator = null;
    MatchElement element = null;
    boolean matched = true;
    boolean isFirst = true;

    iterator = getElements().iterator();
    while (iterator.hasNext())
    {
      element = iterator.next();
      if (isFirst)
      {
        isFirst = false;
        matched = element.matches(dictionary);
      }
      else
      {
        if (element.getAnd())
        {
          matched = matched && element.matches(dictionary);
        }
        else
        {
          matched = matched || element.matches(dictionary);
        }
      }
    }
    return matched;
  }

  @Override
  protected void ignoreCase(boolean ignoreIt)
  {
    for (int i = 0; i < getElements().size(); i++)
    {
      getElement(i).ignoreCase(ignoreIt);
    }
  }

  /**
   * Defines whether or not the case of characters in attribute names
   * must be ignored.
   */
  @Override
  public void ignoreCaseInName(boolean ignoreIt)
  {
    for (int i = 0; i < getElements().size(); i++)
    {
      getElement(i).ignoreCaseInName(ignoreIt);
    }
  }

  @Override
  protected void multiCharWildcardMatchesEmptyString(boolean yesOrNo)
  {
    for (int i = 0; i < getElements().size(); i++)
    {
      getElement(i).multiCharWildcardMatchesEmptyString(yesOrNo);
    }
  }

  @Override
  protected void apply(MatchRuleVisitor visitor)
  {
    visitor.startGroup(getAnd(), getNot());

    for (int i = 0; i < getElements().size(); i++)
    {
      getElement(i).apply(visitor);
    }

    visitor.endGroup();
  }

  @Override
  protected void applyDatatypes(Map<String, Class<?>> datatypes) throws MatchRuleException
  {
    for (int i = 0; i < getElements().size(); i++)
    {
      getElement(i).applyDatatypes(datatypes);
    }
  }

  protected void optimize()
  {
    MatchGroup group;

    for (int i = 0; i < elementCount(); i++)
    {
      if (getElement(i).isAttribute())
      {
        optimizeAttribute(i);
      }
      else
      {
        group = (MatchGroup)getElement(i);
        group.optimize();
      }
    }
  }

  protected void optimizeAttribute(int index)
  {
    MatchAttribute attr;
    MatchAttribute otherAttr;
    Collection<StringPattern> patterns = null;
    boolean optimized = false;
    boolean done = false;
    int i;

    if (getElement(index).isAttribute())
    {
      attr = (MatchAttribute)getElement(index);
    }
    else
    {
      return;
    }

    i = index + 1;
    while (!done && (i < elementCount()))
    {
      done = true;
      if (getElement(i).isAttribute())
      {
        otherAttr = (MatchAttribute)getElement(i);
        if (!(otherAttr.getNot() || otherAttr.getAnd()))
        {
          if (attr.getAttributeName().equals(otherAttr.getAttributeName()))
          {
            done = false;
            if (!optimized)
            {
              optimized = true;
              patterns = new ArrayList<StringPattern>(attr.getPatterns().length + otherAttr.getPatterns().length);
              addAll(patterns, attr.getPatterns());
            }
            addAll(patterns, otherAttr.getPatterns());
          }
        }
      }
      i++;
    }
    if (optimized && (patterns != null))
    {
      attr.setPatterns(patterns.toArray(new StringPattern[patterns.size()]));
      removeElements(index + 1, i - 1);
    }
  }

  protected void addAll(Collection<StringPattern> coll, StringPattern[] patternArray)
  {
    for (int i = 0; i < patternArray.length; i++)
    {
      coll.add(patternArray[i]);
    }
  }

  protected void removeElements(int from, int to)
  {
    List<MatchElement> list;

    list = newElementList();
    for (int i = 0; i < elementCount(); i++)
    {
      if ((i < from) || (i > to))
      {
        list.add(getElement(i));
      }
    }
    setElements(list);
  }

  protected List<MatchElement> newElementList()
  {
    return new ArrayList<MatchElement>();
  }

  protected List<MatchElement> getElements()
  {
    return this.elements;
  }

  protected void setElements(List<MatchElement> newValue)
  {
    this.elements = newValue;
  }
}
