// ===========================================================================
// CONTENT  : CLASS MatchAttribute
// AUTHOR   : Manfred Duchrow
// VERSION  : 2.0 - 16/08/2007
// HISTORY  :
//  11/07/2001  duma  CREATED
//  09/10/2001  duma  changed -> Made class and constructor public and added a constructor
//  24/11/2001  duma  changed -> Supports now String[] and List objects as values of an attribute
//  08/01/2002  duma  changed -> Made serializable
//	14/08/2002	duma	changed	-> New constructor with no arguments
//	23/08/2002	duma	re-designed	-> Moved parsing and printing to other classes
//	26/12/2002	duma	added		-> Operators to support =,<,>,<= and >=
//	21/03/2003	duma	changed	-> Supports Integer values in value map
//	24/10/2003	duma	added		-> multiCharWildcardMatchesEmptyString()
//										changed	-> matchValue()
//	20/12/2004	duma	added		-> support for various datatypes (Float,Double,BigDecimal,Integer,Long,Date)
//	16/08/2007	mdu		changed	-> moved conversion to MatchRuleStringConverter
//
// Copyright (c) 2001-2007, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import static org.pfsw.text.MatchRuleCompareOperator.*;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.pfsw.bif.conversion.IIntegerRepresentation;
import org.pfsw.bif.text.IStringRepresentation;

/**
 * An instance of this class holds the name and the pattern values
 * for one attribute.
 * With the matches() method it can be checked against a Map of attributes.
 *
 * @author Manfred Duchrow
 * @version 2.0
 */
public class MatchAttribute extends MatchElement
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  private static final long serialVersionUID = -8277461956122062665L;

  private static MatchRuleTypeConverter TYPE_CONVERTER = new MatchRuleTypeConverter();

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private MatchRuleCompareOperator operator = OPERATOR_EQUALS;
  private String attributeName = null;
  private StringPattern[] patterns = null;
  private boolean ignoreCaseInName = false;
  private Object valueType = null; // null means String!
  private Float[] floatValues = null;
  private Double[] doubleValues = null;
  private BigDecimal[] bigDecimalValues = null;
  private Integer[] integerValues = null;
  private Long[] longValues = null;
  private Date[] dateValues = null;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with default values.
   */
  public MatchAttribute()
  {
    super();
  }

  /**
   * Initialize the new instance with a name.
   * @param name The name of the attribute
   */
  public MatchAttribute(String name)
  {
    super();
    setAttributeName(name);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns the name of the attribute that will be checked by this rule element
   */
  public String getAttributeName()
  {
    return this.attributeName;
  }

  /**
   * Sets the name of the attribute that will be checked by this rule element
   */
  public void setAttributeName(String newValue)
  {
    this.attributeName = newValue;
  }

  /**
   * Returns the value pattern(s) against that will be matched
   */
  public StringPattern[] getPatterns()
  {
    return this.patterns;
  }

  /**
   * Sets the value pattern(s) against that will be matched
   */
  public void setPatterns(StringPattern[] newValue)
  {
    this.patterns = newValue;
  }

  /**
   * Returns true, if the attribute name should be treated not case-sensitive.
   */
  public boolean ignoreCaseInName()
  {
    return this.ignoreCaseInName;
  }

  /**
   * Returns true, if the element is an attribute element.
   * <br>
   * Here this method always returns true.
   */
  @Override
  public boolean isAttribute()
  {
    return true;
  }

  /**
   * Sets the specified pattern as the sole pattern to be checked when
   * matching this attribute against a map.
   */
  public void setPattern(StringPattern aPattern)
  {
    StringPattern[] p = new StringPattern[1];
    p[0] = aPattern;
    setPatterns(p);
  }

  /**
   * Returns a string containing the attribute name, the operator and the
   * value(s) set in this part of a match rule.
   */
  @Override
  public String toString()
  {
    StringBuffer str = new StringBuffer(40);
    boolean hasValueList = false;

    if (getNot())
    {
      str.append(MatchRuleChars.DEFAULT_NOT_CHAR);
    }

    str.append(getAttributeName());
    hasValueList = getPatterns().length > 1;
    if (hasValueList)
    {
      str.append(MatchRuleChars.DEFAULT_VALUE_START_CHAR);
      for (int i = 0; i < getPatterns().length; i++)
      {
        if (i > 0)
        {
          str.append(MatchRuleChars.DEFAULT_VALUE_SEP_CHAR);
        }

        str.append((getPatterns()[i]).getPattern());
      }
      str.append(MatchRuleChars.DEFAULT_VALUE_END_CHAR);
    }
    else
    {
      switch (operator())
      {
        case OPERATOR_EQUALS :
          str.append(MatchRuleChars.DEFAULT_EQUALS_CHAR);
          break;
        case OPERATOR_GREATER :
          str.append(MatchRuleChars.DEFAULT_GREATER_CHAR);
          break;
        case OPERATOR_GREATER_OR_EQUAL :
          str.append(MatchRuleChars.DEFAULT_GREATER_CHAR);
          str.append(MatchRuleChars.DEFAULT_EQUALS_CHAR);
          break;
        case OPERATOR_LESS :
          str.append(MatchRuleChars.DEFAULT_LESS_CHAR);
          break;
        case OPERATOR_LESS_OR_EQUAL :
          str.append(MatchRuleChars.DEFAULT_LESS_CHAR);
          str.append(MatchRuleChars.DEFAULT_EQUALS_CHAR);
          break;
      }
      str.append(getPatterns()[0].toString());
    }

    return str.toString();
  }

  /**
   * Sets the operator for value comparisons of this attribute to EQUALS.
   */
  public void setEqualsOperator()
  {
    operator(OPERATOR_EQUALS);
  }

  /**
   * Sets the operator for value comparisons of this attribute to GREATER.
   */
  public void setGreaterOperator()
  {
    operator(OPERATOR_GREATER);
  }

  /**
   * Sets the operator for value comparisons of this attribute to LESS.
   */
  public void setLessOperator()
  {
    operator(OPERATOR_LESS);
  }

  /**
   * Sets the operator for value comparisons of this attribute to GREATER OR EQUAL.
   */
  public void setGreaterOrEqualOperator()
  {
    operator(OPERATOR_GREATER_OR_EQUAL);
  }

  /**
   * Sets the operator for value comparisons of this attribute to LESS OR EQUAL.
   */
  public void setLessOrEqualOperator()
  {
    operator(OPERATOR_LESS_OR_EQUAL);
  }

  /**
   * Sets the datatype this attribute's value must have. 
   * Implicitly the current value (pattern) gets converted to that datatype.
   * <p>
   * Currently supported datatypes are:
   * <ul>
   * <li>Float.class
   * <li>Double.class
   * <li>BigDecimal.class
   * <li>Integer.class
   * <li>Long.class
   * <li>String.class
   * <li>SimpleDateFormat
   * </ul>
   * 
   * @param type The type of the attribute's value
   * @throws MatchRuleException if the current value (pattern) cannot be converted to the specified datatype
   */
  public void setDatatype(Object type) throws MatchRuleException
  {
    if ((type == null) || (type == String.class))
    {
      setValueType(null);
      return;
    }
    convertToType(type);
    setValueType(type);
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================

  @Override
  protected boolean doMatch(Map<String, ?> dictionary)
  {
    Object value = null;

    value = valueInMap(dictionary);
    if (value == null)
      return false;

    if (isTyped())
    {
      return doTypedMatch(value);
    }

    if (value instanceof String)
    {
      return matchValue(value);
    }
    else if (value instanceof String[])
    {
      return matchValueArray((Object[])value);
    }
    else if (value instanceof java.util.Collection)
    {
      return matchValueCollection((Collection<?>)value);
    }
    else if (value instanceof Integer)
    {
      return matchValue(value);
    }
    else if (value instanceof IIntegerRepresentation)
    {
      return matchValue(value);
    }
    else if (value instanceof Integer[])
    {
      return matchValueArray((Object[])value);
    }
    else if (value instanceof IStringRepresentation)
    {
      return matchValue(value);
    }

    return false;
  }

  protected boolean doTypedMatch(final Object object)
  {
    Object value = object;

    if (!isCorrectType(value))
    {
      value = TYPE_CONVERTER.convertToType(object, getValueType());
    }

    if (isCorrectType(value))
    {
      try
      {
        if (getValueType() == Float.class)
        {
          return doFloatMatch(value);
        }
        if (getValueType() == Double.class)
        {
          return doDoubleMatch(value);
        }
        if (getValueType() == BigDecimal.class)
        {
          return doBigDecimalMatch(value);
        }
        if (getValueType() == Integer.class)
        {
          return doIntegerMatch(value);
        }
        if (getValueType() == Long.class)
        {
          return doLongMatch(value);
        }
        if (getValueType() instanceof SimpleDateFormat)
        {
          return doDateMatch(value);
        }
      }
      catch (@SuppressWarnings("unused") RuntimeException e)
      {
        return false; // Class cast exception - should not occur
      }
    }
    return false; // Type mismatch
  }

  protected boolean isCorrectType(Object value)
  {
    if ((getValueType() instanceof SimpleDateFormat) && (value.getClass() == Date.class))
    {
      return true;
    }
    return getValueType() == getTypeOf(value);
  }

  protected boolean doFloatMatch(Object value)
  {
    Float[] dataValues;

    dataValues = toArray(value, Float.class);
    for (int i = 0; i < dataValues.length; i++)
    {
      if (matchValueAgainstValues(dataValues[i], floatValues))
      {
        return true;
      }
    }
    return false;
  }

  protected boolean doDoubleMatch(Object value)
  {
    Double[] dataValues;

    dataValues = toArray(value, Double.class);
    for (int i = 0; i < dataValues.length; i++)
    {
      if (matchValueAgainstValues(dataValues[i], doubleValues))
        return true;
    }
    return false;
  }

  protected boolean doBigDecimalMatch(Object value)
  {
    BigDecimal[] dataValues;

    dataValues = toArray(value, BigDecimal.class);
    for (int i = 0; i < dataValues.length; i++)
    {
      if (matchValueAgainstValues(dataValues[i], bigDecimalValues))
      {
        return true;
      }
    }
    return false;
  }

  protected boolean doIntegerMatch(Object value)
  {
    Integer[] dataValues;

    dataValues = toArray(value, Integer.class);
    for (int i = 0; i < dataValues.length; i++)
    {
      if (matchValueAgainstValues(dataValues[i], integerValues))
      {
        return true;
      }
    }
    return false;
  }

  protected boolean doLongMatch(Object value)
  {
    Long[] dataValues;

    dataValues = toArray(value, Long.class);
    for (int i = 0; i < dataValues.length; i++)
    {
      if (matchValueAgainstValues(dataValues[i], longValues))
      {
        return true;
      }
    }
    return false;
  }

  protected boolean doDateMatch(Object value)
  {
    Date[] dataValues;

    dataValues = toArray(value, Date.class);
    for (int i = 0; i < dataValues.length; i++)
    {
      if (matchValueAgainstValues(dataValues[i], dateValues))
        return true;
    }
    return false;
  }

  protected boolean matchValueArray(Object[] values)
  {
    for (int i = 0; i < values.length; i++)
    {
      if (matchValue(values[i]))
      {
        return true;
      }
    }
    return false;
  }

  protected boolean matchValueCollection(Collection<?> values)
  {
    for (Object value : values)
    {      
      try
      {
        if (matchValue(value))
        {
          return true;
        }
      }
      catch (@SuppressWarnings("unused") Throwable t)
      {
        // Just ignore that value
      }
    }
    return false;
  }

  protected boolean matchValue(Object value)
  {
    if (value == null)
    {
      return false;
    }

    StringPattern pattern;
    String strValue;

    for (int i = 0; i < getPatterns().length; i++)
    {
      pattern = getPatterns()[i];
      if ((operator() == OPERATOR_EQUALS) && (pattern.hasWildcard()))
      {
        strValue = objectAsString(value);
        if (pattern.matches(strValue))
          return true;
      }
      else
      {
        Integer intValue = objectAsInteger(value);
        if (intValue != null)
        {
          if (compare(intValue, pattern.toString()))
          {
            return true;
          }
        }
        else
        {
          strValue = objectAsString(value);
          if (compare(strValue, pattern.toString(), pattern.getIgnoreCase()))
          {
            return true;
          }
        }
      }
    }

    return false;
  }

  protected <T> boolean matchValueAgainstValues(Comparable<T> value, T[] values)
  {
    int result;

    if (value == null)
    {
      return false;
    }

    for (int i = 0; i < values.length; i++)
    {
      result = value.compareTo(values[i]);
      if (compareIntegers(result, 0))
      {
        return true;
      }
    }
    return false;
  }

  /**
   * Returns true if the given value compared by using the current operator
   * to the rule value evaluates to true.
   */
  protected boolean compare(String value, String ruleValue, boolean ignoreCase)
  {
    int result;

    if (ignoreCase)
      result = value.compareToIgnoreCase(ruleValue);
    else
      result = value.compareTo(ruleValue);

    return compareIntegers(result, 0);
  }

  /**
   * Returns true if the given value compared by using the current operator
   * to the rule value evaluates to true.
   */
  protected boolean compare(Integer value, String ruleValue)
  {
    int ruleIntValue;

    try
    {
      ruleIntValue = Integer.parseInt(ruleValue);
    }
    catch (@SuppressWarnings("unused") RuntimeException e)
    {
      return false;
    }
    return compareIntegers(value.intValue(), ruleIntValue);
  }

  protected boolean compareIntegers(int a, int b)
  {
    switch (operator())
    {
      case OPERATOR_EQUALS :
        return (a == b);
      case OPERATOR_GREATER_OR_EQUAL :
        return (a >= b);
      case OPERATOR_LESS_OR_EQUAL :
        return a <= b;
      case OPERATOR_GREATER :
        return (a > b);
      case OPERATOR_LESS :
        return a < b;
    }
    return false;
  }

  @Override
  protected void ignoreCase(boolean ignoreIt)
  {
    for (int i = 0; i < getPatterns().length; i++)
      getPatterns()[i].setIgnoreCase(ignoreIt);
  }

  @Override
  protected void multiCharWildcardMatchesEmptyString(boolean yesOrNo)
  {
    for (int i = 0; i < getPatterns().length; i++)
    {
      getPatterns()[i].multiCharWildcardMatchesEmptyString(yesOrNo);
    }
  }

  @Override
  protected void apply(MatchRuleVisitor visitor)
  {
    String[] values = new String[getPatterns().length];

    for (int i = 0; i < getPatterns().length; i++)
    {
      values[i] = getPatterns()[i].getPattern();
    }

    visitor.attribute(getAttributeName(), operator(), values, getAnd(), getNot());
  }

  protected Object valueInMap(Map<String, ?> map)
  {
    String attrName;

    attrName = nameOfAttribute(map);
    if (attrName == null)
    {
      return null;
    }
    return map.get(attrName);
  }

  protected String nameOfAttribute(Map<String, ?> map)
  {
    String name;
    Set<String> keyNames;
    String key;
    Iterator<String> iterator;

    name = getAttributeName();
    if (ignoreCaseInName())
    {
      keyNames = map.keySet();
      iterator = keyNames.iterator();
      while (iterator.hasNext())
      {
        key = iterator.next();
        if (name.equalsIgnoreCase(key))
        {
          return key;
        }
      }
      name = null;
    }
    return name;
  }

  protected void convertToType(Object type) throws MatchRuleException
  {
    String[] strValues;

    strValues = new String[getPatterns().length];
    for (int i = 0; i < getPatterns().length; i++)
    {
      strValues[i] = getPatterns()[i].getPattern();
    }

    if (type == Float.class)
    {
      convertToFloat(strValues);
      return;
    }
    if (type == Double.class)
    {
      convertToDouble(strValues);
      return;
    }
    if (type == BigDecimal.class)
    {
      convertToBigDecimal(strValues);
      return;
    }
    if (type == Integer.class)
    {
      convertToInteger(strValues);
      return;
    }
    if (type == Long.class)
    {
      convertToLong(strValues);
      return;
    }
    if (type instanceof SimpleDateFormat)
    {
      convertToDate(strValues, (SimpleDateFormat)type);
      return;
    }
    throw new MatchRuleException("Type " + type + " not supported.");
  }

  protected void convertToFloat(String[] strValues) throws MatchRuleException
  {
    floatValues = new Float[strValues.length];

    for (int i = 0; i < strValues.length; i++)
    {
      try
      {
        floatValues[i] = Float.valueOf(strValues[i]);
      }
      catch (@SuppressWarnings("unused") NumberFormatException e)
      {
        throw createTypeConversionException(strValues[i], Float.class);
      }
    }
  }

  protected void convertToDouble(String[] strValues) throws MatchRuleException
  {
    doubleValues = new Double[strValues.length];

    for (int i = 0; i < strValues.length; i++)
    {
      try
      {
        doubleValues[i] = Double.valueOf(strValues[i]);
      }
      catch (@SuppressWarnings("unused") NumberFormatException e)
      {
        throw createTypeConversionException(strValues[i], Double.class);
      }
    }
  }

  protected void convertToBigDecimal(String[] strValues) throws MatchRuleException
  {
    bigDecimalValues = new BigDecimal[strValues.length];

    for (int i = 0; i < strValues.length; i++)
    {
      try
      {
        bigDecimalValues[i] = new BigDecimal(strValues[i]);
      }
      catch (@SuppressWarnings("unused") NumberFormatException e)
      {
        throw createTypeConversionException(strValues[i], BigDecimal.class);
      }
    }
  }

  protected void convertToInteger(String[] strValues) throws MatchRuleException
  {
    integerValues = new Integer[strValues.length];

    for (int i = 0; i < strValues.length; i++)
    {
      try
      {
        integerValues[i] = Integer.valueOf(strValues[i]);
      }
      catch (@SuppressWarnings("unused") NumberFormatException e)
      {
        throw createTypeConversionException(strValues[i], Integer.class);
      }
    }
  }

  protected void convertToLong(String[] strValues) throws MatchRuleException
  {
    longValues = new Long[strValues.length];

    for (int i = 0; i < strValues.length; i++)
    {
      try
      {
        longValues[i] = Long.valueOf(strValues[i]);
      }
      catch (@SuppressWarnings("unused") NumberFormatException e)
      {
        throw createTypeConversionException(strValues[i], Long.class);
      }
    }
  }

  protected void convertToDate(String[] strValues, SimpleDateFormat dateFormat) throws MatchRuleException
  {
    dateValues = new Date[strValues.length];

    for (int i = 0; i < strValues.length; i++)
    {
      try
      {
        dateValues[i] = dateFormat.parse(strValues[i]);
      }
      catch (@SuppressWarnings("unused") ParseException e)
      {
        throw new MatchRuleException("Unable to convert '" + strValues[i] + "' to Date with format \"" + dateFormat.toPattern() + "\" for attribute <" + getAttributeName() + ">");
      }
    }
  }

  protected MatchRuleException createTypeConversionException(String value, Class<?> type)
  {
    return new MatchRuleException("Unable to convert '%s' to %s for attribute <%s>", value, type, getAttributeName());
  }

  protected boolean isTyped()
  {
    return getValueType() != null;
  }

  /**
   * Returns the type of the given object or if it is an array or a list the
   * type of its first element.
   */
  protected Class<?> getTypeOf(Object object)
  {
    try
    {
      if (object instanceof List)
      {
        return ((List<?>)object).get(0).getClass();
      }
      if (object instanceof Collection)
      {
        return ((Collection<?>)object).iterator().next().getClass();
      }
      if (object.getClass().isArray())
      {
        return object.getClass().getComponentType();
      }
      return object.getClass();
    }
    catch (@SuppressWarnings("unused") RuntimeException e)
    {
      return null;
    }
  }

  @SuppressWarnings("unchecked")
  protected <T> T[] toArray(Object object, Class<T> type)
  {
    T[] array;
    Collection<?> coll;

    if (object instanceof Collection<?>)
    {
      coll = (Collection<?>)object;
      array = (T[])Array.newInstance(type, coll.size());
      return coll.toArray(array);
    }
    if (object.getClass().isArray())
    {
      return (T[])object;
    }
    array = (T[])Array.newInstance(type, 1);
    array[0] = (T)object;
    return array;
  }

  @Override
  protected void applyDatatypes(Map<String, Class<?>> datatypes) throws MatchRuleException
  {
    setDatatype(valueInMap(datatypes));
  }

  /**
   * Returns the string representation of the given object.
   * 
   * @param object The object to get the string representation of (may be null).
   * @return A string representation of the given object or null if the given object was null.
   */
  protected String objectAsString(Object object) 
  {
    if (object == null)
    {
      return null;
    }
    if (object instanceof String)
    {
      return (String)object;
    }
    if (object instanceof IStringRepresentation)
    {
      IStringRepresentation str = (IStringRepresentation)object;
      return str.asString();
    }
    return object.toString();
  }
  
  /**
   * Returns the integer representation of the given object or null if it is
   * no integer.
   */
  private Integer objectAsInteger(Object value)
  {
    if (value instanceof Integer)
    {
      return (Integer)value;
    }
    if (value instanceof IIntegerRepresentation)
    {
      return ((IIntegerRepresentation)value).asInteger();
    }
    return null;
  }

  /**
   * Sets whether the attribute name should be treated not case-sensitive.
   */
  @Override
  protected void ignoreCaseInName(boolean newValue)
  {
    this.ignoreCaseInName = newValue;
  }

  protected Object getValueType()
  {
    return this.valueType;
  }

  protected void setValueType(Object newValue)
  {
    this.valueType = newValue;
  }

  protected MatchRuleCompareOperator operator()
  {
    return this.operator;
  }

  protected void operator(MatchRuleCompareOperator newValue)
  {
    this.operator = newValue;
  }
}
