// ===========================================================================
// CONTENT  : CLASS LdapMatchRuleVisitor
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.2 - 27/12/2002
// HISTORY  :
//  17/08/2001  duma  CREATED
//  12/11/2001  duma  changed ->  Redesign with inner class GroupInfo
//	27/12/2002	duma	changed ->  Redesign with inner class AttributeInfo
//
// Copyright (c) 2001-2002, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

/**
 * Walks over a MatchRule to create a LDAP filter string out of it.
 *
 * @author Manfred Duchrow
 * @version 1.2
 */
@SuppressWarnings("synthetic-access")
public class LdapMatchRuleVisitor implements MatchRuleVisitor
{
  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private StringBuffer buffer = null;
  private GroupInfo rootGroup = null;
  private Stack<GroupInfo> stack = null;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with default values.
   */
  public LdapMatchRuleVisitor()
  {
    super();
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * This method will be called right before the MatchRule walks
   * through its elements.
   */
  @Override
  public void walkThroughInit()
  {
    setBuffer(new StringBuffer(100));
    setStack(new Stack<GroupInfo>());
  }

  /**
   * This method will be called when the MatchRule has finished to walk
   * through its elements.
   */
  @Override
  public void walkThroughFinished()
  {
    appendGroup(getRootGroup());
  }

  /**
   * This method will be called for each start of a new group.
   *
   * @param andOperator If true it is an AND combination otherwise it is OR
   * @param notOperator Is only true for a NOT operation
   */
  @Override
  public void startGroup(boolean andOperator, boolean notOperator)
  {
    GroupInfo group = null;

    group = createGroup();
    group.not_flag = notOperator;
    if (isFirstGroup())
    {
      setRootGroup(group);
    }
    else
    {
      addElementToCurrentGroup(group, andOperator);
    }
    push(group);
  }

  /**
   * This method will be called for each group end occurrence.
   */
  @Override
  public void endGroup()
  {
    pop();
  }

  /**
   * This method will be called for each attribute.
   *
   * @param name The attribute's name
   * @param compareOperator The operator used to compare values
   * @param values All values the attribute my match (implicit OR combination !)
   * @param andOperator If true it is an AND combination otherwise it is OR
   * @param notOperator Is only true for a NOT operation
   */
  @Override
  public void attribute(String name, MatchRuleCompareOperator compareOperator, String[] values, boolean andOperator, boolean notOperator)
  {
    AttributeInfo element = new AttributeInfo();

    element.name = name;
    element.operator = compareOperator;
    element.values = values;
    element.not_flag = notOperator;

    addElementToCurrentGroup(element, andOperator);
  }

  /**
   * Converts the given match rule into a LDAP search string
   * compliant to RFC 1558.
   *
   * @param matchRule The rule to be converted
   */
  public String asSearchString(MatchRule matchRule)
  {
    matchRule.apply(this);
    return getBuffer().toString();
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================

  public void appendGroup(GroupInfo group)
  {
    if (group.not_flag)
      getBuffer().append("(!");

    if (group.neutral_elements == null)
    {
      appendGroupElements(group.and_elements, true);
      appendGroupElements(group.or_elements, false);
    }
    else
    {
      appendGroupElement(group.neutral_elements.get(0));
    }

    if (group.not_flag)
    {
      getBuffer().append(')');
    }
  }

  protected void appendGroupElement(ElementInfo element)
  {
    if (element.isGroupInfo()) // is it a group ?
    {
      appendGroup((GroupInfo)element);
    }
    else
    {
      appendAttribute((AttributeInfo)element);
    }
  }

  protected void appendGroupElements(List<ElementInfo> elements, boolean andOperator)
  {
    if (elements.size() == 0)
    {
      return;
    }

    getBuffer().append('(');
    getBuffer().append(andOperator ? '&' : '|');
    for (int i = 0; i < elements.size(); i++)
    {
      appendGroupElement(elements.get(i));
    }
    getBuffer().append(')');
  }

  protected void appendAttribute(AttributeInfo element)
  {
    appendAttribute(element.name, element.operator, element.values, element.not_flag);
  }

  protected void appendAttribute(String name, MatchRuleCompareOperator operator, String[] values, boolean notOperator)
  {
    boolean manyValues = false;

    manyValues = values.length > 1;

    if (notOperator)
    {
      getBuffer().append("(!");
    }
    if (manyValues)
    {
      getBuffer().append("(|");
    }

    for (int i = 0; i < values.length; i++)
    {
      appendAttribute(name, operator, values[i]);
    }

    if (manyValues)
    {
      getBuffer().append(")");
    }
    if (notOperator)
    {
      getBuffer().append(")");
    }
  }

  protected void appendAttribute(String name, MatchRuleCompareOperator operator, String value)
  {
    getBuffer().append('(');
    getBuffer().append(name);
    switch (operator)
    {
      case OPERATOR_EQUALS :
        getBuffer().append('=');
        break;
      case OPERATOR_GREATER_OR_EQUAL :
        getBuffer().append(">=");
        break;
      case OPERATOR_LESS_OR_EQUAL :
        getBuffer().append("<=");
        break;
      default :
        getBuffer().append('=');
    }
    getBuffer().append(value);
    getBuffer().append(')');
  }

  protected void addElementToCurrentGroup(ElementInfo element, boolean andOperator)
  {
    GroupInfo group = null;
    List<ElementInfo> groupSlot = null;

    group = currentGroup();
    groupSlot = andOperator ? group.and_elements : group.or_elements;
    if (group.neutral_elements == null)
    {
      groupSlot.add(element);
    }
    else
    {
      if (group.neutral_elements.size() == 0)
      {
        group.neutral_elements.add(element);
      }
      else
      {
        groupSlot.add(group.neutral_elements.get(0));
        groupSlot.add(element);
        group.neutral_elements = null;
      }
    }
  }

  protected GroupInfo createGroup()
  {
    return new GroupInfo();
  }

  protected GroupInfo pop()
  {
    return getStack().pop();
  }

  protected void push(GroupInfo obj)
  {
    getStack().push(obj);
  }

  protected GroupInfo currentGroup()
  {
    return getStack().peek();
  }

  protected boolean isFirstGroup()
  {
    return getStack().empty();
  }

  protected StringBuffer getBuffer()
  {
    return buffer;
  }

  protected void setBuffer(StringBuffer newValue)
  {
    buffer = newValue;
  }

  protected GroupInfo getRootGroup()
  {
    return rootGroup;
  }

  protected void setRootGroup(GroupInfo newValue)
  {
    rootGroup = newValue;
  }

  protected Stack<GroupInfo> getStack()
  {
    return stack;
  }

  protected void setStack(Stack<GroupInfo> newValue)
  {
    stack = newValue;
  }

  private class AttributeInfo extends ElementInfo
  {
    protected String name = null;
    protected MatchRuleCompareOperator operator = MatchRuleCompareOperator.OPERATOR_EQUALS;
    protected String[] values = null;
  }

  // ---- INNER CLASS --------------------------------------------------------

  private class ElementInfo
  {
    protected boolean not_flag = false;

    protected boolean isGroupInfo()
    {
      return false;
    }
  }

  private class GroupInfo extends ElementInfo
  {
    protected List<ElementInfo> neutral_elements = new ArrayList<ElementInfo>();
    protected List<ElementInfo> and_elements = new ArrayList<ElementInfo>();
    protected List<ElementInfo> or_elements = new ArrayList<ElementInfo>();

    @Override
    protected boolean isGroupInfo()
    {
      return true;
    }
  }
}