// ===========================================================================
// CONTENT  : CLASS FormattedTextWriter
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.0 - 14/07/2019
// HISTORY  :
//  14/07/2019  mdu  CREATED
//
// Copyright (c) 2019, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.IllegalFormatException;
import java.util.concurrent.atomic.AtomicBoolean;

import org.pfsw.bif.exceptions.IORuntimeException;
import org.pfsw.bif.text.ICharsets;
import org.pfsw.bif.text.IStringRepresentation;

/**
 * This is a wrapper on an {@link Appendable} object.
 * It supports writing to the underlying appendable and particularly provides
 * easy to use indentation.
 *
 * @author Manfred Duchrow
 * @version 1.0
 */
public class FormattedTextWriter implements Appendable, Closeable
{
  private final Appendable output;
  private final AtomicBoolean isOpen = new AtomicBoolean(true);
  private int indentLevel = 0;
  private int indentSize = 2;
  private String newline = NewLine.CURRENT_OS.asString();
  private boolean onNewLine = true; 

  // =========================================================================
  // CLASS METHODS
  // =========================================================================
  /**
   * Creates a new instance that writes to the given output object.
   * 
   * @param output The target object to append data to (must not be null).
   * @return The new created writer.
   * @throws IllegalArgumentException if the given output is null.
   */
  public static FormattedTextWriter create(Appendable output)
  {
    return new FormattedTextWriter(output);
  }
  
  /**
   * Creates a new instance that writes to the given output object an indentation size.
   * 
   * @param output The target object to append data to (must not be null).
   * @param indentSize The number of spaces per indentation level (must be in the range of 0-20).
   * @return The new created writer.
   * @throws IllegalArgumentException if the given output is null.
   * @throws IllegalArgumentException if the given indentation size is less than 0 or greater than 20.
   */
  public static FormattedTextWriter create(Appendable output, int indentSize)
  {
    FormattedTextWriter textWriter = create(output);
    textWriter.setIndentSize(indentSize);
    return textWriter;
  }
  
  /**
   * Creates a new instance that writes to the given output object, indentation size and newline characters.
   * 
   * @param output The target object to append data to (must not be null).
   * @param indentSize The number of spaces per indentation level (must be in the range of 0-20).
   * @param newLine The new line character sequence to use.
   * @return The new created writer.
   * @throws IllegalArgumentException if the given output is null.
   * @throws IllegalArgumentException if the given indentation size is less than 0 or greater than 20.
   */
  public static FormattedTextWriter create(Appendable output, int indentSize, NewLine newLine)
  {
    FormattedTextWriter textWriter = create(output, indentSize);
    textWriter.setNewline(newLine.asString());
    return textWriter;
  }
  
  /**
   * Creates a new instance that writes to the given output file with the specified character encoding.
   * 
   * @param outputFile The target file to append data to (must not be null).
   * @param charEncoding The encoding of the text to be written to the file.
   * @throws IOException if the file cannot be found or not be opened.
   * @throws IllegalArgumentException if the given outputFile or charEncoding is null.
   */
  public static FormattedTextWriter create(File outputFile, Charset charEncoding) throws IOException
  {
    return new FormattedTextWriter(outputFile, charEncoding);
  }
  
  /**
   * Creates a new instance that writes to the given output file with UTF-8 character encoding.
   * 
   * @param outputFile The target file to append data to (must not be null).
   * @throws IOException if the file cannot be found or not be opened.
   * @throws IllegalArgumentException if the given outputFile or charEncoding is null.
   */
  public static FormattedTextWriter create(File outputFile) throws IOException
  {
    return new FormattedTextWriter(outputFile);
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Creates a new instance that writes to the given output object.
   * 
   * @param output The target object to append data to (must not be null).
   * @throws IllegalArgumentException if the given output is null.
   */
  public FormattedTextWriter(Appendable output)
  {
    super();
    if (output == null)
    {
      throw new IllegalArgumentException("output must not be null");
    }
    this.output = output;
  }

  /**
   * Creates a new instance that writes to the given output file with the specified character encoding.
   * 
   * @param outputFile The target file to append data to (must not be null).
   * @param charEncoding The encoding of the text to be written to the file.
   * @throws IOException if the file cannot be found or not be opened.
   * @throws IllegalArgumentException if the given outputFile or charEncoding is null.
   */
  public FormattedTextWriter(File outputFile, Charset charEncoding) throws IOException
  {
    super();
    if (outputFile == null)
    {
      throw new IllegalArgumentException("outputFile must not be null");
    }
    if (charEncoding == null)
    {
      throw new IllegalArgumentException("charEncoding must not be null");
    }
    Writer writer = new OutputStreamWriter(new FileOutputStream(outputFile), charEncoding);
    this.output = writer;
  }
  
  /**
   * Creates a new instance that writes to the given output file with UTF-8 character encoding.
   * 
   * @param outputFile The target file to append data to (must not be null).
   * @throws IOException if the file cannot be found or not be opened.
   * @throws IllegalArgumentException if the given outputFile or charEncoding is null.
   */
  public FormattedTextWriter(File outputFile) throws IOException
  {
    this(outputFile, ICharsets.UTF_8);
  }
  
  /**
   * Replaces any placeholders (e.g. %s, %d) in text by the specified arguments an
   * appends it to the underlying output. 
   * 
   * @param text The text to write.
   * @param args The optional arguments to replace the placeholders in text.
   * @return This object to allow fluent API usage.
   * @throws IllegalFormatException
   *          If a format string contains an illegal syntax, a format
   *          specifier that is incompatible with the given arguments,
   *          insufficient arguments given the format string, or other
   *          illegal conditions.  For specification of all possible
   *          formatting errors, see the <a
   *          href="../util/Formatter.html#detail">Details</a> section of the
   *          formatter class specification.
   *
   * @throws  NullPointerException
   *          If the <tt>text</tt> is <tt>null</tt>
   *  
   * @throws IORuntimeException
   *          If any IOException occurred by writing to the underlying output.
   *          
   * @throws IllegalStateException
   *          If this writer has been closed.
   */
  public FormattedTextWriter write(String text, Object... args)
  {
    checkOpenState();
    if (indentationRequired())
    {
      writeIndentation();
    }
    try
    {
      getOutput().append(String.format(text, args));
    }
    catch (IOException e)
    {
      throw new IORuntimeException(e);
    }
    return this;
  }

  /**
   * The same as {@link #write(String, Object...)}, except that only the given
   * object is converted to its string representation. 
   */
  public FormattedTextWriter write(IStringRepresentation object)
  {
    return write(object.asString());
  }
    
  /**
   * Replaces any placeholders (e.g. %s, %d) in text by the specified arguments an
   * appends it to the underlying output, followed by a line break. 
   * The placeholder replacement is based on {@link String#format(String, Object...)}. 
   * 
   * @param text The text to write.
   * @param args The optional arguments to replace the placeholders in text.
   * @return This object to allow fluent API usage.
   * @throws IllegalFormatException
   *          If a format string contains an illegal syntax, a format
   *          specifier that is incompatible with the given arguments,
   *          insufficient arguments given the format string, or other
   *          illegal conditions.  For specification of all possible
   *          formatting errors, see the <a
   *          href="../util/Formatter.html#detail">Details</a> section of the
   *          formatter class specification.
   *
   * @throws  NullPointerException
   *          If the <tt>text</tt> is <tt>null</tt>
   *  
   * @throws IORuntimeException
   *          If any IOException occurred by writing to the underlying output.
   * 
   * @throws IllegalStateException
   *          If this writer has been closed.
   */
  public FormattedTextWriter writeln(String text, Object... args)
  {
    write(text, args);
    newLine();
    return this;
  }

  /**
   * The same as {@link #writeln(String, Object...)}, except that only the given
   * object is converted to its string representation. 
   */
  public FormattedTextWriter writeln(IStringRepresentation object)
  {
    return writeln(object.asString());
  }
    
  /**
   * Writes the newline character(s).
   */
  public FormattedTextWriter newLine()
  {
    write(getNewline());
    onNewLine = true;
    return this;
  }

  /**
   * Increases the indentation level by 1. 
   */
  public FormattedTextWriter indent()
  {
    this.indentLevel++;
    return this;
  }

  /**
   * Decreases the indentation level by 1. 
   */
  public FormattedTextWriter outdent()
  {
    if (this.indentLevel > 0)
    {
      this.indentLevel--;
    }
    return this;
  }

  /**
   * @throws IllegalStateException If this writer has been closed.
   */
  @Override
  public Appendable append(char c) throws IOException
  {
    write(String.valueOf(c));
    return this;
  }

  /**
   * @throws IllegalStateException If this writer has been closed.
   */
  @Override
  public Appendable append(CharSequence csq) throws IOException
  {
    write(csq.toString());
    return this;
  }

  /**
   * @throws IllegalStateException If this writer has been closed.
   */
  @Override
  public Appendable append(CharSequence csq, int start, int end) throws IOException
  {
    write(csq.subSequence(start, end).toString());
    return this;
  }

  @Override
  public void close() throws IOException
  {
    if (getOutput() instanceof Closeable)
    {
      Closeable closeable = (Closeable)getOutput();
      closeable.close();
    }
    this.isOpen.set(false);
  }

  /**
   * Returns the number of spaces to be used for indentation per indentation level. 
   */
  public int getIndentSize()
  {
    return this.indentSize;
  }

  /**
   * Sets the number of spaces to be used for indentation per indentation level. 
   * 
   * @param indentSize The number of spaces per indentation level (must be in the range of 0-20).
   * @throws IllegalArgumentException if the given size is less than 0 or greater than 20.
   */
  public FormattedTextWriter setIndentSize(int indentSize)
  {
    if ((indentSize < 0) || (indentSize > getMaxIndentSize()))
    {
      throw new IllegalArgumentException(String.format("Invalid indentation size %d. It must be in the range of 0..%d", indentSize, getMaxIndentSize()));
    }
    this.indentSize = indentSize;
    return this;
  }

  /**
   * Returns the new line character(s) used by this writer.
   */
  public String getNewline()
  {
    return this.newline;
  }

  /**
   * Sets the new line character(s) to be used by this writer.
   * @throws IllegalArgumentException if the given string is null.
   */
  public FormattedTextWriter setNewline(String newline)
  {
    if (newline == null)
    {
      throw new IllegalArgumentException("newline must not be null");
    }
    this.newline = newline;
    return this;
  }

  public int getIndentLevel()
  {
    return this.indentLevel;
  }

  protected Appendable getOutput()
  {
    return this.output;
  }

  protected int getMaxIndentSize()
  {
    return 20;
  }

  protected boolean indentationRequired() 
  {
    return onNewLine && (getNumberOfIndentSpaces() > 0);
  }
  
  protected int getNumberOfIndentSpaces()
  {
    return getIndentLevel() * getIndentSize();
  }

  protected FormattedTextWriter writeIndentation()
  {
    onNewLine = false;
    int numSpaces = getNumberOfIndentSpaces();
    if (numSpaces > 0)
    {      
      write(StringUtil.current().repeat(' ', numSpaces));
    }
    return this;
  }

  protected void checkOpenState()
  {
    if (!this.isOpen.get())
    {
      throw new IllegalStateException("This writer has been closed, no further data appending to it allowed.");
    }
  }
}