// ===========================================================================
// CONTENT  : CLASS DefaultMatchRuleParser
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.4 - 20/12/2004
// HISTORY  :
//  23/08/2002  duma  CREATED
//	22/11/2002	duma	added		->	Supports special characters in attribute names
//	01/01/2003	duma	added		->	Parsing of <, >, <=, >=
//	24/10/2003	duma	changed	->	parse( String rule ), added createMatchRuleOn()
//	04/12/2003	duma	changed	->	Throw exception if closing parenthesis is missing
//	20/12/2004	duma	added		->	static create(), static create(chars)
//
// Copyright (c) 2002-2004, by Manfred Duchrow. All rights reserved.
// ===========================================================================
package org.pfsw.text;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * This parser translates the match-rule syntax to a MatchRule object.
 * <p>
 * <b>It is recommended to use the static create() methods rather than the 
 * constructors to get a new parser instance.</b>
 * <p>
 * <h4>Example 1</h4>
 * Here is a sample rule:
 * <p>
 * <em>firstName=John & lastName=M*</em>
 * <p>
 * It means that attribute 'firstName' must have the value <i>"John"</i> 
 * and that the attribute 'lastName' must match <i>"M*"</i>, that is all 
 * values starting with a capital 'M' will evaluate to true.
 * <p>
 * A slightly different syntax for the above rule is:
 * <p>
 * <em>firstName{John} & lastName{M*}</em>
 * <p>
 * Using the following dictionary will evaluate the rules from above to false:
 * <p>
 * <pre>
 * Map map = new HashMap() ;
 * map.put( "firstName", "Conner" ) ;
 * map.put( "lastName", "McLeod" ) ;
 * MatchRule mr = DefaultMatchRuleParser.parseRule( "firstName{John} & lastName{M*}" ) ;
 * boolean found = mr.matches(map) ; // returns false
 * </pre>
 * <p>
 * The next dictionary will evaluate the rule to true:
 * <p>
 * <pre>
 * Map map = new HashMap() ;
 * map.put( "firstName", "John" ) ;
 * map.put( "lastName", "Miles" ) ;
 * MatchRule mr = DefaultMatchRuleParser.parseRule( "firstName{John} & lastName{M*}" ) ;
 * boolean found = mr.matches(map) ; // returns true
 * </pre> 
 * <p>
 * The parser generally supports the following comparisons in rules:
 * <table border="1" cellpadding=2">
 *   <tr>
 *     <td><b>Comparison</b></td>
 *     <td><b>Description</b></td>
 *     <td><b>Example(s)</b></td>
 *   </tr>
 *   <tr>
 *     <td>equals</td>
 *     <td>Compares if the value of the attribute is equal to the specified value</td>
 *     <td>name=Roger</td>
 *   </tr>
 *   <tr>
 *     <td>equals any</td>
 *     <td>Compares if the value of the attribute is equal to any of a list of specified values</td>
 *     <td>name{Fred,Peter,John,Mike}</td>
 *   </tr>
 *   <tr>
 *     <td>matches</td>
 *     <td>Compares if the value of the attribute matches the specified pattern</td>
 *     <td>name=J*y</td>
 *   </tr>
 *   <tr>
 *     <td>matches any</td>
 *     <td>Compares if the value of the attribute matches any of a list of specified values</td>
 *     <td>name{Fred,A*,B?n,R*so?,Carl}</td>
 *   </tr>
 *   <tr>
 *     <td>less</td>
 *     <td>Compares if the value of the attribute is less than the specified value</td>
 *     <td>name&lt;Henderson<br>
 *         age&lt;20
 *     </td>
 *   </tr>
 *   <tr>
 *     <td>greater</td>
 *     <td>Compares if the value of the attribute is greater than the specified value</td>
 *     <td>name&gt;Franklin<br>
 *         age&gt;15
 *     </td>
 *   </tr>
 *   <tr>
 *     <td>less or equal</td>
 *     <td>Compares if the value of the attribute is less or equal to the specified value</td>
 *     <td>name&lt;=Anderson<br>
 *         age&lt;=50
 *     </td>
 *   </tr>
 *   <tr>
 *     <td>greater or equal</td>
 *     <td>Compares if the value of the attribute is greater or equal to the specified value</td>
 *     <td>name&gt;=Franklin<br>
 *         age&gt;=3
 *     </td>
 *   </tr>
 * </table>
 * <p>
 * There are some characters with special purpose in a rule. 
 * The table below describes each of them and lists the method that can 
 * be used to change the character.
 * <table border="1" cellpadding=2">
 * <tr>
 *     <td><b>char</b></td> <td><b>Description</b></td> <td><b>Method to change in MatchRuleChars</b></td>
 * </tr>
 * <tr>
 *   <td>&</td> <td>AND operator</td> <td>setAndChar()</td>
 * </tr>
 * <tr>
 *   <td>|</td> <td>OR operator</td> <td>setOrChar( )</td>
 * </tr>
 * <tr>
 *   <td>!</td> <td>NOT operator</td> <td>setNotChar()</td>
 * </tr>
 * <tr>
 *   <td>{</td> <td>Starts a list of values</td> <td>setValueStartChar()</td>
 * </tr>
 * <tr>
 *   <td>,</td> <td>Separator of values in a value list</td> <td>setValueSeparatorChar()</td>
 * </tr>
 * <tr>
 *   <td>}</td> <td>Ends a list of values</td> <td>setValueEndChar()</td>
 * </tr>
 * <tr>
 *   <td>(</td> <td>Starts a group of attribute rules</td> <td>setGroupStartChar()</tr>
 * </tr>
 * <tr>
 *   <td>)</td> <td>Ends a group of attribute rules</td> <td>setGroupEndChar()</td>
 * </tr>
 * <tr>
 *   <td>=</td> <td>Compares equality of the attribute's value</td> <td>setEqualsChar()</td>
 * </tr>
 * <tr>
 *   <td>&lt;</td> <td>Compares the attribute's value to be less than the specified value</td> <td>setLessChar()</td>
 * </tr>
 * <tr>
 *   <td>&gt;</td> <td>Compares the attribute's value to be greater zhan the specified value</td> <td>setGreaterChar()</td>
 * </tr>
 * <tr>
 *   <td>?</td> <td>Wildcard for a single character in a value</td> <td>---</td>
 * </tr>
 * <tr>
 *   <td>*</td> <td>Wildcard for any count of characters in a value</td> <td>---</td>
 * </tr>
 * </table>
 * <p>
 * Any rule must comply to the following restrictions:
 * <ul>
 *   <li>A rule must not contain any carriage-return or line-feed characters!</li>
 *   <li>The NOT operator is only allowed after an AND or an OR operator!</li>
 *   <li>There might be any amount of spaces in front or after operators</li>
 *   <li>There might be any amount of spaces in front or after group parenthesis</li>
 *   <li>Attribute names must only consist of characters ( A-Z, a-z ) and/or digits ( 0-9 )
 *       and any additional characters that are specified in MatchRuleChars.setSpecialNameCharacters()
 *   </li>
 *   <li>Between an attribute name and the value list starting bracket might be any amount of whitespace characters.</li>
 *   <li>Between an attribute name and a compare operator might be any amount of whitespace characters.</li>
 *   <li>Any character inside a value list is treated as part of a value except the VALUE SEPARATOR, the VALUE LIST END CHARACTER and the two WILDCARD characters.</li>
 * </ul>
 * <p>
 * <h4>Example 2:</h4>
 * <p>
 * A more complex rule could look like this:
 * <p>
 * <em>( city{P*,Ch*} & ! city{Paris,Pretoria} ) | ( language{en,de,fr,it,es} & currency{??D} )</em>
 * <p>
 * The dictonary below will evaluate to true if checked against the above rule:
 * <p>
 * <pre>
 * DefaultMatchRuleParser parser = new DefaultMatchRuleParser() ;
 * MatchRule rule = parser.parse("( city{P*,Ch*} & ! city{Paris,Pretoria} ) | " +
 *  ( language{en,de,fr,it,es} & currency{??D} )" ) ;
 * Map map = new HashMap() ; 
 * map.put( "city", "Pittsburg" ) ;
 * map.put( "language", "en" ) ;
 * map.put( "currency", "USD" ) ;
 * boolean ok = rule.matches( map ) ;
 * </pre>
 * <p>
 * Whereas the following values produce a false:
 * <p>
 * <pre>
 * MatchRuleChars chars = new  MatchRuleChars() ;
 * chars.setValueSeparatorChar( ';' ) ;
 * DefaultMatchRuleParser parser = new DefaultMatchRuleParser(chars) ;
 * MatchRule rule = parser.parse( "( city{P*;Ch*} &amp; ! city{Paris;Pretoria} ) | " +
 *   ( language{en;de;fr;it;es} & currency{??D} )" ) ;
 * Map map = new HashMap() ;
 * map.put( "city", "Pretoria" ) ;
 * map.put( "language", "de" ) ;
 * map.put( "currency", "USD" ) ;
 * boolean ok = rule.matches( map ) ;
 * </pre>
 * <p>   
 *
 * @author Manfred Duchrow
 * @version 1.4
 */
public class DefaultMatchRuleParser extends BaseMatchRuleParser
{
  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private MatchRuleChars ruleChars = new MatchRuleChars();
  private boolean ignoreCaseInNames = false;
  private boolean ignoreCaseInValues = false;
  private boolean multiCharWildcardMatchesEmptyString = false;

  // =========================================================================
  // PUBLIC CLASS METHODS
  // =========================================================================
  /**
   * Parse the given rule string to a MatchRule object that can
   * be used to check attributes in a Map, if they match the rule.
   * This method creates a new parser for each call by invocation of a 
   * constructor.
   * 
   * @param rule The rule in a string compliant to the MatchRule syntax
   * @throws MatchRuleParseException Each syntax error in the given rule causes
   * 																	this exception with a short description
   * 																	of what is wrong
   */
  public static MatchRule parseRule(String rule) throws MatchRuleParseException
  {
    DefaultMatchRuleParser parser;

    parser = new DefaultMatchRuleParser();

    return parser.parse(rule);
  }

  /**
   * Returns a new parser that generates rules which treat the multi-char
   * wildcard (i.e. '*') in a way that it matches empty strings.
   * <br>
   * Using the constructor is different. In that case a multi-char wildcard
   * won't match an empty string!
   */
  public static DefaultMatchRuleParser create()
  {
    DefaultMatchRuleParser parser;

    parser = new DefaultMatchRuleParser();
    parser.setMultiCharWildcardMatchesEmptyString(true);
    return parser;
  }

  /**
   * Returns a new parser that generates rules which treat the multi-char
   * wildcard (i.e. '*') in a way that it matches empty strings.
   * <br>
   * Using the constructor is different. In that case a multi-char wildcard
   * won't match an empty string!
   * 
   * @param chars The charscter set that is used for the rules operators etc.
   */
  public static DefaultMatchRuleParser create(MatchRuleChars chars)
  {
    DefaultMatchRuleParser parser;

    parser = new DefaultMatchRuleParser(chars);
    parser.setMultiCharWildcardMatchesEmptyString(true);
    return parser;
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with default values.
   */
  public DefaultMatchRuleParser()
  {
    super();
  }

  /**
   * Initialize the new instance with a set of rule characters.
   */
  public DefaultMatchRuleParser(MatchRuleChars ruleCharacters)
  {
    this();
    if (ruleCharacters != null)
    {
      setRuleChars(ruleCharacters);
    }
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns true, if the parser produces MatchRules that treat
   * attribute names case-insensitive.
   */
  public boolean getIgnoreCaseInNames()
  {
    return ignoreCaseInNames;
  }

  /**
   * Sets whether or not the parser produces MatchRules that treat
   * attribute names case-insensitive.
   */
  public void setIgnoreCaseInNames(boolean newValue)
  {
    ignoreCaseInNames = newValue;
  }

  /**
   * Returns true, if the parser produces MatchRules that are case-insensitive
   * when comparing values. 
   */
  public boolean getIgnoreCaseInValues()
  {
    return ignoreCaseInValues;
  }

  /**
   * Sets whether or not the parser produces MatchRules that are 
   * case-insensitive when comparing values. 
   */
  public void setIgnoreCaseInValues(boolean newValue)
  {
    ignoreCaseInValues = newValue;
  }

  /**
   * Returns true, if this parser creates match rules that allow empty strings 
   * at the position of the multi character wildcard ('*').
   * <p>
   * The default value is false. 
   */
  public boolean getMultiCharWildcardMatchesEmptyString()
  {
    return multiCharWildcardMatchesEmptyString;
  }

  /**
   * Sets whether or not this parser creates match rules that allow empty 
   * strings at the position of the multi character wildcard ('*').
   * <p>
   * The default value is false. 
   */
  public void setMultiCharWildcardMatchesEmptyString(boolean newValue)
  {
    multiCharWildcardMatchesEmptyString = newValue;
  }

  /**
   * Parse the given rule string to a MatchRule object that can
   * be used to check attributes in a Map, if they match the rule.
   * 
   * @param rule The rule in a string compliant to the MatchRule syntax
   * @throws MatchRuleParseException Each syntax error in the given rule causes
   * 																	this exception with a short description
   * 																	of what is wrong
   */
  public MatchRule parse(String rule) throws MatchRuleParseException
  {
    MatchGroup group;

    group = parseToGroup(rule);
    if (group == null)
    {
      return null;
    }
    return createMatchRuleOn(group);
  }

  /**
   * Parse the given rule string to a MatchRule and apply the given datatypes
   * to it. Such a rule will do attribute comparisons according to the
   * corresponding datatype. 
   * 
   * @param rule The rule in a string compliant to the MatchRule syntax
   * @param datatypes The attributes and their associated datatypes
   * @throws MatchRuleParseException Each syntax error in the given rule causes
   * 																	this exception with a short description
   * 																	of what is wrong
   * @see MatchRule#setDatatypes(Map)
   */
  public MatchRule parseTypedRule(String rule, Map<String, Class<?>> datatypes) throws MatchRuleException
  {
    MatchRule matchRule;

    matchRule = parse(rule);
    matchRule.setDatatypes(datatypes);
    return matchRule;
  }

  /** 
   * Returns the character for AND operations ( DEFAULT = '&' ) 
   */
  public char getAndChar()
  {
    return getRuleChars().getAndChar();
  }

  /** 
   * Sets the character for AND operations 
   */
  public void setAndChar(char newValue)
  {
    getRuleChars().setAndChar(newValue);
  }

  /** 
   * Returns the character for OR operations ( DEFAULT = '|' ) 
   */
  public char getOrChar()
  {
    return getRuleChars().getOrChar();
  }

  /** 
   * Sets the character for OR operations 
   */
  public void setOrChar(char newValue)
  {
    getRuleChars().setOrChar(newValue);
  }

  /** 
   * Returns the character for NOT operations ( DEFAULT = '!' ) 
   */
  public char getNotChar()
  {
    return getRuleChars().getNotChar();
  }

  /** 
   * Sets the character for NOT operations 
   */
  public void setNotChar(char newValue)
  {
    getRuleChars().setNotChar(newValue);
  }

  /** 
   * Returns the character for separation of values ( DEFAULT = ',' ) 
   */
  public char getValueSeparatorChar()
  {
    return getRuleChars().getValueSeparatorChar();
  }

  /** 
   * Sets the character that separates values in a value list 
   */
  public void setValueSeparatorChar(char newValue)
  {
    getRuleChars().setValueSeparatorChar(newValue);
  }

  /** 
   * Returns the character that is used to enclose a value ( DEFAULT = '\'' ) 
   */
  public char getValueDelimiterChar()
  {
    return getRuleChars().getValueDelimiterChar();
  }

  /** 
   * Returns the character that starts a list of values ( DEFAULT = '{' ) 
   */
  public char getValueStartChar()
  {
    return getRuleChars().getValueStartChar();
  }

  /** 
   * Sets the character that starts a value list 
   */
  public void setValueStartChar(char newValue)
  {
    getRuleChars().setValueStartChar(newValue);
  }

  /** 
   * Returns the character ends a list of values ( DEFAULT = '}' ) 
   */
  public char getValueEndChar()
  {
    return getRuleChars().getValueEndChar();
  }

  /** 
   * Sets the character that ends a value list 
   */
  public void setValueEndChar(char newValue)
  {
    getRuleChars().setValueEndChar(newValue);
  }

  /** 
   * Returns the character that starts a logical group ( DEFAULT = '(' ) 
   */
  public char getGroupStartChar()
  {
    return getRuleChars().getGroupStartChar();
  }

  /** 
   * Sets the character that starts a group 
   */
  public void setGroupStartChar(char newValue)
  {
    getRuleChars().setGroupStartChar(newValue);
  }

  /** 
   * Returns the character that ends a logical group ( DEFAULT = ')' ) 
   */
  public char getGroupEndChar()
  {
    return getRuleChars().getGroupEndChar();
  }

  /** 
   * Sets the character that ends a group 
   */
  public void setGroupEndChar(char newValue)
  {
    getRuleChars().setGroupEndChar(newValue);
  }

  /** 
   * Returns the character for greater than comparisons ( DEFAULT = '>' ) 
   */
  public char getGreaterChar()
  {
    return getRuleChars().getGreaterChar();
  }

  /** Sets the character that is used to compare if a value is greater than another */
  public void setGreaterChar(char newValue)
  {
    getRuleChars().setGreaterChar(newValue);
  }

  /** Returns the character for less than comparisons ( DEFAULT = '<' ) */
  public char getLessChar()
  {
    return getRuleChars().getLessChar();
  }

  /** Sets the character that is used to compare if a value is less than another */
  public void setLessChar(char newValue)
  {
    getRuleChars().setLessChar(newValue);
  }

  /** Returns the character for equals comparisons ( DEFAULT = '=' ) */
  public char getEqualsChar()
  {
    return getRuleChars().getEqualsChar();
  }

  /** Sets the character that is used to compare if two values are equal  */
  public void setEqualsChar(char newValue)
  {
    getRuleChars().setEqualsChar(newValue);
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================

  /**
   * Parse the given rule string to a MatchGroup which can be used to
   * create a MatchRule.
   * 
   * @param rule The rule in a string compliant to the MatchRule syntax
   * @throws MatchRuleParseException Each syntax error in the given rule causes
   * 																	this exception with a short description
   * 																	of what is wrong
   */
  protected MatchGroup parseToGroup(String rule) throws MatchRuleParseException
  {
    StringBuffer buffer;
    MatchGroup group;

    if (rule == null)
      return null;

    buffer = new StringBuffer(rule.length() + 2);
    buffer.append(getGroupStartChar());
    buffer.append(rule);
    buffer.append(getGroupEndChar());

    scanner(new StringScanner(buffer.toString()));
    group = parseGroup();
    checkExpectedEnd(scanner().peek());
    //group.optimize() ;
    return group;
  }

  // ******** GROUP PARSING **************************************************

  protected MatchGroup parseGroup() throws MatchRuleParseException
  {
    MatchGroup group;
    char ch = ' ';
    boolean enclosed = false;

    group = new MatchGroup();
    readOperators(group);
    ch = scanner().nextNoneWhitespaceChar();
    if (ch == getGroupStartChar())
      enclosed = true;
    else
      scanner().skip(-1);

    readElements(group);
    ch = scanner().peek();
    if (enclosed)
    {
      checkUnexpectedEnd(ch);
      if (isGroupEnd(ch))
      {
        scanner().nextChar();
      }
      else
      {
        scanner().skip(-1);
        throwException("Expected '" + getGroupEndChar() + "', but found '" + ch + "' at position " + scanner().getPosition());
      }
    }
    else
    {
      checkExpectedEnd(ch);
    }
    return group;
  }

  protected void readElements(MatchGroup group) throws MatchRuleParseException
  {
    char ch = ' ';

    ch = scanner().peek();
    checkUnexpectedEnd(ch);
    do
    {
      group.addElement(readElement());
      ch = scanner().nextNoneWhitespaceChar();
      if (!atEnd(ch))
        scanner().skip(-1);
    }
    while (isOperator(ch));

  }

  protected MatchElement readElement() throws MatchRuleParseException
  {
    MatchElement element = null;

    if (nextIsGroupElement())
    {
      element = parseGroup();
    }
    else
    {
      element = parseAttribute();
    }
    return element;
  }

  protected boolean nextIsGroupElement()
  {
    char ch = ' ';
    boolean isGroup = false;

    scanner().markPosition();

    ch = scanner().nextNoneWhitespaceChar();
    while ((isOperator(ch)) || (ch == getNotChar()))
    {
      ch = scanner().nextNoneWhitespaceChar();
    }

    if (ch == getGroupStartChar())
    {
      isGroup = true;
    }
    scanner().restorePosition();
    return isGroup;
  }

  // ******** ATTRIBUTE PARSING **********************************************

  protected MatchAttribute parseAttribute() throws MatchRuleParseException
  {
    MatchAttribute attr;
    char ch;

    if (scanner().length() < 3) // at least "x=*"
    {
      throwException("Impossible length for attribute at " + "position : " + scanner().getPosition());
    }

    attr = new MatchAttribute();
    readOperators(attr);
    readAttributeName(attr);
    ch = readCompareOperator(attr);
    if (ch == getValueStartChar())
    {
      readMatchValues(attr);
    }
    else
    {
      readMatchValue(attr);
    }
    return attr;
  }

  protected void readAttributeName(MatchAttribute attribute) throws MatchRuleParseException
  {
    StringBuffer strbuf = new StringBuffer(40);
    char ch = scanner().nextNoneWhitespaceChar();

    while (isValidNameCharacter(ch))
    {
      strbuf.append(ch);
      ch = scanner().nextChar();
    }

    if (ch == StringUtil.CH_SPACE)
    {
      ch = scanner().nextNoneWhitespaceChar();
    }
    checkUnexpectedEnd(ch);

    if (isValidCompareOperator(ch))
    {
      scanner().skip(-1); // To get back to start of operator 
    }
    else
    {
      throwException("Invalid character '" + ch + "' in " + "attribute \"" + scanner().toString() + "\"");
    }

    attribute.setAttributeName(strbuf.toString());
  }

  protected boolean isValidCompareOperator(char ch)
  {
    if (ch == getValueStartChar())
      return true;

    if (ch == getEqualsChar())
      return true;

    if (ch == getGreaterChar())
      return true;

    if (ch == getLessChar())
      return true;

    return false;
  }

  protected char readCompareOperator(MatchAttribute attribute) throws MatchRuleParseException
  {
    char ch;

    ch = scanner().nextNoneWhitespaceChar();
    checkUnexpectedEnd(ch);

    if ((ch == getValueStartChar()) || (ch == getEqualsChar()))
    {
      attribute.setEqualsOperator();
      return ch;
    }

    if (ch == getGreaterChar())
    {
      if (scanner().peek() == getEqualsChar())
      {
        ch = scanner().nextChar();
        attribute.setGreaterOrEqualOperator();
      }
      else
      {
        attribute.setGreaterOperator();
      }
      return ch;
    }

    if (ch == getLessChar())
    {
      if (scanner().peek() == getEqualsChar())
      {
        ch = scanner().nextChar();
        attribute.setLessOrEqualOperator();
      }
      else
      {
        attribute.setLessOperator();
      }
      return ch;
    }

    throwException("Invalid compare operator '" + ch + "' after " + "attribute \"" + scanner().toString() + "\"");
    return ch;
  }

  protected void readMatchValues(MatchAttribute attribute) throws MatchRuleParseException
  {
    List<StringPattern> patterns = new ArrayList<StringPattern>();
    StringPattern[] strPatterns = null;
    char ch;
    StringPattern pattern;

    do
    {
      pattern = readMatchValue();
      patterns.add(pattern);

      ch = scanner().nextNoneWhitespaceChar();
      checkUnexpectedEnd(ch);

      if ((ch != getValueSeparatorChar()) && (ch != getValueEndChar()))
      {
        scanner().skip(-1);
        throwException("Expected '" + getValueSeparatorChar() + "' or '" + getValueEndChar() + "' at position " + scanner().getPosition() + " but found '" + ch + "'");
      }
    }
    while (ch != getValueEndChar());

    strPatterns = new StringPattern[patterns.size()];
    patterns.toArray(strPatterns);
    attribute.setPatterns(strPatterns);
  }

  protected void readMatchValue(MatchAttribute attribute) throws MatchRuleParseException
  {
    StringPattern pattern;

    pattern = readMatchValue();
    attribute.setPattern(pattern);
  }

  protected StringPattern readMatchValue() throws MatchRuleParseException
  {
    String value;
    char ch;

    ch = scanner().nextNoneWhitespaceChar();
    checkUnexpectedEnd(ch);

    if (ch == getValueDelimiterChar())
    {
      value = readDelimitedMatchValue();
    }
    else
    {
      value = readUndelimitedMatchValue(ch);
    }
    return new StringPattern(value);
  }

  protected String readDelimitedMatchValue() throws MatchRuleParseException
  {
    StringBuffer strbuf;
    char delimiter;
    char ch;

    delimiter = getValueDelimiterChar();
    strbuf = new StringBuffer(40);

    ch = scanner().nextChar();
    checkUnexpectedEnd(ch);

    while (ch != delimiter)
    {
      strbuf.append(ch);
      ch = scanner().nextChar();
      checkUnexpectedEnd(ch);
    }
    return strbuf.toString();
  }

  protected String readUndelimitedMatchValue(char character)
  {
    StringBuffer strbuf;
    char ch = character;

    strbuf = new StringBuffer(40);
    while (isPartOfValue(ch))
    {
      strbuf.append(ch);
      ch = scanner().nextChar();
    }

    if (!atEnd(ch))
    {
      scanner().skip(-1);
    }

    return strbuf.toString().trim();
  }

  /**
   * Returns true if the given character can be part of a value.
   * Returns false if the character is a special character that terminates 
   * a value.
   */
  protected boolean isPartOfValue(char ch)
  {
    if (ch == getValueEndChar())
      return false;

    if (ch == getValueSeparatorChar())
      return false;

    return !(isOperator(ch) || isGroupEnd(ch) || atEnd(ch));
  }

  // ******** COMMON METHODS *************************************************

  /**
   * Return two operator flags depending on the next characters in the 
   * scanner.
   * 
   * The first value defines if the operator is an AND (true) or an OR (false).
   * The second value defines if NOT is set (true) or not (false).
   */
  protected void readOperators(MatchElement element)
  {
    char ch;

    ch = scanner().nextNoneWhitespaceChar();
    if ((ch == getAndChar()) || (ch == getOrChar()))
    {
      element.setAnd(ch == getAndChar());
      if (scanner().nextNoneWhitespaceChar() == getNotChar())
      {
        element.setNot(true);
      }
      else
      {
        scanner().skip(-1);
      }
    }
    else
    {
      if (ch == getNotChar())
      {
        element.setNot(true);
      }
      else
      {
        scanner().skip(-1);
      }
    }
  }

  protected String readUpTo(char exitChar) throws MatchRuleParseException
  {
    StringBuffer strbuf;
    char ch;

    strbuf = new StringBuffer(100);
    ch = scanner().nextChar();
    while (ch != exitChar)
    {
      checkUnexpectedEnd(ch);
      strbuf.append(ch);
      ch = scanner().nextChar();
    }
    return strbuf.toString();
  }

  protected boolean isOperator(char ch)
  {
    return ((ch == getAndChar()) || (ch == getOrChar()));
  }

  protected boolean isValidNameCharacter(char ch)
  {
    boolean valid;

    valid = Character.isLetterOrDigit(ch);
    if (!valid)
    {
      valid = getSpecialNameCharacters().indexOf(ch) >= 0;
    }
    return valid;
  }

  protected boolean isGroupEnd(char ch)
  {
    return (ch == getGroupEndChar());
  }

  /** 
   * Returns the special character allowed in attribute names 
   */
  protected String getSpecialNameCharacters()
  {
    return getRuleChars().getSpecialNameCharacters();
  }

  protected MatchRule createMatchRuleOn(MatchGroup group)
  {
    MatchRule matchRule;

    matchRule = new MatchRule(group);

    if (getIgnoreCaseInNames())
    {
      matchRule.ignoreCaseInNames(true);
    }

    if (getIgnoreCaseInValues())
    {
      matchRule.ignoreCase(true);
    }

    if (getMultiCharWildcardMatchesEmptyString())
    {
      matchRule.multiCharWildcardMatchesEmptyString(true);
    }

    return matchRule;
  }
  
  protected MatchRuleChars getRuleChars()
  {
    return ruleChars;
  }

  protected void setRuleChars(MatchRuleChars newValue)
  {
    ruleChars = newValue;
  }
}
