// ===========================================================================
// CONTENT  : CLASS ByteSizeValue
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.0 - 07/09/2014
// HISTORY  :
//  07/09/2014  mdu  CREATED
//
// Copyright (c) 2014, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

// ===========================================================================
// IMPORTS
// ===========================================================================
import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * Represents a number-of-bytes value and provides parsing of strings and 
 * various different getters for various byte size units.
 *
 * @author Manfred Duchrow
 * @version 1.0
 */
public class ByteSizeValue
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  /**
   * The factor 1024.
   */
  public static final BigDecimal KB_FACTOR = BigDecimal.valueOf(1024L);

  private final static char DECIMAL_SEPARATOR = '.';

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private BigDecimal value;
  private ByteSizeUnit currentUnit;

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Initialize the new instance with a value in bytes.
   */
  public ByteSizeValue(BigDecimal numBytes)
  {
    this(numBytes, ByteSizeUnit.BYTES);
  }

  /**
   * Initialize the new instance with a value in bytes.
   */
  public ByteSizeValue(long numBytes)
  {
    this(BigDecimal.valueOf(numBytes));
  }

  /**
   * Initialize the new instance with a value and unit.
   */
  public ByteSizeValue(BigDecimal value, ByteSizeUnit unit)
  {
    super();
    if (value == null)
    {
      throw new IllegalArgumentException("The provided number of bytes must not be null!");
    }
    if (unit == null)
    {
      throw new IllegalArgumentException("The provided unit must not be null!");
    }
    this.setValue(value);
    this.setCurrentUnit(unit);
  }

  /**
   * Initialize the new instance with a value and unit.
   */
  public ByteSizeValue(long value, ByteSizeUnit unit)
  {
    this(BigDecimal.valueOf(value), unit);
  }

  /**
   * Initialize the new instance with a string that must contain digits,
   * an optional decimal point and optionally followed by the prefix name of 
   * a unit ({@link ByteSizeUnit}).
   * <br>
   * If no explicit unit is set in the string then the given default unit will be used.
   * @param strValue The value with the optional unit definition (must not be null).
   * @param defaultUnit The unit to be used if no explicit unit is defined in the string (must not be null).
   * @throws NumberFormatException If the string does not contain a valid value or unit.
   */
  public ByteSizeValue(String strValue, ByteSizeUnit defaultUnit)
  {
    this();
    this.parseAndInit(strValue, defaultUnit);
  }

  /**
   * Initialize the new instance with a string that must contain digits,
   * an optional decimal point and
   * optionally followed by the prefix name of a unit ({@link ByteSizeUnit}).
   * <br>
   * The default unit is "B" ({@link ByteSizeUnit#BYTES}) if no other unit is
   * explicitly set in the string.
   * @param strValue The value with the optional unit definition (must not be null).
   * @throws NumberFormatException If the string does not contain a valid value or unit.
   */
  public ByteSizeValue(String strValue)
  {
    this(strValue, ByteSizeUnit.BYTES);
  }

  /**
   * Creates a new instance with the value: 0 bytes.
   */
  protected ByteSizeValue()
  {
    this(0L);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns the value in BYTES.
   */
  public long getSizeInBytes()
  {
    return this.asBytes().longValue();
  }

  /**
   * Returns the value in BYTES.
   */
  public BigDecimal asBytes()
  {
    return this.convertTo(ByteSizeUnit.BYTES);
  }

  /**
   * Returns the value in KILO_BYTES.
   */
  public BigDecimal asKiloBytes()
  {
    return this.convertTo(ByteSizeUnit.KILO_BYTES);
  }

  /**
   * Returns the value in MEGA_BYTES.
   */
  public BigDecimal asMegaBytes()
  {
    return this.convertTo(ByteSizeUnit.MEGA_BYTES);
  }

  /**
   * Returns the value in GIGA_BYTES.
   */
  public BigDecimal asGigaBytes()
  {
    return this.convertTo(ByteSizeUnit.GIGA_BYTES);
  }

  /**
   * Returns the value in TERA_BYTES.
   */
  public BigDecimal asTeraBytes()
  {
    return this.convertTo(ByteSizeUnit.TERA_BYTES);
  }

  /**
   * Returns the value in PETA_BYTES.
   */
  public BigDecimal asPetaBytes()
  {
    return this.convertTo(ByteSizeUnit.PETA_BYTES);
  }

  /**
   * Returns the value in EXA_BYTES.
   */
  public BigDecimal asExaBytes()
  {
    return this.convertTo(ByteSizeUnit.EXA_BYTES);
  }

  /**
   * Returns the value converted to the specified unit.
   * @param unit The unit of the result value (must not be null).
   */
  public BigDecimal convertTo(ByteSizeUnit unit)
  {
    BigDecimal newValue;
    int powerDiff;
    int precision = 0;
    int len;

    newValue = this.getValue();
    powerDiff = this.getCurrentUnit().getPowerOf1024() - unit.getPowerOf1024();
    if (powerDiff < 0)
    {
      powerDiff = powerDiff * (-1);
      precision = powerDiff;
      for (int i = 1; i <= powerDiff; i++)
      {
        len = newValue.setScale(0, RoundingMode.HALF_UP).toPlainString().length();
        newValue = newValue.divide(KB_FACTOR);
        if (len < 5)
        {
          precision = precision + len;
        }
      }
    }
    else if (powerDiff > 0)
    {
      for (int i = 1; i <= powerDiff; i++)
      {
        newValue = newValue.multiply(KB_FACTOR);
      }
      precision = 0;
    }
    newValue = newValue.setScale(precision, RoundingMode.HALF_UP);
    newValue = newValue.stripTrailingZeros();
    newValue = new BigDecimal(newValue.toPlainString());
    return newValue;
  }

  @Override
  public String toString()
  {
    return this.getValue().toPlainString() + " " + this.getCurrentUnit().getIEEE1541Prefix();
  }

  @Override
  public boolean equals(Object obj)
  {
    ByteSizeValue other;

    if (obj instanceof ByteSizeValue)
    {
      other = (ByteSizeValue)obj;
      return this.asBytes().equals(other.asBytes());
    }
    return false;
  }

  @Override
  public int hashCode()
  {
    return this.asBytes().hashCode();
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  /**
   * Parse the given string, taking any optional unit string at the end into account.
   * Allowed unit strings are the short names or IEEE-1541 prefixes 
   * defined in ByteSizeUnit (e.g. "B", "KB", "MB", "KiB", "GiB").
   * 
   * @param strValue The string to parse (may contain decimal point - except for unit "bytes").
   * @param defaultUnit The unit to use if the string does not contain any explicit short name.
   * @throws NumberFormatException If the string does not contain a valid decimal value or an invalid unit.
   */
  protected void parseAndInit(final String strValue, final ByteSizeUnit defaultUnit)
  {
    ByteSizeUnit unit = null;
    StringPair stringPair;

    stringPair = this.splitDigitsAndUnit(strValue.trim());
    unit = ByteSizeUnit.findByPrefix(stringPair.getString2());
    if (unit == null) // Needing default unit?
    {
      if (StringUtil.current().notNullOrEmpty(stringPair.getString2()))
      {
        throw new NumberFormatException("Invalid byte size unit specified: " + strValue);
      }
      unit = defaultUnit;
    }
    if ((unit == ByteSizeUnit.BYTES) && (StringUtil.current().contains(stringPair.getString1(), DECIMAL_SEPARATOR)))
    {
      throw new NumberFormatException("Byte size must not contain decimal point for unit 'Bytes': " + strValue);
    }
    this.setValue(new BigDecimal(stringPair.getString1()));
    this.setCurrentUnit(unit);
  }

  protected StringPair splitDigitsAndUnit(String str)
  {
    StringPair pair;
    int noneDigitIndex = 0;

    pair = new StringPair();
    for (char ch : str.toCharArray())
    {
      if ((!Character.isDigit(ch)) && (ch != DECIMAL_SEPARATOR))
      {
        break;
      }
      noneDigitIndex++;
    }
    if (noneDigitIndex >= str.length())
    {
      pair.setString1(str);
      pair.setString2("");
    }
    else
    {
      pair.setString1(str.substring(0, noneDigitIndex));
      pair.setString2(str.substring(noneDigitIndex).trim());
    }
    return pair;
  }

  protected BigDecimal getValue()
  {
    return this.value;
  }

  protected void setValue(BigDecimal value)
  {
    this.value = value;
  }

  protected ByteSizeUnit getCurrentUnit()
  {
    return this.currentUnit;
  }

  protected void setCurrentUnit(ByteSizeUnit currentUnit)
  {
    this.currentUnit = currentUnit;
  }
}
