// ===========================================================================
// CONTENT  : ENUM ByteSizeUnit
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.0 - 07/09/2014
// HISTORY  :
//  07/09/2014  mdu  CREATED
//
// Copyright (c) 2014, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text;

// =========================================================================
// IMPORTS
// =========================================================================
import java.math.BigDecimal;

/**
 * This enum provides several memory size units with an associated factor that is 
 * useful to calculate the number of bytes from a value given in a different
 * memory size unit.
 * See also http://en.wikipedia.org/wiki/IEEE_1541-2002
 *
 * @author Manfred Duchrow
 * @version 1.0
 */
public enum ByteSizeUnit
{
  // =========================================================================
  // ENUM VALUES
  // =========================================================================
  BYTES("bytes", "B", "byte", "B", 0), 
  KILO_BYTES("kilobytes", "KB", "kibibyte", "KiB", 1), 
  MEGA_BYTES("megabytes", "MB", "mebibyte", "MiB", 2), 
  GIGA_BYTES("gigabytes", "GB", "gibibyte", "GiB", 3), 
  TERA_BYTES("terabytes", "TB", "tebibyte", "TiB", 4), 
  PETA_BYTES("petabytes", "PB", "pebibyte", "PiB", 5), 
  EXA_BYTES("exabytes", "EB", "exbibyte", "EiB", 6);

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private final int powerOf1024;
  private final String commonName;
  private final String commonPrefix;
  private final String IEEE_1541_Name;
  private final String IEEE_1541_Prefix;
  private final BigDecimal bytesFactor;

  // =========================================================================
  // STATIC METHODS 
  // =========================================================================
  /**
   * Returns the ByteSiteUnit enum value that matches the given string.
   * 
   * @param str The prefix value to be used for the look-up.
   * @return The found enum or null if not found.
   */
  public static ByteSizeUnit findByPrefix(final String str)
  {
    if (str != null)
    {
      for (ByteSizeUnit unit : ByteSizeUnit.values())
      {
        if (unit.getCommonPrefix().equals(str))
        {
          return unit;
        }
        if (unit.getIEEE1541Prefix().equals(str))
        {
          return unit;
        }
      }
    }
    return null;
  }

  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  private ByteSizeUnit(String commonName, String commonPrefix, String IEEE_1541_Name, String IEEE_1541_Prefix, int powerOf1024)
  {
    this.powerOf1024 = powerOf1024;
    this.commonName = commonName;
    this.commonPrefix = commonPrefix;
    this.IEEE_1541_Name = IEEE_1541_Name;
    this.IEEE_1541_Prefix = IEEE_1541_Prefix;
    this.bytesFactor = this.power(1024L, powerOf1024);
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Returns the power of 1024 (bytes) this unit represents.
   */
  public int getPowerOf1024()
  {
    return powerOf1024;
  }

  /**
   * Returns the commonly used (but incorrect) name of the unit. 
   */
  public String getCommonnName()
  {
    return this.commonName;
  }

  /**
   * Returns the commonly used (but incorrect) short name of the unit. 
   */
  public String getCommonPrefix()
  {
    return this.commonPrefix;
  }

  /**
   * Returns the standardized IEEE 1541 name of the unit. 
   */
  public String getIEEE1541Name()
  {
    return this.IEEE_1541_Name;
  }

  /**
   * Returns the standardized IEEE 1541 prefix of the unit. 
   */
  public String getIEEE1541Prefix()
  {
    return this.IEEE_1541_Prefix;
  }

  /**
   * Returns the multiplication factor for this unit to 
   * convert a corresponding value to its number of bytes. 
   */
  public BigDecimal getBytesFactor()
  {
    return this.bytesFactor;
  }

  /**
   * Returns the multiplication factor for this unit to 
   * convert a corresponding value to its number of bytes
   * as long value. 
   */
  public long getFactor()
  {
    return this.getBytesFactor().longValue();
  }

  private BigDecimal power(long value, int power)
  {
    long result = 1L;
    for (int i = 1; i <= power; i++)
    {
      result = result * value;
    }
    return BigDecimal.valueOf(result);
  }

}