/*
 * This file is part of dependency-check-core.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright (c) 2022 Jeremy Long. All Rights Reserved.
 */
package org.owasp.dependencycheck.data.update;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.sql.SQLException;
import org.owasp.dependencycheck.Engine;
import org.owasp.dependencycheck.data.knownexploited.json.KnownExploitedVulnerabilitiesSchema;
import org.owasp.dependencycheck.data.nvdcve.CveDB;
import org.owasp.dependencycheck.data.nvdcve.DatabaseException;
import org.owasp.dependencycheck.data.nvdcve.DatabaseProperties;
import org.owasp.dependencycheck.data.update.cisa.KnownExploitedVulnerabilityParser;
import org.owasp.dependencycheck.data.update.exception.CorruptedDatastreamException;
import org.owasp.dependencycheck.data.update.exception.UpdateException;
import org.owasp.dependencycheck.utils.DateUtil;
import org.owasp.dependencycheck.utils.HttpResourceConnection;
import org.owasp.dependencycheck.utils.ResourceNotFoundException;
import org.owasp.dependencycheck.utils.Settings;
import org.owasp.dependencycheck.utils.TooManyRequestsException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author jeremy
 */
public class KnownExploitedDataSource implements CachedWebDataSource {

    /**
     * The logger.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(KnownExploitedDataSource.class);
    /**
     * The default URL for the CISA Known Exploited Vulnerability Catalog.
     */
    private static final String DEFAULT_URL = "https://www.cisa.gov/sites/default/files/feeds/known_exploited_vulnerabilities.json";
    /**
     * A reference to the CVE DB.
     */
    private CveDB cveDB;
    /**
     * A reference to the ODC settings.
     */
    private Settings settings;
    /**
     * The properties obtained from the database.
     */
    private DatabaseProperties dbProperties = null;

    @Override
    public boolean update(Engine engine) throws UpdateException {
        this.cveDB = engine.getDatabase();
        this.settings = engine.getSettings();
        dbProperties = cveDB.getDatabaseProperties();
        final boolean autoUpdate = settings.getBoolean(Settings.KEYS.AUTO_UPDATE, true);
        final boolean kevEnabled = settings.getBoolean(Settings.KEYS.ANALYZER_KNOWN_EXPLOITED_ENABLED, true);
        if (autoUpdate && kevEnabled && shouldUpdate()) {
            try {
                final URL url = new URL(settings.getString(Settings.KEYS.KEV_URL, DEFAULT_URL));
                LOGGER.info("Updating CISA Known Exploited Vulnerability list: " + url.toString());
                //TODO - add all the proxy config, likely use the same as configured for NVD
                final HttpResourceConnection conn = new HttpResourceConnection(settings);
                try (InputStream in = conn.fetch(url)) {
                    final KnownExploitedVulnerabilityParser parser = new KnownExploitedVulnerabilityParser();
                    final KnownExploitedVulnerabilitiesSchema data = parser.parse(in);
                    final String currentVersion = dbProperties.getProperty(DatabaseProperties.KEV_VERSION, "");
                    if (!currentVersion.equals(data.getCatalogVersion())) {
                        cveDB.updateKnownExploitedVulnerabilities(data.getVulnerabilities());
                    }
                    //all dates in the db are now stored in seconds as opposed to previously milliseconds.
                    dbProperties.save(DatabaseProperties.KEV_LAST_CHECKED, Long.toString(System.currentTimeMillis() / 1000));
                    return true;
                }
            } catch (TooManyRequestsException | ResourceNotFoundException | IOException
                    | CorruptedDatastreamException | DatabaseException | SQLException ex) {
                throw new UpdateException(ex);
            }
        }
        return false;
    }

    @Override
    public boolean purge(Engine engine) {
        //do nothing - covered by the NvdApiDataSource.
        return true;
    }

    /**
     * Checks if the Known Exploited Vulnerabilities (KEV) were last checked
     * recently. As an optimization, we can avoid repetitive checks against the
     * KEV. Setting KEV_CHECK_VALID_FOR_HOURS determines the duration since last
     * check before checking again. A database property stores the timestamp of
     * the last check.
     *
     * @return true to proceed with the check, or false to skip
     * @throws UpdateException thrown when there is an issue checking for
     * updates
     */
    private boolean shouldUpdate() throws UpdateException {
        boolean proceed = true;
        // If the valid setting has not been specified, then we proceed to check...
        final int validForHours = settings.getInt(Settings.KEYS.KEV_CHECK_VALID_FOR_HOURS, 24);
        if (cveDB.dataExists() && 0 < validForHours) {
            // ms Valid = valid (hours) x 60 min/hour x 60 sec/min x 1000 ms/sec
            final long validForSeconds = validForHours * 60L * 60L;
            final long lastChecked = dbProperties.getPropertyInSeconds(DatabaseProperties.KEV_LAST_CHECKED);
            final long now = System.currentTimeMillis() / 1000;
            proceed = (now - lastChecked) > validForSeconds;
            if (!proceed) {
                LOGGER.info("Skipping Known Exploited Vulnerabilities update check since last check was within {} hours.", validForHours);
            }
        }
        return proceed;
    }

}
