/*
 * This file is part of dependency-check-core.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright (c) 2022 Jeremy Long. All Rights Reserved.
 */
package org.owasp.dependencycheck.data.update.cisa;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.module.afterburner.AfterburnerModule;
import com.fasterxml.jackson.module.blackbird.BlackbirdModule;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import static java.nio.charset.StandardCharsets.UTF_8;
import java.util.zip.ZipException;
import org.owasp.dependencycheck.data.knownexploited.json.KnownExploitedVulnerabilitiesSchema;
import org.owasp.dependencycheck.data.update.exception.CorruptedDatastreamException;
import org.owasp.dependencycheck.data.update.exception.UpdateException;
import org.owasp.dependencycheck.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Jeremy Long
 */
public class KnownExploitedVulnerabilityParser {

    /**
     * The logger.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(KnownExploitedVulnerabilityParser.class);

    /**
     * Parses the CISA Known Exploited JSON file and inserts/updates data into
     * the database.
     *
     * @param in the CISA Known Exploited JSON input stream to parse
     * @return the Known Exploited Vulnerabilities object
     * @throws UpdateException thrown if the file could not be read
     * @throws CorruptedDatastreamException thrown if the file was found to be a
     * corrupted download (ZipException or premature EOF)
     */
    public KnownExploitedVulnerabilitiesSchema parse(InputStream in) throws UpdateException, CorruptedDatastreamException {

        final ObjectMapper objectMapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        final Module module;
        if (Utils.getJavaVersion() <= 8) {
            module = new AfterburnerModule();
        } else {
            module = new BlackbirdModule();
        }
        objectMapper.registerModule(module);

        final ObjectReader objectReader = objectMapper.readerFor(KnownExploitedVulnerabilitiesSchema.class);

        //InputStream in = new GZIPInputStream(fin);
        try (InputStreamReader isr = new InputStreamReader(in, UTF_8);
                JsonParser parser = objectReader.getFactory().createParser(isr)) {
            final KnownExploitedVulnerabilitiesSchema data = objectReader.readValue(parser);
            return data;
        } catch (ZipException | EOFException ex) {
            throw new CorruptedDatastreamException("Error parsing CISA Known Exploited Vulnerabilities file", ex);
        } catch (IOException ex) {
            LOGGER.error("Error reading CISA Known Exploited Vulnerabilities JSON data");
            LOGGER.debug("Error extracting the CISA Known Exploited Vulnerabilities JSON data", ex);
            throw new UpdateException("Unable to find the CISA Known Exploited Vulnerabilities file to parse", ex);
        }
    }
}
