/*
 * This file is part of dependency-check-core.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright (c) 2012 Jeremy Long. All Rights Reserved.
 */
package org.owasp.dependencycheck.dependency;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.commons.lang3.builder.CompareToBuilder;

/**
 * Contains the information about a vulnerability.
 *
 * @author Jeremy Long
 */
@NotThreadSafe
public class Vulnerability implements Serializable, Comparable<Vulnerability> {

    /**
     * An enumeration for the source of vulnerability.
     */
    public enum Source {
        /**
         * National Vulnerability Database.
         */
        NVD,
        /**
         * NPM Public Advisory.
         */
        NPM,
        /**
         * RetireJS.
         */
        RETIREJS
    }

    /**
     * The serial version uid.
     */
    private static final long serialVersionUID = 307319490326651052L;

    /**
     * The name of the vulnerability.
     */
    private String name;
    /**
     * the description of the vulnerability.
     */
    private String description;
    /**
     * References for this vulnerability.
     */
    private Set<Reference> references = new HashSet<>();
    /**
     * A set of vulnerable software.
     */
    private Set<VulnerableSoftware> vulnerableSoftware = new HashSet<>();
    /**
     * The CWE(s) for the vulnerability.
     */
    private CweSet cwes = new CweSet();
    /**
     * The severity a {@link Source} has assigned for which a CVSS score is not
     * available. Severity could be anything ranging from 'critical', 'high',
     * 'medium', and 'low', to non-traditional labels like 'major', 'minor', and
     * 'important'.
     */
    private String unscoredSeverity;
    /**
     * The CVSS V2 scoring information.
     */
    private CvssV2 cvssV2;
    /**
     * The CVSS V3 scoring information.
     */
    private CvssV3 cvssV3;
    /**
     * The Vulnerable Software that caused this vulnerability to be flagged.
     */
    private VulnerableSoftware matchedVulnerableSoftware;
    /**
     * Notes about the vulnerability. Generally used for suppression
     * information.
     */
    private String notes;

    /**
     * The source that identified the vulnerability.
     */
    private Source source = Source.NVD;

    /**
     * Default constructor.
     */
    public Vulnerability() {
        //empty
    }

    /**
     * Constructs a new Vulnerability by its name.
     *
     * @param name the name of the vulnerability
     */
    public Vulnerability(String name) {
        this.name = name;
    }

    /**
     * Get the value of name.
     *
     * @return the value of name
     */
    public String getName() {
        return name;
    }

    /**
     * Set the value of name.
     *
     * @param name new value of name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Get the value of description.
     *
     * @return the value of description
     */
    public String getDescription() {
        return description;
    }

    /**
     * Set the value of description.
     *
     * @param description new value of description
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Get the value of references.
     *
     * @return the value of references
     */
    public Set<Reference> getReferences() {
        return references;
    }

    /**
     * Returns the list of references. This is primarily used within the
     * generated reports.
     *
     * @param sorted whether the returned list should be sorted
     * @return the list of references
     */
    public List<Reference> getReferences(boolean sorted) {
        final List<Reference> sortedRefs = new ArrayList<>(this.references);
        if (sorted) {
            Collections.sort(sortedRefs);
        }
        return sortedRefs;
    }

    /**
     * Set the value of references.
     *
     * @param references new value of references
     */
    public void setReferences(Set<Reference> references) {
        this.references = references;
    }

    /**
     * Adds a reference to the references collection.
     *
     * @param ref a reference for the vulnerability
     */
    public void addReference(Reference ref) {
        this.references.add(ref);
    }

    /**
     * Adds a reference.
     *
     * @param referenceSource the source of the reference
     * @param referenceName the referenceName of the reference
     * @param referenceUrl the url of the reference
     */
    public void addReference(String referenceSource, String referenceName, String referenceUrl) {
        final Reference ref = new Reference();
        ref.setSource(referenceSource);
        ref.setName(referenceName);
        ref.setUrl(referenceUrl);
        this.references.add(ref);
    }

    /**
     * Get the value of vulnerableSoftware.
     *
     * @return the value of vulnerableSoftware
     */
    public Set<VulnerableSoftware> getVulnerableSoftware() {
        return vulnerableSoftware;
    }

    /**
     * Returns a sorted list of vulnerable software. This is primarily used for
     * display within reports.
     *
     * @param sorted whether or not the list should be sorted
     * @return the list of vulnerable software
     */
    public List<VulnerableSoftware> getVulnerableSoftware(boolean sorted) {
        final List<VulnerableSoftware> sortedVulnerableSoftware = new ArrayList<>(this.vulnerableSoftware);
        if (sorted) {
            Collections.sort(sortedVulnerableSoftware);
        }
        return sortedVulnerableSoftware;
    }

    /**
     * Set the value of vulnerableSoftware.
     *
     * @param vulnerableSoftware new value of vulnerableSoftware
     */
    public void setVulnerableSoftware(Set<VulnerableSoftware> vulnerableSoftware) {
        this.vulnerableSoftware = vulnerableSoftware;
    }

    /**
     * Adds an entry for vulnerable software.
     *
     * @param software the vulnerable software reference to add
     */
    public void addVulnerableSoftware(VulnerableSoftware software) {
        vulnerableSoftware.add(software);
    }

    /**
     * Get the CVSS V2 scoring information.
     *
     * @return the CVSS V2 scoring information
     */
    public CvssV2 getCvssV2() {
        return cvssV2;
    }

    /**
     * Sets the CVSS V2 scoring information.
     *
     * @param cvssV2 the CVSS V2 scoring information
     */
    public void setCvssV2(CvssV2 cvssV2) {
        this.cvssV2 = cvssV2;
    }

    /**
     * Get the CVSS V3 scoring information.
     *
     * @return the CVSS V3 scoring information
     */
    public CvssV3 getCvssV3() {
        return cvssV3;
    }

    /**
     * Sets the CVSS V3 scoring information.
     *
     * @param cvssV3 the CVSS V3 scoring information
     */
    public void setCvssV3(CvssV3 cvssV3) {
        this.cvssV3 = cvssV3;
    }

    /**
     * Get the set of CWEs.
     *
     * @return the set of CWEs
     */
    public CweSet getCwes() {
        return cwes;
    }

    /**
     * Adds a CWE to the set.
     *
     * @param cwe new CWE to add
     */
    public void addCwe(String cwe) {
        this.cwes.addCwe(cwe);
    }

    /**
     * Retrieves the severity a {@link Source} has assigned for which a CVSS
     * score is not available. Severity could be anything ranging from
     * 'critical', 'high', 'medium', and 'low', to non-traditional labels like
     * 'major', 'minor', and 'important'.
     *
     * @return the un-scored severity
     */
    public String getUnscoredSeverity() {
        return unscoredSeverity;
    }

    /**
     * Sets the severity a {@link Source} has assigned for which a CVSS score is
     * not available. Severity could be anything ranging from 'critical',
     * 'high', 'medium', and 'low', to non-traditional labels like 'major',
     * 'minor', and 'important'.
     *
     * @param unscoredSeverity the un-scored severity
     */
    public void setUnscoredSeverity(String unscoredSeverity) {
        this.unscoredSeverity = unscoredSeverity;
    }

    /**
     * Get the value of notes from suppression notes.
     *
     * @return the value of notes
     */
    public String getNotes() {
        return notes;
    }

    /**
     * Set the value of notes.
     *
     * @param notes new value of cwes
     */
    public void setNotes(String notes) {
        this.notes = notes;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Vulnerability)) {
            return false;
        }
        final Vulnerability other = (Vulnerability) obj;
        return !((this.name == null) ? (other.name != null) : !this.name.equals(other.name));
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 41 * hash + (this.name != null ? this.name.hashCode() : 0);
        return hash;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("Vulnerability ");
        sb.append(this.name);
        sb.append("\nReferences:\n");
        for (Reference reference : getReferences(true)) {
            sb.append("=> ");
            sb.append(reference);
            sb.append("\n");
        }
        sb.append("\nSoftware:\n");

        for (VulnerableSoftware software : getVulnerableSoftware(true)) {
            sb.append("=> ");
            sb.append(software);
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * Compares two vulnerabilities.
     *
     * @param v a vulnerability to be compared
     * @return a negative integer, zero, or a positive integer as this object is
     * less than, equal to, or greater than the specified vulnerability
     */
    @Override
    public int compareTo(Vulnerability v) {
        return new CompareToBuilder()
                .append(this.name, v.name)
                .toComparison();
    }

    /**
     * Sets the CPE that caused this vulnerability to be flagged.
     *
     * @param software a Vulnerable Software identifier
     */
    public void setMatchedVulnerableSoftware(VulnerableSoftware software) {
        matchedVulnerableSoftware = software;
    }

    /**
     * Get the value of matchedVulnerableSoftware.
     *
     * @return the value of matchedVulnerableSoftware
     */
    public VulnerableSoftware getMatchedVulnerableSoftware() {
        return matchedVulnerableSoftware;
    }

    /**
     * Returns the source that identified the vulnerability.
     *
     * @return the source
     */
    public Source getSource() {
        return source;
    }

    /**
     * Sets the source that identified the vulnerability.
     *
     * @param source the source
     */
    public void setSource(Source source) {
        this.source = source;
    }
}
