/*
 * Decompiled with CFR 0.152.
 */
package org.owasp.dependencycheck.analyzer;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import org.apache.commons.io.FileUtils;
import org.owasp.dependencycheck.Engine;
import org.owasp.dependencycheck.analyzer.AbstractFileTypeAnalyzer;
import org.owasp.dependencycheck.analyzer.AnalysisPhase;
import org.owasp.dependencycheck.analyzer.FileTypeAnalyzer;
import org.owasp.dependencycheck.analyzer.RubyBundlerAnalyzer;
import org.owasp.dependencycheck.analyzer.RubyGemspecAnalyzer;
import org.owasp.dependencycheck.analyzer.exception.AnalysisException;
import org.owasp.dependencycheck.data.nvdcve.CveDB;
import org.owasp.dependencycheck.data.nvdcve.DatabaseException;
import org.owasp.dependencycheck.dependency.Confidence;
import org.owasp.dependencycheck.dependency.Dependency;
import org.owasp.dependencycheck.dependency.Reference;
import org.owasp.dependencycheck.dependency.Vulnerability;
import org.owasp.dependencycheck.exception.InitializationException;
import org.owasp.dependencycheck.utils.FileFilterBuilder;
import org.owasp.dependencycheck.utils.Settings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RubyBundleAuditAnalyzer
extends AbstractFileTypeAnalyzer {
    private static final Logger LOGGER = LoggerFactory.getLogger(RubyBundleAuditAnalyzer.class);
    private static final String ANALYZER_NAME = "Ruby Bundle Audit Analyzer";
    private static final AnalysisPhase ANALYSIS_PHASE = AnalysisPhase.PRE_INFORMATION_COLLECTION;
    private static final FileFilter FILTER = FileFilterBuilder.newInstance().addFilenames("Gemfile.lock").build();
    public static final String NAME = "Name: ";
    public static final String VERSION = "Version: ";
    public static final String ADVISORY = "Advisory: ";
    public static final String CRITICALITY = "Criticality: ";
    private CveDB cvedb;
    private boolean needToDisableGemspecAnalyzer = true;

    @Override
    protected FileFilter getFileFilter() {
        return FILTER;
    }

    private Process launchBundleAudit(File folder) throws AnalysisException {
        if (!folder.isDirectory()) {
            throw new AnalysisException(String.format("%s should have been a directory.", folder.getAbsolutePath()));
        }
        ArrayList<String> args = new ArrayList<String>();
        String bundleAuditPath = Settings.getString((String)"analyzer.bundle.audit.path");
        File bundleAudit = null;
        if (bundleAuditPath != null && !(bundleAudit = new File(bundleAuditPath)).isFile()) {
            LOGGER.warn("Supplied `bundleAudit` path is incorrect: {}", (Object)bundleAuditPath);
            bundleAudit = null;
        }
        args.add(bundleAudit != null && bundleAudit.isFile() ? bundleAudit.getAbsolutePath() : "bundle-audit");
        args.add("check");
        args.add("--verbose");
        ProcessBuilder builder = new ProcessBuilder(args);
        builder.directory(folder);
        try {
            LOGGER.info("Launching: {} from {}", args, (Object)folder);
            return builder.start();
        }
        catch (IOException ioe) {
            throw new AnalysisException("bundle-audit initialization failure; this error can be ignored if you are not analyzing Ruby. Otherwise ensure that bundle-audit is installed and the path to bundle audit is correctly specified", ioe);
        }
    }

    @Override
    public void initializeFileTypeAnalyzer() throws InitializationException {
        int exitValue;
        try {
            this.cvedb = CveDB.getInstance();
        }
        catch (DatabaseException ex) {
            LOGGER.warn("Exception opening the database");
            LOGGER.debug("error", (Throwable)ex);
            this.setEnabled(false);
            throw new InitializationException("Error connecting to the database", ex);
        }
        Process process = null;
        try {
            process = this.launchBundleAudit(Settings.getTempDirectory());
        }
        catch (AnalysisException ae) {
            this.setEnabled(false);
            String msg = String.format("Exception from bundle-audit process: %s. Disabling %s", ae.getCause(), ANALYZER_NAME);
            throw new InitializationException(msg, ae);
        }
        catch (IOException ex) {
            this.setEnabled(false);
            throw new InitializationException("Unable to create temporary file, the Ruby Bundle Audit Analyzer will be disabled", ex);
        }
        try {
            exitValue = process.waitFor();
        }
        catch (InterruptedException ex) {
            this.setEnabled(false);
            String msg = String.format("Bundle-audit process was interrupted. Disabling %s", ANALYZER_NAME);
            Thread.currentThread().interrupt();
            throw new InitializationException(msg);
        }
        if (0 == exitValue) {
            this.setEnabled(false);
            String msg = String.format("Unexpected exit code from bundle-audit process. Disabling %s: %s", ANALYZER_NAME, exitValue);
            throw new InitializationException(msg);
        }
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getErrorStream(), "UTF-8"));){
            if (!reader.ready()) {
                LOGGER.warn("Bundle-audit error stream unexpectedly not ready. Disabling {}", (Object)ANALYZER_NAME);
                this.setEnabled(false);
                throw new InitializationException("Bundle-audit error stream unexpectedly not ready.");
            }
            String line = reader.readLine();
            if (line == null || !line.contains("Errno::ENOENT")) {
                LOGGER.warn("Unexpected bundle-audit output. Disabling {}: {}", (Object)ANALYZER_NAME, (Object)line);
                this.setEnabled(false);
                throw new InitializationException("Unexpected bundle-audit output.");
            }
        }
        catch (UnsupportedEncodingException ex) {
            this.setEnabled(false);
            throw new InitializationException("Unexpected bundle-audit encoding.", ex);
        }
        catch (IOException ex) {
            this.setEnabled(false);
            throw new InitializationException("Unable to read bundle-audit output.", ex);
        }
        if (this.isEnabled()) {
            LOGGER.info("{} is enabled. It is necessary to manually run \"bundle-audit update\" occasionally to keep its database up to date.", (Object)ANALYZER_NAME);
        }
    }

    @Override
    public void closeAnalyzer() {
        if (this.cvedb != null) {
            this.cvedb.close();
            this.cvedb = null;
        }
    }

    @Override
    public String getName() {
        return ANALYZER_NAME;
    }

    @Override
    public AnalysisPhase getAnalysisPhase() {
        return ANALYSIS_PHASE;
    }

    @Override
    protected String getAnalyzerEnabledSettingKey() {
        return "analyzer.bundle.audit.enabled";
    }

    @Override
    protected void analyzeDependency(Dependency dependency, Engine engine) throws AnalysisException {
        int exitValue;
        if (this.needToDisableGemspecAnalyzer) {
            boolean failed = true;
            String className = RubyGemspecAnalyzer.class.getName();
            for (FileTypeAnalyzer analyzer : engine.getFileTypeAnalyzers()) {
                if (analyzer instanceof RubyBundlerAnalyzer) {
                    ((RubyBundlerAnalyzer)analyzer).setEnabled(false);
                    LOGGER.info("Disabled {} to avoid noisy duplicate results.", (Object)RubyBundlerAnalyzer.class.getName());
                    continue;
                }
                if (!(analyzer instanceof RubyGemspecAnalyzer)) continue;
                ((RubyGemspecAnalyzer)analyzer).setEnabled(false);
                LOGGER.info("Disabled {} to avoid noisy duplicate results.", (Object)className);
                failed = false;
            }
            if (failed) {
                LOGGER.warn("Did not find {}.", (Object)className);
            }
            this.needToDisableGemspecAnalyzer = false;
        }
        File parentFile = dependency.getActualFile().getParentFile();
        Process process = this.launchBundleAudit(parentFile);
        try {
            exitValue = process.waitFor();
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            throw new AnalysisException("bundle-audit process interrupted", ie);
        }
        if (exitValue < 0 || exitValue > 1) {
            String msg = String.format("Unexpected exit code from bundle-audit process; exit code: %s", exitValue);
            throw new AnalysisException(msg);
        }
        try {
            try (BufferedReader errReader = new BufferedReader(new InputStreamReader(process.getErrorStream(), "UTF-8"));){
                while (errReader.ready()) {
                    String error = errReader.readLine();
                    LOGGER.warn(error);
                }
            }
            var7_11 = null;
            try (BufferedReader rdr = new BufferedReader(new InputStreamReader(process.getInputStream(), "UTF-8"));){
                this.processBundlerAuditOutput(dependency, engine, rdr);
            }
            catch (Throwable throwable) {
                var7_11 = throwable;
                throw throwable;
            }
        }
        catch (IOException ioe) {
            LOGGER.warn("bundle-audit failure", (Throwable)ioe);
        }
    }

    private void processBundlerAuditOutput(Dependency original, Engine engine, BufferedReader rdr) throws IOException {
        String nextLine;
        String parentName = original.getActualFile().getParentFile().getName();
        String fileName = original.getFileName();
        String filePath = original.getFilePath();
        Dependency dependency = null;
        Vulnerability vulnerability = null;
        String gem = null;
        HashMap<String, Dependency> map = new HashMap<String, Dependency>();
        boolean appendToDescription = false;
        while (rdr.ready() && null != (nextLine = rdr.readLine())) {
            if (nextLine.startsWith(NAME)) {
                appendToDescription = false;
                gem = nextLine.substring(NAME.length());
                if (!map.containsKey(gem)) {
                    map.put(gem, this.createDependencyForGem(engine, parentName, fileName, filePath, gem));
                }
                dependency = (Dependency)map.get(gem);
                LOGGER.debug("bundle-audit ({}): {}", (Object)parentName, (Object)nextLine);
                continue;
            }
            if (nextLine.startsWith(VERSION)) {
                vulnerability = this.createVulnerability(parentName, dependency, gem, nextLine);
                continue;
            }
            if (nextLine.startsWith(ADVISORY)) {
                this.setVulnerabilityName(parentName, dependency, vulnerability, nextLine);
                continue;
            }
            if (nextLine.startsWith(CRITICALITY)) {
                this.addCriticalityToVulnerability(parentName, vulnerability, nextLine);
                continue;
            }
            if (nextLine.startsWith("URL: ")) {
                this.addReferenceToVulnerability(parentName, vulnerability, nextLine);
                continue;
            }
            if (nextLine.startsWith("Description:")) {
                appendToDescription = true;
                if (null == vulnerability) continue;
                vulnerability.setDescription("*** Vulnerability obtained from bundle-audit verbose report. Title link may not work. CPE below is guessed. CVSS score is estimated (-1.0  indicates unknown). See link below for full details. *** ");
                continue;
            }
            if (!appendToDescription || null == vulnerability) continue;
            vulnerability.setDescription(vulnerability.getDescription() + nextLine + "\n");
        }
    }

    private void setVulnerabilityName(String parentName, Dependency dependency, Vulnerability vulnerability, String nextLine) {
        String advisory = nextLine.substring(ADVISORY.length());
        if (null != vulnerability) {
            vulnerability.setName(advisory);
        }
        if (null != dependency) {
            dependency.getVulnerabilities().add(vulnerability);
        }
        LOGGER.debug("bundle-audit ({}): {}", (Object)parentName, (Object)nextLine);
    }

    private void addReferenceToVulnerability(String parentName, Vulnerability vulnerability, String nextLine) {
        String url = nextLine.substring("URL: ".length());
        if (null != vulnerability) {
            Reference ref = new Reference();
            ref.setName(vulnerability.getName());
            ref.setSource("bundle-audit");
            ref.setUrl(url);
            vulnerability.getReferences().add(ref);
        }
        LOGGER.debug("bundle-audit ({}): {}", (Object)parentName, (Object)nextLine);
    }

    private void addCriticalityToVulnerability(String parentName, Vulnerability vulnerability, String nextLine) {
        if (null != vulnerability) {
            String criticality = nextLine.substring(CRITICALITY.length()).trim();
            float score = -1.0f;
            Vulnerability v = null;
            try {
                v = this.cvedb.getVulnerability(vulnerability.getName());
            }
            catch (DatabaseException ex) {
                LOGGER.debug("Unable to look up vulnerability {}", (Object)vulnerability.getName());
            }
            if (v != null) {
                score = v.getCvssScore();
            } else if ("High".equalsIgnoreCase(criticality)) {
                score = 8.5f;
            } else if ("Medium".equalsIgnoreCase(criticality)) {
                score = 5.5f;
            } else if ("Low".equalsIgnoreCase(criticality)) {
                score = 2.0f;
            }
            vulnerability.setCvssScore(score);
        }
        LOGGER.debug("bundle-audit ({}): {}", (Object)parentName, (Object)nextLine);
    }

    private Vulnerability createVulnerability(String parentName, Dependency dependency, String gem, String nextLine) {
        Vulnerability vulnerability = null;
        if (null != dependency) {
            String version = nextLine.substring(VERSION.length());
            dependency.getVersionEvidence().addEvidence("bundler-audit", "Version", version, Confidence.HIGHEST);
            vulnerability = new Vulnerability();
            vulnerability.setMatchedCPE(String.format("cpe:/a:%1$s_project:%1$s:%2$s::~~~ruby~~", gem, version), null);
            vulnerability.setCvssAccessVector("-");
            vulnerability.setCvssAccessComplexity("-");
            vulnerability.setCvssAuthentication("-");
            vulnerability.setCvssAvailabilityImpact("-");
            vulnerability.setCvssConfidentialityImpact("-");
            vulnerability.setCvssIntegrityImpact("-");
        }
        LOGGER.debug("bundle-audit ({}): {}", (Object)parentName, (Object)nextLine);
        return vulnerability;
    }

    private Dependency createDependencyForGem(Engine engine, String parentName, String fileName, String filePath, String gem) throws IOException {
        File gemFile = new File(Settings.getTempDirectory(), gem + "_Gemfile.lock");
        if (!gemFile.createNewFile()) {
            throw new IOException("Unable to create temporary gem file");
        }
        String displayFileName = String.format("%s%c%s:%s", parentName, Character.valueOf(File.separatorChar), fileName, gem);
        FileUtils.write((File)gemFile, (CharSequence)displayFileName, (Charset)Charset.defaultCharset());
        Dependency dependency = new Dependency(gemFile);
        dependency.getProductEvidence().addEvidence("bundler-audit", "Name", gem, Confidence.HIGHEST);
        dependency.setDisplayFileName(displayFileName);
        dependency.setFileName(fileName);
        dependency.setFilePath(filePath);
        engine.getDependencies().add(dependency);
        return dependency;
    }
}

