/*
 * Decompiled with CFR 0.152.
 */
package org.owasp.dependencycheck.analyzer;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.queryparser.classic.ParseException;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TopDocs;
import org.owasp.dependencycheck.Engine;
import org.owasp.dependencycheck.analyzer.AnalysisPhase;
import org.owasp.dependencycheck.analyzer.Analyzer;
import org.owasp.dependencycheck.analyzer.exception.AnalysisException;
import org.owasp.dependencycheck.data.cpe.CpeMemoryIndex;
import org.owasp.dependencycheck.data.cpe.IndexEntry;
import org.owasp.dependencycheck.data.cpe.IndexException;
import org.owasp.dependencycheck.data.lucene.LuceneUtils;
import org.owasp.dependencycheck.data.nvdcve.CveDB;
import org.owasp.dependencycheck.data.nvdcve.DatabaseException;
import org.owasp.dependencycheck.dependency.Confidence;
import org.owasp.dependencycheck.dependency.Dependency;
import org.owasp.dependencycheck.dependency.Evidence;
import org.owasp.dependencycheck.dependency.EvidenceCollection;
import org.owasp.dependencycheck.dependency.Identifier;
import org.owasp.dependencycheck.dependency.VulnerableSoftware;
import org.owasp.dependencycheck.utils.DependencyVersion;
import org.owasp.dependencycheck.utils.DependencyVersionUtil;

public class CPEAnalyzer
implements Analyzer {
    static final int MAX_QUERY_RESULTS = 25;
    static final String WEIGHTING_BOOST = "^5";
    static final String CLEANSE_CHARACTER_RX = "[^A-Za-z0-9 ._-]";
    static final String CLEANSE_NONALPHA_RX = "[^A-Za-z]*";
    static final int STRING_BUILDER_BUFFER = 20;
    private CpeMemoryIndex cpe;
    private CveDB cve;

    @Override
    public String getName() {
        return "CPE Analyzer";
    }

    @Override
    public AnalysisPhase getAnalysisPhase() {
        return AnalysisPhase.IDENTIFIER_ANALYSIS;
    }

    @Override
    public void initialize() throws Exception {
        this.open();
    }

    public void open() throws IOException, DatabaseException {
        Logger.getLogger(CPEAnalyzer.class.getName()).log(Level.FINE, "Opening the CVE Database");
        this.cve = new CveDB();
        this.cve.open();
        Logger.getLogger(CPEAnalyzer.class.getName()).log(Level.FINE, "Creating the Lucene CPE Index");
        this.cpe = CpeMemoryIndex.getInstance();
        try {
            this.cpe.open(this.cve);
        }
        catch (IndexException ex) {
            Logger.getLogger(CPEAnalyzer.class.getName()).log(Level.FINE, "IndexException", ex);
            throw new DatabaseException(ex);
        }
    }

    @Override
    public void close() {
        if (this.cpe != null) {
            this.cpe.close();
        }
        if (this.cve != null) {
            this.cve.close();
        }
    }

    protected void determineCPE(Dependency dependency) throws CorruptIndexException, IOException, ParseException {
        Confidence confidence = Confidence.HIGHEST;
        String vendors = this.addEvidenceWithoutDuplicateTerms("", dependency.getVendorEvidence(), confidence);
        String products = this.addEvidenceWithoutDuplicateTerms("", dependency.getProductEvidence(), confidence);
        this.addEvidenceWithoutDuplicateTerms("", dependency.getVersionEvidence(), confidence);
        int ctr = 0;
        do {
            if (!vendors.isEmpty() && !products.isEmpty()) {
                List<IndexEntry> entries = this.searchCPE(vendors, products, dependency.getProductEvidence().getWeighting(), dependency.getVendorEvidence().getWeighting());
                for (IndexEntry e : entries) {
                    if (!this.verifyEntry(e, dependency)) continue;
                    String vendor = e.getVendor();
                    String product = e.getProduct();
                    this.determineIdentifiers(dependency, vendor, product);
                }
            }
            confidence = this.reduceConfidence(confidence);
            if (dependency.getVendorEvidence().contains(confidence)) {
                vendors = this.addEvidenceWithoutDuplicateTerms(vendors, dependency.getVendorEvidence(), confidence);
            }
            if (dependency.getProductEvidence().contains(confidence)) {
                products = this.addEvidenceWithoutDuplicateTerms(products, dependency.getProductEvidence(), confidence);
            }
            if (!dependency.getVersionEvidence().contains(confidence)) continue;
            this.addEvidenceWithoutDuplicateTerms("", dependency.getVersionEvidence(), confidence);
        } while (++ctr < 4);
    }

    private String addEvidenceWithoutDuplicateTerms(String text, EvidenceCollection ec, Confidence confidenceFilter) {
        String txt = text == null ? "" : text;
        StringBuilder sb = new StringBuilder(txt.length() + 20 * ec.size());
        sb.append(' ').append(txt).append(' ');
        for (Evidence e : ec.iterator(confidenceFilter)) {
            String value = e.getValue();
            if (value.startsWith("http://")) {
                value = value.substring(7).replaceAll("\\.", " ");
            }
            if (value.startsWith("https://")) {
                value = value.substring(8).replaceAll("\\.", " ");
            }
            if (sb.indexOf(" " + value + " ") >= 0) continue;
            sb.append(value).append(' ');
        }
        return sb.toString().trim();
    }

    private Confidence reduceConfidence(Confidence c) {
        if (c == Confidence.HIGHEST) {
            return Confidence.HIGH;
        }
        if (c == Confidence.HIGH) {
            return Confidence.MEDIUM;
        }
        return Confidence.LOW;
    }

    protected List<IndexEntry> searchCPE(String vendor, String product, Set<String> vendorWeightings, Set<String> productWeightings) throws CorruptIndexException, IOException, ParseException {
        ArrayList<IndexEntry> ret = new ArrayList<IndexEntry>(25);
        String searchString = this.buildSearch(vendor, product, vendorWeightings, productWeightings);
        if (searchString == null) {
            return ret;
        }
        TopDocs docs = this.cpe.search(searchString, 25);
        for (ScoreDoc d : docs.scoreDocs) {
            if (!((double)d.score >= 0.08)) continue;
            Document doc = this.cpe.getDocument(d.doc);
            IndexEntry entry = new IndexEntry();
            entry.setVendor(doc.get("vendor"));
            entry.setProduct(doc.get("product"));
            entry.setSearchScore(d.score);
            if (ret.contains(entry)) continue;
            ret.add(entry);
        }
        return ret;
    }

    protected String buildSearch(String vendor, String product, Set<String> vendorWeighting, Set<String> productWeightings) {
        String v = vendor;
        String p = product;
        StringBuilder sb = new StringBuilder(v.length() + p.length() + "product".length() + "vendor".length() + 20);
        if (!this.appendWeightedSearch(sb, "product", p, productWeightings)) {
            return null;
        }
        sb.append(" AND ");
        if (!this.appendWeightedSearch(sb, "vendor", v, vendorWeighting)) {
            return null;
        }
        return sb.toString();
    }

    private boolean appendWeightedSearch(StringBuilder sb, String field, String searchText, Set<String> weightedText) {
        sb.append(" ").append(field).append(":( ");
        String cleanText = this.cleanseText(searchText);
        if ("".equals(cleanText)) {
            return false;
        }
        if (weightedText == null || weightedText.isEmpty()) {
            LuceneUtils.appendEscapedLuceneQuery(sb, cleanText);
        } else {
            StringTokenizer tokens = new StringTokenizer(cleanText);
            while (tokens.hasMoreElements()) {
                String word = tokens.nextToken();
                String temp = null;
                for (String weighted : weightedText) {
                    String weightedStr = this.cleanseText(weighted);
                    if (!this.equalsIgnoreCaseAndNonAlpha(word, weightedStr)) continue;
                    temp = LuceneUtils.escapeLuceneQuery(word) + WEIGHTING_BOOST;
                    if (word.equalsIgnoreCase(weightedStr)) continue;
                    temp = temp + " " + LuceneUtils.escapeLuceneQuery(weightedStr) + WEIGHTING_BOOST;
                }
                if (temp == null) {
                    temp = LuceneUtils.escapeLuceneQuery(word);
                }
                sb.append(" ").append(temp);
            }
        }
        sb.append(" ) ");
        return true;
    }

    private String cleanseText(String text) {
        return text.replaceAll(CLEANSE_CHARACTER_RX, " ");
    }

    private boolean equalsIgnoreCaseAndNonAlpha(String l, String r) {
        if (l == null || r == null) {
            return false;
        }
        String left = l.replaceAll(CLEANSE_NONALPHA_RX, "");
        String right = r.replaceAll(CLEANSE_NONALPHA_RX, "");
        return left.equalsIgnoreCase(right);
    }

    private boolean verifyEntry(IndexEntry entry, Dependency dependency) {
        boolean isValid = false;
        if (this.collectionContainsString(dependency.getProductEvidence(), entry.getProduct()) && this.collectionContainsString(dependency.getVendorEvidence(), entry.getVendor())) {
            isValid = true;
        }
        return isValid;
    }

    private boolean collectionContainsString(EvidenceCollection ec, String text) {
        if (text == null) {
            return false;
        }
        String[] words = text.split("[\\s_-]");
        ArrayList<String> list = new ArrayList<String>();
        String tempWord = null;
        for (String word : words) {
            if (tempWord != null) {
                list.add(tempWord + word);
                tempWord = null;
                continue;
            }
            if (word.length() <= 2) {
                tempWord = word;
                continue;
            }
            list.add(word);
        }
        if (tempWord != null && !list.isEmpty()) {
            String tmp = (String)list.get(list.size() - 1) + tempWord;
            list.add(tmp);
        }
        boolean contains = true;
        for (String word : list) {
            contains &= ec.containsUsedString(word);
        }
        return contains;
    }

    @Override
    public void analyze(Dependency dependency, Engine engine) throws AnalysisException {
        try {
            this.determineCPE(dependency);
        }
        catch (CorruptIndexException ex) {
            throw new AnalysisException("CPE Index is corrupt.", ex);
        }
        catch (IOException ex) {
            throw new AnalysisException("Failure opening the CPE Index.", ex);
        }
        catch (ParseException ex) {
            throw new AnalysisException("Unable to parse the generated Lucene query for this dependency.", ex);
        }
    }

    private void determineIdentifiers(Dependency dependency, String vendor, String product) throws UnsupportedEncodingException {
        Set<VulnerableSoftware> cpes = this.cve.getCPEs(vendor, product);
        DependencyVersion bestGuess = new DependencyVersion("-");
        Confidence bestGuessConf = null;
        ArrayList<IdentifierMatch> collected = new ArrayList<IdentifierMatch>();
        for (Confidence conf : Confidence.values()) {
            for (Evidence evidence : dependency.getVersionEvidence().iterator(conf)) {
                DependencyVersion evVer = DependencyVersionUtil.parseVersion(evidence.getValue());
                if (evVer == null) continue;
                for (VulnerableSoftware vs : cpes) {
                    DependencyVersion dbVer = vs.getRevision() != null && !vs.getRevision().isEmpty() ? DependencyVersionUtil.parseVersion(vs.getVersion() + "." + vs.getRevision()) : DependencyVersionUtil.parseVersion(vs.getVersion());
                    if (dbVer == null || evVer.equals(dbVer)) {
                        String url = String.format("http://web.nvd.nist.gov/view/vuln/search?cpe=%s", URLEncoder.encode(vs.getName(), "UTF-8"));
                        IdentifierMatch match = new IdentifierMatch("cpe", vs.getName(), url, IdentifierConfidence.EXACT_MATCH, conf);
                        collected.add(match);
                        continue;
                    }
                    if (evVer.getVersionParts().size() > dbVer.getVersionParts().size() || !evVer.matchesAtLeastThreeLevels(dbVer) || bestGuessConf != null && bestGuessConf.compareTo(conf) <= 0 || bestGuess.getVersionParts().size() >= dbVer.getVersionParts().size()) continue;
                    bestGuess = dbVer;
                    bestGuessConf = conf;
                }
                if (bestGuessConf != null && bestGuessConf.compareTo(conf) <= 0 || bestGuess.getVersionParts().size() >= evVer.getVersionParts().size()) continue;
                bestGuess = evVer;
                bestGuessConf = conf;
            }
        }
        String cpeName = String.format("cpe:/a:%s:%s:%s", vendor, product, bestGuess.toString());
        String url = null;
        if (bestGuessConf == null) {
            bestGuessConf = Confidence.LOW;
        }
        IdentifierMatch match = new IdentifierMatch("cpe", cpeName, url, IdentifierConfidence.BEST_GUESS, bestGuessConf);
        collected.add(match);
        Collections.sort(collected);
        IdentifierConfidence bestIdentifierQuality = ((IdentifierMatch)collected.get(0)).getConfidence();
        Confidence bestEvidenceQuality = ((IdentifierMatch)collected.get(0)).getEvidenceConfidence();
        for (IdentifierMatch m : collected) {
            if (!bestIdentifierQuality.equals((Object)m.getConfidence()) || !bestEvidenceQuality.equals((Object)m.getEvidenceConfidence())) continue;
            Identifier i = m.getIdentifier();
            if (bestIdentifierQuality == IdentifierConfidence.BEST_GUESS) {
                i.setConfidence(Confidence.LOW);
            } else {
                i.setConfidence(bestEvidenceQuality);
            }
            dependency.addIdentifier(i);
        }
    }

    private static class IdentifierMatch
    implements Comparable<IdentifierMatch> {
        private Confidence evidenceConfidence;
        private IdentifierConfidence confidence;
        private Identifier identifier;

        IdentifierMatch(String type, String value, String url, IdentifierConfidence identifierConfidence, Confidence evidenceConfidence) {
            this.identifier = new Identifier(type, value, url);
            this.confidence = identifierConfidence;
            this.evidenceConfidence = evidenceConfidence;
        }

        public Confidence getEvidenceConfidence() {
            return this.evidenceConfidence;
        }

        public void setEvidenceConfidence(Confidence evidenceConfidence) {
            this.evidenceConfidence = evidenceConfidence;
        }

        public IdentifierConfidence getConfidence() {
            return this.confidence;
        }

        public void setConfidence(IdentifierConfidence confidence) {
            this.confidence = confidence;
        }

        public Identifier getIdentifier() {
            return this.identifier;
        }

        public void setIdentifier(Identifier identifier) {
            this.identifier = identifier;
        }

        public String toString() {
            return "IdentifierMatch{evidenceConfidence=" + (Object)((Object)this.evidenceConfidence) + ", confidence=" + (Object)((Object)this.confidence) + ", identifier=" + this.identifier + '}';
        }

        public int hashCode() {
            int hash = 5;
            hash = 97 * hash + (this.evidenceConfidence != null ? this.evidenceConfidence.hashCode() : 0);
            hash = 97 * hash + (this.confidence != null ? this.confidence.hashCode() : 0);
            hash = 97 * hash + (this.identifier != null ? this.identifier.hashCode() : 0);
            return hash;
        }

        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            IdentifierMatch other = (IdentifierMatch)obj;
            if (this.evidenceConfidence != other.evidenceConfidence) {
                return false;
            }
            if (this.confidence != other.confidence) {
                return false;
            }
            return this.identifier == other.identifier || this.identifier != null && this.identifier.equals(other.identifier);
        }

        @Override
        public int compareTo(IdentifierMatch o) {
            int conf = this.confidence.compareTo(o.confidence);
            if (conf == 0 && (conf = this.evidenceConfidence.compareTo(o.evidenceConfidence)) == 0) {
                conf = this.identifier.compareTo(o.identifier);
            }
            return conf;
        }
    }

    private static enum IdentifierConfidence {
        EXACT_MATCH,
        BEST_GUESS;

    }
}

