/**
 * Copyright 2007-2012 Bull S.A.S.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.util.ant.archive.info;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.tools.ant.types.FileSet;

/**
 * Class that holds information about an archive.
 * @author Florent Benoit
 */
public class ArchiveInfo {

    /**
     * Reference to the standard deployment descriptor.
     */
    private File dd = null;

    /**
     * Reference to the specific deployment descriptor.
     */
    private File specificDD = null;

    /**
     * Reference to the persistence deployment descriptor.
     */
    private File persistenceDD = null;

    /**
     * Full path to the archive.
     */
    private File dest = null;

    /**
     * List of fileset used to add files.
     */
    private List<FileSet> fileSetList = null;

    /**
     * Exploded mode or not ? (default = file).
     */
    private boolean exploded = false;

    /**
     * Reference to the manifest.
     */
    private File manifest = null;

    /**
     * Default constructor.
     */
    public ArchiveInfo() {
        this.fileSetList = new ArrayList<FileSet>();

    }

    /**
     * Add the given fileset to the list of existing fileset.
     * @param fileSet the fileset to add.
     */
    public void addFileSet(final FileSet fileSet) {
        this.fileSetList.add(fileSet);
    }

    /**
     * Gets the list of fileset to include in the archive.
     * @return the list of fileset to include in the archive.
     */
    public List<FileSet> getFileSetList() {
        return this.fileSetList;
    }

    /**
     * Sets the reference to the deployment descriptor.
     * @param dd the given deployment descriptor.
     */
    public void setDD(final File dd) {
        this.dd = dd;
    }

    /**
     * Gets the reference to the deployment descriptor.
     * @return the standard deployment descriptor.
     */
    public File getDD() {
        return this.dd;
    }

    /**
     * Sets the reference to the specific deployment descriptor.
     * @param specificDD the given specific deployment descriptor.
     */
    public void setSpecificDD(final File specificDD) {
        this.specificDD = specificDD;
    }

    /**
     * Gets the reference to the specific deployment descriptor.
     * @return the specific deployment descriptor.
     */
    public File getSpecificDD() {
        return this.specificDD;
    }

    /**
     * Sets the reference to the persistence deployment descriptor.
     * @param persistenceDD the given persistence deployment descriptor.
     */
    public void setPersistenceDD(final File persistenceDD) {
        this.persistenceDD = persistenceDD;
    }

    /**
     * Gets the reference to the persistence deployment descriptor.
     * @return the persistence deployment descriptor.
     */
    public File getPersistenceDD() {
        return this.persistenceDD;
    }

    /**
     * Sets the path to the archive that will be built.
     * @param dest the reference to resulting archive path.
     */
    public void setDest(final File dest) {
        this.dest = dest;
    }

    /**
     * Gets the path to the archive that will be built.
     * @return the reference to resulting archive path.
     */
    public File getDest() {
        return this.dest;
    }

    /**
     * Sets the exploded mode to true or false.
     * @param exploded boolean true/false
     */
    public void setExploded(final boolean exploded) {
        this.exploded = exploded;
    }

    /**
     * Gets the state : exploded mode or not ?
     * @return the state : exploded mode or not ?
     */
    public boolean isExploded() {
        return this.exploded;
    }

    /**
     * Gets the reference to the manifest.
     * @return the reference to the manifest.
     */
    public File getManifest() {
        return this.manifest;
    }

    /**
     * Sets the reference to the manifest.
     * @param manifest the reference to the manifest.
     */
    public void setManifest(final File manifest) {
        this.manifest = manifest;
    }
}
