/*
 * Copyright (c) OSGi Alliance (2016). All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.osgi.service.zigbee.types;

import java.io.IOException;
import org.osgi.service.zigbee.ZigBeeDataInput;
import org.osgi.service.zigbee.ZigBeeDataOutput;
import org.osgi.service.zigbee.ZigBeeDataTypes;
import org.osgi.service.zigbee.descriptions.ZCLSimpleTypeDescription;

/**
 * A singleton class that represents the 'Time Of Day' data type, as it is
 * defined in the ZigBee Cluster Library specification.
 * 
 * The ZigBee data type is mapped to a byte[4] array where byte[0] must contain
 * the Hour field and byte[3] the Hundredths of seconds. The array is
 * marshaled/unmarshaled starting from byte[0].
 * 
 * @author $Id: 8eccfa7d3299bc368853808d14965ea0ab80c082 $
 * 
 */
public class ZigBeeTimeOfDay
		implements ZCLSimpleTypeDescription {

	private final static ZigBeeTimeOfDay instance = new ZigBeeTimeOfDay();

	private ZigBeeTimeOfDay() {
	}

	/**
	 * Gets a singleton instance of this class.
	 * 
	 * @return the singleton instance.
	 */
	public static ZigBeeTimeOfDay getInstance() {
		return instance;
	}

	public String getName() {
		return "TimeOfDay";
	}

	public boolean isAnalog() {
		return true;
	}

	public Class getJavaDataType() {
		return byte[].class;
	}

	public short getId() {
		return ZigBeeDataTypes.TIME_OF_DAY;
	}

	public void serialize(ZigBeeDataOutput os, Object value) throws IOException {
		ZigBeeDefaultSerializer.serializeDataType(os, ZigBeeDataTypes.TIME_OF_DAY, value);
	}

	public Object deserialize(ZigBeeDataInput is) throws IOException {
		return ZigBeeDefaultSerializer.deserializeDataType(is, ZigBeeDataTypes.TIME_OF_DAY);
	}

}
