/*******************************************************************************
 * Copyright (c) Contributors to the Eclipse Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0 
 *******************************************************************************/

package org.osgi.service.servlet.runtime.dto;

import java.util.Map;

import org.osgi.dto.DTO;

/**
 * Represents a {@code jakarta.servlet.ServletContext} created for servlets,
 * resources, servlet Filters, and listeners associated with that servlet
 * context. The Servlet Context is usually backed by a
 * {@link org.osgi.service.servlet.context.ServletContextHelper} service.
 * 
 * @NotThreadSafe
 * @author $Id: edeab606739715e38fdab31edab148665874d68a $
 */
public class ServletContextDTO extends DTO {
	/**
	 * The name of the servlet context.
	 * The name of the corresponding
	 * {@link org.osgi.service.servlet.context.ServletContextHelper}.
	 * <p>
	 * This is the value returned by the
	 * {@code ServletContext.getServletContextName()} method.
	 */
	public String name;

	/**
	 * The servlet context path.
	 * 
	 * This is the value returned by the {@code ServletContext.getContextPath()}
	 * method.
	 */
	public String				contextPath;

	/**
	 * The servlet context initialization parameters. This is the set of
	 * parameters provided when registering this context. Additional parameters
	 * like the Http Service Runtime attributes are not included. If the context
	 * has no initialization parameters, this map is empty.
	 */
	public Map<String, String>	initParams;

	/**
	 * The servlet context attributes.
	 * 
	 * <p>
	 * The value type must be a numerical type, {@code Boolean}, {@code String},
	 * {@code DTO} or an array of any of the former. Therefore this method will
	 * only return the attributes of the servlet context conforming to this
	 * constraint. Other attributes are omitted. If there are no attributes
	 * conforming to the constraint, an empty map is returned.
	 */
	public Map<String, Object>	attributes;

	/**
	 * Service property identifying the servlet context. In the case of a
	 * servlet context backed by a
	 * {@code org.osgi.service.servlet.context.ServletContextHelper} registered
	 * in the service registry and picked up by a Servlet Whiteboard
	 * Implementation, this value is not negative and corresponds to the service
	 * id in the registry. If the servlet context is not backed by a service
	 * registered in the service registry, the value is negative and a unique
	 * negative value is generated by the Http Service Runtime in this case.
	 */
	public long					serviceId;

	/**
	 * Returns the representations of the {@code Servlet} services associated
	 * with this context.
	 * 
	 * The representations of the {@code Servlet} services associated with this
	 * context. The returned array may be empty if this context is currently not
	 * associated with any {@code Servlet} services.
	 */
	public ServletDTO[]			servletDTOs;

	/**
	 * Returns the representations of the resource services associated with this
	 * context.
	 * 
	 * The representations of the resource services associated with this
	 * context. The returned array may be empty if this context is currently not
	 * associated with any resource services.
	 */
	public ResourceDTO[]		resourceDTOs;

	/**
	 * Returns the representations of the servlet {@code Filter} services
	 * associated with this context.
	 * 
	 * The representations of the servlet {@code Filter} services associated
	 * with this context. The returned array may be empty if this context is
	 * currently not associated with any servlet {@code Filter} services.
	 */
	public FilterDTO[]			filterDTOs;

	/**
	 * Returns the representations of the error page {@code Servlet} services
	 * associated with this context.
	 * 
	 * The representations of the error page {@code Servlet} services associated
	 * with this context. The returned array may be empty if this context is
	 * currently not associated with any error pages.
	 */
	public ErrorPageDTO[]		errorPageDTOs;

	/**
	 * Returns the representations of the listener services associated with this
	 * context.
	 * 
	 * The representations of the listener services associated with this
	 * context. The returned array may be empty if this context is currently not
	 * associated with any listener services.
	 */
	public ListenerDTO[]		listenerDTOs;
}
