/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.updater.trip;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Multimaps;
import com.google.transit.realtime.GtfsRealtime;
import de.mfdz.MfdzRealtimeExtensions;
import java.text.ParseException;
import java.time.Duration;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import javax.annotation.Nonnull;
import org.opentripplanner.framework.i18n.NonLocalizedString;
import org.opentripplanner.framework.lang.StringUtils;
import org.opentripplanner.framework.time.ServiceDateUtils;
import org.opentripplanner.gtfs.mapping.TransitModeMapper;
import org.opentripplanner.model.StopTime;
import org.opentripplanner.model.Timetable;
import org.opentripplanner.model.TimetableSnapshot;
import org.opentripplanner.model.TimetableSnapshotProvider;
import org.opentripplanner.model.TripTimesPatch;
import org.opentripplanner.model.UpdateError;
import org.opentripplanner.model.UpdateSuccess;
import org.opentripplanner.routing.algorithm.raptoradapter.transit.mappers.TransitLayerUpdater;
import org.opentripplanner.transit.model.basic.TransitMode;
import org.opentripplanner.transit.model.framework.Deduplicator;
import org.opentripplanner.transit.model.framework.FeedScopedId;
import org.opentripplanner.transit.model.framework.Result;
import org.opentripplanner.transit.model.network.Route;
import org.opentripplanner.transit.model.network.RouteBuilder;
import org.opentripplanner.transit.model.network.StopPattern;
import org.opentripplanner.transit.model.network.TripPattern;
import org.opentripplanner.transit.model.organization.Agency;
import org.opentripplanner.transit.model.site.RegularStop;
import org.opentripplanner.transit.model.site.StopLocation;
import org.opentripplanner.transit.model.timetable.RealTimeState;
import org.opentripplanner.transit.model.timetable.Trip;
import org.opentripplanner.transit.model.timetable.TripBuilder;
import org.opentripplanner.transit.model.timetable.TripTimes;
import org.opentripplanner.transit.service.DefaultTransitService;
import org.opentripplanner.transit.service.TransitEditorService;
import org.opentripplanner.transit.service.TransitModel;
import org.opentripplanner.updater.GtfsRealtimeFuzzyTripMatcher;
import org.opentripplanner.updater.GtfsRealtimeMapper;
import org.opentripplanner.updater.TimetableSnapshotSourceParameters;
import org.opentripplanner.updater.spi.ResultLogger;
import org.opentripplanner.updater.spi.UpdateResult;
import org.opentripplanner.updater.trip.AddedRoute;
import org.opentripplanner.updater.trip.AddedStopTime;
import org.opentripplanner.updater.trip.BackwardsDelayPropagationType;
import org.opentripplanner.updater.trip.TripPatternCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimetableSnapshotSource
implements TimetableSnapshotProvider {
    private static final Logger LOG = LoggerFactory.getLogger(TimetableSnapshotSource.class);
    private static final long MAX_ARRIVAL_DEPARTURE_TIME = 172800L;
    private final TimetableSnapshot buffer = new TimetableSnapshot();
    private final ReentrantLock bufferLock = new ReentrantLock(true);
    private final TripPatternCache tripPatternCache = new TripPatternCache();
    private final ZoneId timeZone;
    private final TransitEditorService transitService;
    private final TransitLayerUpdater transitLayerUpdater;
    private final Duration maxSnapshotFrequency;
    private volatile TimetableSnapshot snapshot = null;
    private final boolean purgeExpiredData;
    protected LocalDate lastPurgeDate = null;
    protected long lastSnapshotTime = -1L;
    private final Deduplicator deduplicator;
    private final Map<FeedScopedId, Integer> serviceCodes;
    private final Supplier<LocalDate> localDateNow;

    public TimetableSnapshotSource(TimetableSnapshotSourceParameters parameters, TransitModel transitModel) {
        this(parameters, transitModel, () -> LocalDate.now(transitModel.getTimeZone()));
    }

    TimetableSnapshotSource(TimetableSnapshotSourceParameters parameters, TransitModel transitModel, Supplier<LocalDate> localDateNow) {
        this.timeZone = transitModel.getTimeZone();
        this.transitService = new DefaultTransitService(transitModel);
        this.transitLayerUpdater = transitModel.getTransitLayerUpdater();
        this.deduplicator = transitModel.getDeduplicator();
        this.serviceCodes = transitModel.getServiceCodes();
        this.maxSnapshotFrequency = parameters.maxSnapshotFrequency();
        this.purgeExpiredData = parameters.purgeExpiredData();
        this.localDateNow = localDateNow;
        transitModel.initTimetableSnapshotProvider(this);
    }

    @Override
    public TimetableSnapshot getTimetableSnapshot() {
        TimetableSnapshot snapshotToReturn;
        if (this.bufferLock.tryLock()) {
            try {
                snapshotToReturn = this.getTimetableSnapshot(false);
            }
            finally {
                this.bufferLock.unlock();
            }
        } else {
            snapshotToReturn = this.snapshot;
        }
        return snapshotToReturn;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public UpdateResult applyTripUpdates(GtfsRealtimeFuzzyTripMatcher fuzzyTripMatcher, BackwardsDelayPropagationType backwardsDelayPropagationType, boolean fullDataset, List<GtfsRealtime.TripUpdate> updates, String feedId) {
        if (updates == null) {
            LOG.warn("updates is null");
            return UpdateResult.empty();
        }
        this.bufferLock.lock();
        HashMap<GtfsRealtime.TripDescriptor.ScheduleRelationship, Integer> failuresByRelationship = new HashMap<GtfsRealtime.TripDescriptor.ScheduleRelationship, Integer>();
        ArrayList<Result<UpdateSuccess, UpdateError>> results = new ArrayList<Result<UpdateSuccess, UpdateError>>();
        try {
            if (fullDataset) {
                this.buffer.clear(feedId);
            }
            LOG.debug("message contains {} trip updates", (Object)updates.size());
            int uIndex = 0;
            for (GtfsRealtime.TripUpdate tripUpdate : updates) {
                LocalDate serviceDate;
                FeedScopedId tripId;
                GtfsRealtime.TripDescriptor tripDescriptor;
                block25: {
                    if (!tripUpdate.hasTrip()) {
                        TimetableSnapshotSource.debug(feedId, "", "Missing TripDescriptor in gtfs-rt trip update: \n{}", tripUpdate);
                        continue;
                    }
                    if (fuzzyTripMatcher != null) {
                        GtfsRealtime.TripDescriptor trip = fuzzyTripMatcher.match(feedId, tripUpdate.getTrip());
                        tripUpdate = tripUpdate.toBuilder().setTrip(trip).build();
                    }
                    if (!(tripDescriptor = tripUpdate.getTrip()).hasTripId() || tripDescriptor.getTripId().isBlank()) {
                        TimetableSnapshotSource.debug(feedId, "", "No trip id found for gtfs-rt trip update: \n{}", tripUpdate);
                        continue;
                    }
                    tripId = new FeedScopedId(feedId, tripUpdate.getTrip().getTripId());
                    if (tripDescriptor.hasStartDate()) {
                        try {
                            serviceDate = ServiceDateUtils.parseString(tripDescriptor.getStartDate());
                            break block25;
                        }
                        catch (ParseException e) {
                            TimetableSnapshotSource.debug(tripId, "Failed to parse start date in gtfs-rt trip update: {}", tripDescriptor.getStartDate());
                            continue;
                        }
                    }
                    serviceDate = this.localDateNow.get();
                }
                LOG.debug("trip update #{} ({} updates) :", (Object)(++uIndex), (Object)tripUpdate.getStopTimeUpdateCount());
                LOG.trace("{}", (Object)tripUpdate);
                GtfsRealtime.TripDescriptor.ScheduleRelationship tripScheduleRelationship = this.determineTripScheduleRelationship(tripDescriptor);
                Result result = switch (tripScheduleRelationship) {
                    default -> throw new IncompatibleClassChangeError();
                    case GtfsRealtime.TripDescriptor.ScheduleRelationship.SCHEDULED -> this.handleScheduledTrip(tripUpdate, tripId, serviceDate, backwardsDelayPropagationType);
                    case GtfsRealtime.TripDescriptor.ScheduleRelationship.ADDED -> this.validateAndHandleAddedTrip(tripUpdate, tripDescriptor, tripId, serviceDate);
                    case GtfsRealtime.TripDescriptor.ScheduleRelationship.CANCELED -> this.handleCanceledTrip(tripId, serviceDate, CancelationType.CANCEL);
                    case GtfsRealtime.TripDescriptor.ScheduleRelationship.DELETED -> this.handleCanceledTrip(tripId, serviceDate, CancelationType.DELETE);
                    case GtfsRealtime.TripDescriptor.ScheduleRelationship.REPLACEMENT -> this.validateAndHandleModifiedTrip(tripUpdate, tripDescriptor, tripId, serviceDate);
                    case GtfsRealtime.TripDescriptor.ScheduleRelationship.UNSCHEDULED -> UpdateError.result(tripId, UpdateError.UpdateErrorType.NOT_IMPLEMENTED_UNSCHEDULED);
                    case GtfsRealtime.TripDescriptor.ScheduleRelationship.DUPLICATED -> UpdateError.result(tripId, UpdateError.UpdateErrorType.NOT_IMPLEMENTED_DUPLICATED);
                };
                results.add(result);
                if (!result.isFailure()) continue;
                TimetableSnapshotSource.debug(tripId, "Failed to apply TripUpdate.", new Object[0]);
                LOG.trace(" Contents: {}", (Object)tripUpdate);
                if (failuresByRelationship.containsKey((Object)tripScheduleRelationship)) {
                    Integer c = (Integer)failuresByRelationship.get((Object)tripScheduleRelationship);
                    c = c + 1;
                    failuresByRelationship.put(tripScheduleRelationship, c);
                    continue;
                }
                failuresByRelationship.put(tripScheduleRelationship, 1);
            }
            if (this.purgeExpiredData) {
                boolean modified = this.purgeExpiredData();
                this.getTimetableSnapshot(modified);
            } else {
                this.getTimetableSnapshot(false);
            }
        }
        finally {
            this.bufferLock.unlock();
        }
        UpdateResult updateResult = UpdateResult.ofResults(results);
        if (fullDataset) {
            TimetableSnapshotSource.logUpdateResult(feedId, failuresByRelationship, updateResult);
        }
        return updateResult;
    }

    private static void logUpdateResult(String feedId, Map<GtfsRealtime.TripDescriptor.ScheduleRelationship, Integer> failuresByRelationship, UpdateResult updateResult) {
        ResultLogger.logUpdateResult(feedId, "gtfs-rt-trip-updates", updateResult);
        if (!failuresByRelationship.isEmpty()) {
            LOG.info("[feedId: {}] Failures by scheduleRelationship {}", (Object)feedId, failuresByRelationship);
        }
        ImmutableListMultimap warnings = Multimaps.index(updateResult.warnings(), w -> w);
        warnings.keySet().forEach(key -> {
            int count = warnings.get((Object)key).size();
            LOG.info("[feedId: {}] {} warnings of type {}", new Object[]{feedId, count, key});
        });
    }

    private TimetableSnapshot getTimetableSnapshot(boolean force) {
        long now = System.currentTimeMillis();
        if (force || now - this.lastSnapshotTime > this.maxSnapshotFrequency.toMillis()) {
            if (force || this.buffer.isDirty()) {
                LOG.debug("Committing {}", (Object)this.buffer);
                this.snapshot = this.buffer.commit(this.transitLayerUpdater, force);
            } else {
                LOG.debug("Buffer was unchanged, keeping old snapshot.");
            }
            this.lastSnapshotTime = System.currentTimeMillis();
        } else {
            LOG.debug("Snapshot frequency exceeded. Reusing snapshot {}", (Object)this.snapshot);
        }
        return this.snapshot;
    }

    private GtfsRealtime.TripDescriptor.ScheduleRelationship determineTripScheduleRelationship(GtfsRealtime.TripDescriptor tripDescriptor) {
        GtfsRealtime.TripDescriptor.ScheduleRelationship tripScheduleRelationship = GtfsRealtime.TripDescriptor.ScheduleRelationship.SCHEDULED;
        if (tripDescriptor.hasScheduleRelationship()) {
            tripScheduleRelationship = tripDescriptor.getScheduleRelationship();
        }
        return tripScheduleRelationship;
    }

    private Result<UpdateSuccess, UpdateError> handleScheduledTrip(GtfsRealtime.TripUpdate tripUpdate, FeedScopedId tripId, LocalDate serviceDate, BackwardsDelayPropagationType backwardsDelayPropagationType) {
        TripPattern pattern = this.getPatternForTripId(tripId);
        if (pattern == null) {
            TimetableSnapshotSource.debug(tripId, "No pattern found for tripId, skipping TripUpdate.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TRIP_NOT_FOUND);
        }
        if (tripUpdate.getStopTimeUpdateCount() < 1) {
            TimetableSnapshotSource.debug(tripId, "TripUpdate contains no updates, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_UPDATES);
        }
        FeedScopedId serviceId = this.transitService.getTripForId(tripId).getServiceId();
        Set<LocalDate> serviceDates = this.transitService.getCalendarService().getServiceDatesForServiceId(serviceId);
        if (!serviceDates.contains(serviceDate)) {
            TimetableSnapshotSource.debug(tripId, "SCHEDULED trip has service date {} for which trip's service is not valid, skipping.", serviceDate.toString());
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_SERVICE_ON_DATE);
        }
        this.cancelPreviouslyAddedTrip(tripId, serviceDate, CancelationType.DELETE);
        Result<TripTimesPatch, UpdateError> result = pattern.getScheduledTimetable().createUpdatedTripTimes(tripUpdate, this.timeZone, serviceDate, backwardsDelayPropagationType);
        if (result.isFailure()) {
            return result.toFailureResult();
        }
        TripTimesPatch tripTimesPatch = result.successValue();
        List<Integer> skippedStopIndices = tripTimesPatch.getSkippedStopIndices();
        TripTimes updatedTripTimes = tripTimesPatch.getTripTimes();
        updatedTripTimes.setRealTimeState(RealTimeState.UPDATED);
        if (skippedStopIndices.size() > 0) {
            StopPattern newStopPattern = pattern.getStopPattern().mutate().cancelStops(skippedStopIndices).build();
            Trip trip = this.transitService.getTripForId(tripId);
            TripPattern newPattern = this.tripPatternCache.getOrCreateTripPattern(newStopPattern, trip, pattern);
            this.cancelScheduledTrip(tripId, serviceDate, CancelationType.DELETE);
            return this.buffer.update(newPattern, updatedTripTimes, serviceDate);
        }
        return this.buffer.update(pattern, updatedTripTimes, serviceDate);
    }

    private Result<UpdateSuccess, UpdateError> validateAndHandleAddedTrip(GtfsRealtime.TripUpdate tripUpdate, GtfsRealtime.TripDescriptor tripDescriptor, FeedScopedId tripId, LocalDate serviceDate) {
        Objects.requireNonNull(tripUpdate);
        Objects.requireNonNull(serviceDate);
        Trip trip = this.transitService.getTripForId(tripId);
        if (trip != null) {
            TimetableSnapshotSource.debug(tripId, "Graph already contains trip id of ADDED trip, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TRIP_ALREADY_EXISTS);
        }
        if (!tripDescriptor.hasStartDate()) {
            TimetableSnapshotSource.debug(tripId, "ADDED trip doesn't have a start date in TripDescriptor, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_START_DATE);
        }
        List<GtfsRealtime.TripUpdate.StopTimeUpdate> stopTimeUpdates = this.removeUnknownStops(tripUpdate, tripId);
        ArrayList<UpdateSuccess.WarningType> warnings = new ArrayList<UpdateSuccess.WarningType>(0);
        if (stopTimeUpdates.size() < tripUpdate.getStopTimeUpdateCount()) {
            warnings.add(UpdateSuccess.WarningType.UNKNOWN_STOPS_REMOVED_FROM_ADDED_TRIP);
        }
        if (stopTimeUpdates.size() < 2) {
            TimetableSnapshotSource.debug(tripId, "ADDED trip has fewer than two known stops, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TOO_FEW_STOPS);
        }
        List<StopLocation> stops = this.checkNewStopTimeUpdatesAndFindStops(tripId, stopTimeUpdates);
        if (stops == null) {
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_VALID_STOPS);
        }
        return this.handleAddedTrip(tripUpdate, stopTimeUpdates, tripDescriptor, stops, tripId, serviceDate).mapSuccess(s -> s.addWarnings(warnings));
    }

    @Nonnull
    private List<GtfsRealtime.TripUpdate.StopTimeUpdate> removeUnknownStops(GtfsRealtime.TripUpdate tripUpdate, FeedScopedId tripId) {
        return tripUpdate.getStopTimeUpdateList().stream().filter(GtfsRealtime.TripUpdate.StopTimeUpdate::hasStopId).filter(st -> {
            boolean stopFound;
            FeedScopedId stopId = new FeedScopedId(tripId.getFeedId(), st.getStopId());
            boolean bl = stopFound = this.transitService.getRegularStop(stopId) != null;
            if (!stopFound) {
                TimetableSnapshotSource.debug(tripId, "Stop '{}' not found in graph. Removing from ADDED trip.", st.getStopId());
            }
            return stopFound;
        }).toList();
    }

    private List<StopLocation> checkNewStopTimeUpdatesAndFindStops(FeedScopedId tripId, List<GtfsRealtime.TripUpdate.StopTimeUpdate> stopTimeUpdates) {
        Integer previousStopSequence = null;
        Long previousTime = null;
        ArrayList<StopLocation> stops = new ArrayList<StopLocation>(stopTimeUpdates.size());
        for (int index = 0; index < stopTimeUpdates.size(); ++index) {
            Long time;
            RegularStop stop;
            GtfsRealtime.TripUpdate.StopTimeUpdate stopTimeUpdate = stopTimeUpdates.get(index);
            if (stopTimeUpdate.hasStopSequence()) {
                Integer stopSequence = stopTimeUpdate.getStopSequence();
                if (stopSequence < 0) {
                    TimetableSnapshotSource.debug(tripId, "Trip update contains negative stop sequence, skipping.", new Object[0]);
                    return null;
                }
                if (previousStopSequence != null && previousStopSequence > stopSequence) {
                    TimetableSnapshotSource.debug(tripId, "Trip update contains decreasing stop sequence, skipping.", new Object[0]);
                    return null;
                }
                previousStopSequence = stopSequence;
            }
            if (stopTimeUpdate.hasStopId()) {
                stop = this.transitService.getRegularStop(new FeedScopedId(tripId.getFeedId(), stopTimeUpdate.getStopId()));
                if (stop == null) {
                    TimetableSnapshotSource.debug(tripId, "Graph doesn't contain stop id '{}' of trip update, skipping.", stopTimeUpdate.getStopId());
                    return null;
                }
            } else {
                TimetableSnapshotSource.debug(tripId, "Trip update misses a stop id at stop time list index {}, skipping.", index);
                return null;
            }
            stops.add(stop);
            if (stopTimeUpdate.hasArrival() && stopTimeUpdate.getArrival().hasTime()) {
                time = stopTimeUpdate.getArrival().getTime();
                if (previousTime != null && previousTime > time) {
                    TimetableSnapshotSource.debug(tripId, "Trip update contains decreasing times, skipping.", new Object[0]);
                    return null;
                }
            } else {
                TimetableSnapshotSource.debug(tripId, "Trip update misses arrival time, skipping.", new Object[0]);
                return null;
            }
            previousTime = time;
            if (stopTimeUpdate.hasDeparture() && stopTimeUpdate.getDeparture().hasTime()) {
                time = stopTimeUpdate.getDeparture().getTime();
                if (previousTime != null && previousTime > time) {
                    TimetableSnapshotSource.debug(tripId, "Trip update contains decreasing times, skipping.", new Object[0]);
                    return null;
                }
            } else {
                TimetableSnapshotSource.debug(tripId, "Trip update misses departure time, skipping.", new Object[0]);
                return null;
            }
            previousTime = time;
        }
        return stops;
    }

    private Result<UpdateSuccess, UpdateError> handleAddedTrip(GtfsRealtime.TripUpdate tripUpdate, List<GtfsRealtime.TripUpdate.StopTimeUpdate> stopTimeUpdates, GtfsRealtime.TripDescriptor tripDescriptor, List<StopLocation> stops, FeedScopedId tripId, LocalDate serviceDate) {
        Objects.requireNonNull(stops);
        Preconditions.checkArgument((stopTimeUpdates.size() == stops.size() ? 1 : 0) != 0, (Object)"number of stop should match the number of stop time updates");
        this.cancelPreviouslyAddedTrip(tripId, serviceDate, CancelationType.DELETE);
        Route route = this.getOrCreateRoute(tripDescriptor, tripId);
        TripBuilder tripBuilder = Trip.of(tripId);
        tripBuilder.withRoute(route);
        Set<FeedScopedId> serviceIds = this.transitService.getCalendarService().getServiceIdsOnDate(serviceDate);
        if (serviceIds.isEmpty()) {
            TimetableSnapshotSource.debug(tripId, "ADDED trip has service date {} for which no service id is available, skipping.", serviceDate.toString());
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_SERVICE_ON_DATE);
        }
        tripBuilder.withServiceId(serviceIds.iterator().next());
        return this.addTripToGraphAndBuffer((Trip)tripBuilder.build(), tripUpdate.getVehicle(), stopTimeUpdates, stops, serviceDate, RealTimeState.ADDED);
    }

    private Route getOrCreateRoute(GtfsRealtime.TripDescriptor tripDescriptor, FeedScopedId tripId) {
        if (this.routeExists(tripId.getFeedId(), tripDescriptor)) {
            return this.transitService.getRouteForId(new FeedScopedId(tripId.getFeedId(), tripDescriptor.getRouteId()));
        }
        if (tripDescriptor.hasExtension(MfdzRealtimeExtensions.tripDescriptor) && !this.routeExists(tripId.getFeedId(), tripDescriptor)) {
            FeedScopedId routeId = new FeedScopedId(tripId.getFeedId(), tripDescriptor.getRouteId());
            RouteBuilder builder = Route.of(routeId);
            AddedRoute addedRouteExtension = AddedRoute.ofTripDescriptor(tripDescriptor);
            Agency agency = this.transitService.findAgencyById(new FeedScopedId(tripId.getFeedId(), addedRouteExtension.agencyId())).orElseGet(() -> this.fallbackAgency(tripId.getFeedId()));
            builder.withAgency(agency);
            builder.withGtfsType(addedRouteExtension.routeType());
            TransitMode mode = TransitModeMapper.mapMode(addedRouteExtension.routeType());
            builder.withMode(mode);
            String name = Objects.requireNonNullElse(addedRouteExtension.routeLongName(), tripId.toString());
            builder.withLongName(new NonLocalizedString(name));
            builder.withUrl(addedRouteExtension.routeUrl());
            Route route = (Route)builder.build();
            this.transitService.addRoutes(route);
            return route;
        }
        RouteBuilder builder = Route.of(tripId);
        builder.withAgency(this.fallbackAgency(tripId.getFeedId()));
        builder.withGtfsType(3);
        builder.withMode(TransitMode.BUS);
        NonLocalizedString longName = NonLocalizedString.ofNullable(tripDescriptor.getTripId());
        builder.withLongName(longName);
        Route route = (Route)builder.build();
        this.transitService.addRoutes(route);
        return route;
    }

    private Agency fallbackAgency(String feedId) {
        return (Agency)Agency.of(new FeedScopedId(feedId, "autogenerated-gtfs-rt-added-route")).withName("Agency automatically added by GTFS-RT update").withTimezone(this.transitService.getTimeZone().toString()).build();
    }

    private boolean routeExists(String feedId, GtfsRealtime.TripDescriptor tripDescriptor) {
        if (tripDescriptor.hasRouteId() && StringUtils.hasValue(tripDescriptor.getRouteId())) {
            FeedScopedId routeId = new FeedScopedId(feedId, tripDescriptor.getRouteId());
            return Objects.nonNull(this.transitService.getRouteForId(routeId));
        }
        return false;
    }

    private Result<UpdateSuccess, UpdateError> addTripToGraphAndBuffer(Trip trip, GtfsRealtime.VehicleDescriptor vehicleDescriptor, List<GtfsRealtime.TripUpdate.StopTimeUpdate> stopTimeUpdates, List<StopLocation> stops, LocalDate serviceDate, RealTimeState realTimeState) {
        Objects.requireNonNull(stops);
        Preconditions.checkArgument((stopTimeUpdates.size() == stops.size() ? 1 : 0) != 0, (Object)"number of stop should match the number of stop time updates");
        long midnightSecondsSinceEpoch = ServiceDateUtils.asStartOfService(serviceDate, this.timeZone).toEpochSecond();
        ArrayList<StopTime> stopTimes = new ArrayList<StopTime>(stopTimeUpdates.size());
        for (int index = 0; index < stopTimeUpdates.size(); ++index) {
            GtfsRealtime.TripUpdate.StopTimeUpdate stopTimeUpdate = stopTimeUpdates.get(index);
            StopLocation stop = stops.get(index);
            StopTime stopTime = new StopTime();
            stopTime.setTrip(trip);
            stopTime.setStop(stop);
            if (stopTimeUpdate.hasArrival() && stopTimeUpdate.getArrival().hasTime()) {
                long arrivalTime = stopTimeUpdate.getArrival().getTime() - midnightSecondsSinceEpoch;
                if (arrivalTime < 0L || arrivalTime > 172800L) {
                    TimetableSnapshotSource.debug(trip.getId(), "ADDED trip has invalid arrival time (compared to start date in TripDescriptor), skipping.", new Object[0]);
                    return UpdateError.result(trip.getId(), UpdateError.UpdateErrorType.INVALID_ARRIVAL_TIME);
                }
                stopTime.setArrivalTime((int)arrivalTime);
            }
            if (stopTimeUpdate.hasDeparture() && stopTimeUpdate.getDeparture().hasTime()) {
                long departureTime = stopTimeUpdate.getDeparture().getTime() - midnightSecondsSinceEpoch;
                if (departureTime < 0L || departureTime > 172800L) {
                    TimetableSnapshotSource.debug(trip.getId(), "ADDED trip has invalid departure time (compared to start date in TripDescriptor), skipping.", new Object[0]);
                    return UpdateError.result(trip.getId(), UpdateError.UpdateErrorType.INVALID_DEPARTURE_TIME);
                }
                stopTime.setDepartureTime((int)departureTime);
            }
            stopTime.setTimepoint(1);
            if (stopTimeUpdate.hasStopSequence()) {
                stopTime.setStopSequence(stopTimeUpdate.getStopSequence());
            }
            AddedStopTime added = AddedStopTime.ofStopTime(stopTimeUpdate);
            stopTime.setPickupType(added.pickup());
            stopTime.setDropOffType(added.dropOff());
            stopTimes.add(stopTime);
        }
        StopPattern stopPattern = new StopPattern(stopTimes);
        TripPattern originalTripPattern = this.transitService.getPatternForTrip(trip);
        TripPattern pattern = this.tripPatternCache.getOrCreateTripPattern(stopPattern, trip, originalTripPattern);
        TripTimes newTripTimes = new TripTimes(trip, stopTimes, this.deduplicator);
        for (int stopIndex = 0; stopIndex < newTripTimes.getNumStops(); ++stopIndex) {
            newTripTimes.updateArrivalTime(stopIndex, newTripTimes.getScheduledArrivalTime(stopIndex));
            newTripTimes.updateDepartureTime(stopIndex, newTripTimes.getScheduledDepartureTime(stopIndex));
        }
        int serviceCode = this.serviceCodes.get(trip.getServiceId());
        newTripTimes.setServiceCode(serviceCode);
        newTripTimes.setRealTimeState(realTimeState);
        if (vehicleDescriptor != null && vehicleDescriptor.hasWheelchairAccessible()) {
            GtfsRealtimeMapper.mapWheelchairAccessible(vehicleDescriptor.getWheelchairAccessible()).ifPresent(newTripTimes::updateWheelchairAccessibility);
        }
        LOG.trace("Trip pattern added with mode {} on {} from {} to {}", new Object[]{trip.getRoute().getMode(), serviceDate, pattern.firstStop().getName(), pattern.lastStop().getName()});
        return this.buffer.update(pattern, newTripTimes, serviceDate);
    }

    private boolean cancelScheduledTrip(FeedScopedId tripId, LocalDate serviceDate, CancelationType cancelationType) {
        boolean success = false;
        TripPattern pattern = this.getPatternForTripId(tripId);
        if (pattern != null) {
            Timetable timetable = pattern.getScheduledTimetable();
            int tripIndex = timetable.getTripIndex(tripId);
            if (tripIndex == -1) {
                TimetableSnapshotSource.debug(tripId, "Could not cancel scheduled trip because it's not in the timetable", new Object[0]);
            } else {
                TripTimes newTripTimes = new TripTimes(timetable.getTripTimes(tripIndex));
                switch (cancelationType) {
                    case CANCEL: {
                        newTripTimes.cancelTrip();
                        break;
                    }
                    case DELETE: {
                        newTripTimes.deleteTrip();
                    }
                }
                this.buffer.update(pattern, newTripTimes, serviceDate);
                success = true;
            }
        }
        return success;
    }

    private boolean cancelPreviouslyAddedTrip(FeedScopedId tripId, LocalDate serviceDate, CancelationType cancelationType) {
        boolean success = false;
        TripPattern pattern = this.buffer.getRealtimeAddedTripPattern(tripId, serviceDate);
        if (pattern != null) {
            Timetable timetable = this.buffer.resolve(pattern, serviceDate);
            int tripIndex = timetable.getTripIndex(tripId);
            if (tripIndex == -1) {
                TimetableSnapshotSource.debug(tripId, "Could not cancel previously added trip on {}", serviceDate);
            } else {
                TripTimes newTripTimes = new TripTimes(timetable.getTripTimes(tripIndex));
                switch (cancelationType) {
                    case CANCEL: {
                        newTripTimes.cancelTrip();
                        break;
                    }
                    case DELETE: {
                        newTripTimes.deleteTrip();
                    }
                }
                this.buffer.update(pattern, newTripTimes, serviceDate);
                success = true;
            }
        }
        return success;
    }

    private Result<UpdateSuccess, UpdateError> validateAndHandleModifiedTrip(GtfsRealtime.TripUpdate tripUpdate, GtfsRealtime.TripDescriptor tripDescriptor, FeedScopedId tripId, LocalDate serviceDate) {
        Objects.requireNonNull(tripUpdate);
        Objects.requireNonNull(serviceDate);
        Trip trip = this.transitService.getTripForId(tripId);
        if (trip == null) {
            TimetableSnapshotSource.debug(tripId, "Feed does not contain trip id of MODIFIED trip, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TRIP_NOT_FOUND);
        }
        if (!tripDescriptor.hasStartDate()) {
            TimetableSnapshotSource.debug(tripId, "REPLACEMENT trip doesn't have a start date in TripDescriptor, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_START_DATE);
        }
        Set<FeedScopedId> serviceIds = this.transitService.getCalendarService().getServiceIdsOnDate(serviceDate);
        if (!serviceIds.contains(trip.getServiceId())) {
            TimetableSnapshotSource.debug(tripId, "REPLACEMENT trip has a service date that is not served by trip, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_SERVICE_ON_DATE);
        }
        if (tripUpdate.getStopTimeUpdateCount() < 2) {
            TimetableSnapshotSource.debug(tripId, "REPLACEMENT trip has less then two stops, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TOO_FEW_STOPS);
        }
        List<StopLocation> stops = this.checkNewStopTimeUpdatesAndFindStops(tripId, tripUpdate.getStopTimeUpdateList());
        if (stops == null) {
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_VALID_STOPS);
        }
        return this.handleModifiedTrip(trip, tripUpdate, stops, serviceDate);
    }

    private Result<UpdateSuccess, UpdateError> handleModifiedTrip(Trip trip, GtfsRealtime.TripUpdate tripUpdate, List<StopLocation> stops, LocalDate serviceDate) {
        Objects.requireNonNull(stops);
        Preconditions.checkArgument((tripUpdate.getStopTimeUpdateCount() == stops.size() ? 1 : 0) != 0, (Object)"number of stop should match the number of stop time updates");
        FeedScopedId tripId = trip.getId();
        this.cancelScheduledTrip(tripId, serviceDate, CancelationType.DELETE);
        this.cancelPreviouslyAddedTrip(tripId, serviceDate, CancelationType.DELETE);
        return this.addTripToGraphAndBuffer(trip, tripUpdate.getVehicle(), tripUpdate.getStopTimeUpdateList(), stops, serviceDate, RealTimeState.MODIFIED);
    }

    private Result<UpdateSuccess, UpdateError> handleCanceledTrip(FeedScopedId tripId, LocalDate serviceDate, CancelationType markAsDeleted) {
        boolean cancelScheduledSuccess = this.cancelScheduledTrip(tripId, serviceDate, markAsDeleted);
        boolean cancelPreviouslyAddedSuccess = this.cancelPreviouslyAddedTrip(tripId, serviceDate, markAsDeleted);
        if (!cancelScheduledSuccess && !cancelPreviouslyAddedSuccess) {
            TimetableSnapshotSource.debug(tripId, "No pattern found for tripId. Skipping cancellation.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_TRIP_FOR_CANCELLATION_FOUND);
        }
        return Result.success(UpdateSuccess.noWarnings());
    }

    private boolean purgeExpiredData() {
        LocalDate today = this.localDateNow.get();
        LocalDate previously = today.minusDays(2L);
        if (this.lastPurgeDate != null && this.lastPurgeDate.compareTo(previously) >= 0) {
            return false;
        }
        LOG.debug("purging expired realtime data");
        this.lastPurgeDate = previously;
        return this.buffer.purgeExpiredData(previously);
    }

    private TripPattern getPatternForTripId(FeedScopedId tripId) {
        Trip trip = this.transitService.getTripForId(tripId);
        return this.transitService.getPatternForTrip(trip);
    }

    private static void debug(FeedScopedId id, String message, Object ... params) {
        TimetableSnapshotSource.debug(id.getFeedId(), id.getId(), message, params);
    }

    private static void debug(String feedId, String tripId, String message, Object ... params) {
        String m = "[feedId: %s, tripId: %s] %s".formatted(feedId, tripId, message);
        LOG.debug(m, params);
    }

    private static enum CancelationType {
        CANCEL,
        DELETE;

    }
}

