/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.graph_builder.module.ned;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.geotools.geometry.DirectPosition2D;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.impl.PackedCoordinateSequence;
import org.opengis.coverage.Coverage;
import org.opengis.coverage.PointOutsideCoverageException;
import org.opengis.geometry.DirectPosition;
import org.opengis.referencing.operation.TransformException;
import org.opentripplanner.common.geometry.SphericalDistanceLibrary;
import org.opentripplanner.graph_builder.DataImportIssueStore;
import org.opentripplanner.graph_builder.issues.ElevationFlattened;
import org.opentripplanner.graph_builder.issues.ElevationProfileFailure;
import org.opentripplanner.graph_builder.issues.Graphwide;
import org.opentripplanner.graph_builder.model.GraphBuilderModule;
import org.opentripplanner.graph_builder.module.ned.MissingElevationHandler;
import org.opentripplanner.graph_builder.services.ned.ElevationGridCoverageFactory;
import org.opentripplanner.routing.edgetype.StreetEdge;
import org.opentripplanner.routing.edgetype.StreetElevationExtension;
import org.opentripplanner.routing.graph.Edge;
import org.opentripplanner.routing.graph.Graph;
import org.opentripplanner.routing.graph.Vertex;
import org.opentripplanner.util.ElevationUtils;
import org.opentripplanner.util.PolylineEncoder;
import org.opentripplanner.util.geometry.GeometryUtils;
import org.opentripplanner.util.logging.ProgressTracker;
import org.opentripplanner.util.time.DurationUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElevationModule
implements GraphBuilderModule {
    private static final Logger LOG = LoggerFactory.getLogger(ElevationModule.class);
    private final ElevationGridCoverageFactory gridCoverageFactory;
    private final boolean readCachedElevations;
    private final boolean writeCachedElevations;
    private final Graph graph;
    private final File cachedElevationsFile;
    private final double maxElevationPropagationMeters;
    private final boolean includeEllipsoidToGeoidDifference;
    private final boolean multiThreadElevationCalculations;
    private final AtomicInteger nPointsEvaluated = new AtomicInteger(0);
    private final AtomicInteger nPointsOutsideDEM = new AtomicInteger(0);
    private final double distanceBetweenSamplesM;
    private final ConcurrentHashMap<Integer, Double> geoidDifferenceCache = new ConcurrentHashMap();
    private final ThreadLocal<Coverage> coverageInterpolatorThreadLocal = new ThreadLocal();
    private final DataImportIssueStore issueStore;
    private HashMap<String, PackedCoordinateSequence> cachedElevations;
    private Coordinate examplarCoordinate;
    private Coverage singleThreadedCoverageInterpolator;
    private double minElevation = Double.MAX_VALUE;
    private double maxElevation = Double.MIN_VALUE;
    private final Map<Vertex, Double> elevationData;

    public ElevationModule(ElevationGridCoverageFactory factory, Graph graph) {
        this(factory, graph, DataImportIssueStore.noopIssueStore(), null, new HashMap<Vertex, Double>(), false, false, 10.0, 2000.0, true, false);
    }

    public ElevationModule(ElevationGridCoverageFactory factory, Graph graph, DataImportIssueStore issueStore, File cachedElevationsFile, Map<Vertex, Double> elevationData, boolean readCachedElevations, boolean writeCachedElevations, double distanceBetweenSamplesM, double maxElevationPropagationMeters, boolean includeEllipsoidToGeoidDifference, boolean multiThreadElevationCalculations) {
        this.gridCoverageFactory = factory;
        this.graph = graph;
        this.issueStore = issueStore;
        this.cachedElevationsFile = cachedElevationsFile;
        this.elevationData = elevationData;
        this.readCachedElevations = readCachedElevations;
        this.writeCachedElevations = writeCachedElevations;
        this.maxElevationPropagationMeters = maxElevationPropagationMeters;
        this.includeEllipsoidToGeoidDifference = includeEllipsoidToGeoidDifference;
        this.multiThreadElevationCalculations = multiThreadElevationCalculations;
        this.distanceBetweenSamplesM = distanceBetweenSamplesM;
    }

    @Override
    public void buildGraph() {
        double d;
        Instant start = Instant.now();
        this.gridCoverageFactory.fetchData(this.graph);
        this.graph.setDistanceBetweenElevationSamples(this.distanceBetweenSamplesM);
        if (this.readCachedElevations) {
            try {
                ObjectInputStream in = new ObjectInputStream(new FileInputStream(this.cachedElevationsFile));
                this.cachedElevations = (HashMap)in.readObject();
                LOG.info("Cached elevation data loaded into memory!");
            }
            catch (IOException | ClassNotFoundException e) {
                this.issueStore.add(new Graphwide(String.format("Cached elevations file could not be read in due to error: %s!", e.getMessage())));
            }
        }
        LOG.info("Setting street elevation profiles from digital elevation model...");
        LinkedList<StreetEdge> streetsWithElevationEdges = new LinkedList<StreetEdge>();
        for (Vertex gv : this.graph.getVertices()) {
            for (Edge edge : gv.getOutgoing()) {
                if (!(edge instanceof StreetEdge)) continue;
                if (this.multiThreadElevationCalculations && this.examplarCoordinate == null) {
                    this.examplarCoordinate = edge.getGeometry().getCoordinates()[0];
                }
                streetsWithElevationEdges.add((StreetEdge)edge);
            }
        }
        int totalElevationEdges = streetsWithElevationEdges.size();
        ProgressTracker progress = ProgressTracker.track("Set elevation", 25000, totalElevationEdges);
        if (this.multiThreadElevationCalculations) {
            streetsWithElevationEdges.parallelStream().forEach(ee -> this.processEdgeWithProgress((StreetEdge)ee, progress));
        } else {
            for (StreetEdge streetEdge : streetsWithElevationEdges) {
                this.processEdgeWithProgress(streetEdge, progress);
            }
        }
        int nPoints = this.nPointsEvaluated.get() + this.nPointsOutsideDEM.get();
        if (nPoints > 0 && (d = (double)this.nPointsOutsideDEM.get() / (double)nPoints * 100.0) > 50.0) {
            this.issueStore.add(new Graphwide(String.format("Fetching elevation failed at %d/%d points (%.1f%%)", this.nPointsOutsideDEM.get(), nPoints, d)));
            LOG.warn("Elevation is missing at a large number of points. DEM may be for the wrong region. If it is unprojected, perhaps the axes are not in (longitude, latitude) order.");
        }
        LOG.info(progress.completeMessage());
        LinkedList<StreetEdge> linkedList = new LinkedList<StreetEdge>();
        for (StreetEdge streetEdge : streetsWithElevationEdges) {
            if (!streetEdge.hasElevationExtension() || streetEdge.isElevationFlattened()) continue;
            linkedList.add(streetEdge);
        }
        if (this.writeCachedElevations) {
            LOG.info("Writing elevation cache");
            HashMap<String, PackedCoordinateSequence> newCachedElevations = new HashMap<String, PackedCoordinateSequence>();
            for (StreetEdge streetEdge : linkedList) {
                newCachedElevations.put(PolylineEncoder.encodeGeometry((Geometry)streetEdge.getGeometry()).points(), streetEdge.getElevationProfile());
            }
            try {
                ObjectOutputStream objectOutputStream = new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(this.cachedElevationsFile)));
                objectOutputStream.writeObject(newCachedElevations);
                objectOutputStream.close();
            }
            catch (IOException iOException) {
                this.issueStore.add(new Graphwide("Failed to write cached elevation file: " + iOException.getMessage()));
            }
        }
        Map<Vertex, Double> elevationsForVertices = this.collectKnownElevationsForVertices(this.elevationData, linkedList);
        new MissingElevationHandler(this.issueStore, elevationsForVertices, this.maxElevationPropagationMeters).run();
        this.updateElevationMetadata(this.graph);
        LOG.info("Finished elevation processing in {}", (Object)DurationUtils.durationToStr(Duration.between(start, Instant.now())));
    }

    @Override
    public void checkInputs() {
        this.gridCoverageFactory.checkInputs();
        if (this.readCachedElevations) {
            if (Files.exists(this.cachedElevationsFile.toPath(), new LinkOption[0])) {
                LOG.info("Cached elevations file found!");
            } else {
                LOG.warn("No cached elevations file found at {} or read access not allowed! Unable to load in cached elevations. This could take a while...", (Object)this.cachedElevationsFile.toPath().toAbsolutePath());
            }
        } else {
            LOG.warn("Not using cached elevations! This could take a while...");
        }
    }

    private void updateElevationMetadata(Graph graph) {
        if (this.nPointsOutsideDEM.get() < this.nPointsEvaluated.get()) {
            graph.hasElevation = true;
            graph.minElevation = this.minElevation;
            graph.maxElevation = this.maxElevation;
        }
    }

    private Map<Vertex, Double> collectKnownElevationsForVertices(Map<Vertex, Double> knownElevations, List<StreetEdge> edgesWithElevation) {
        HashMap<Vertex, Double> elevations;
        HashMap<Vertex, Double> hashMap = elevations = knownElevations != null ? new HashMap<Vertex, Double>(knownElevations) : new HashMap();
        if (this.includeEllipsoidToGeoidDifference) {
            elevations.forEach((vertex, elevation) -> {
                try {
                    elevations.put((Vertex)vertex, elevation - this.getApproximateEllipsoidToGeoidDifference(vertex.getY(), vertex.getX()));
                }
                catch (TransformException e) {
                    LOG.error("Error processing elevation for known elevation at vertex: {} due to error: {}", vertex, (Object)e);
                }
            });
        }
        for (StreetEdge e : edgesWithElevation) {
            PackedCoordinateSequence profile = e.getElevationProfile();
            if (!elevations.containsKey(e.getFromVertex())) {
                double firstElevation = profile.getOrdinate(0, 1);
                elevations.put(e.getFromVertex(), firstElevation);
            }
            if (elevations.containsKey(e.getToVertex())) continue;
            double lastElevation = profile.getOrdinate(profile.size() - 1, 1);
            elevations.put(e.getToVertex(), lastElevation);
        }
        return elevations;
    }

    private void processEdgeWithProgress(StreetEdge ee, ProgressTracker progress) {
        this.processEdge(ee);
        progress.step(m -> LOG.info(m));
    }

    private void processEdge(StreetEdge ee) {
        PackedCoordinateSequence coordinateSequence;
        if (ee.hasElevationExtension()) {
            return;
        }
        LineString edgeGeometry = ee.getGeometry();
        if (this.cachedElevations != null && (coordinateSequence = this.cachedElevations.get(PolylineEncoder.encodeGeometry((Geometry)edgeGeometry).points())) != null) {
            this.setEdgeElevationProfile(ee, coordinateSequence);
            return;
        }
        Coverage coverage = this.getThreadSpecificCoverageInterpolator();
        try {
            Coordinate[] coords = edgeGeometry.getCoordinates();
            LinkedList<Coordinate> coordList = new LinkedList<Coordinate>();
            coordList.add(new Coordinate(0.0, this.getElevation(coverage, coords[0])));
            double edgeLenM = 0.0;
            double sampleDistance = this.distanceBetweenSamplesM;
            double previousDistance = 0.0;
            double x1 = coords[0].x;
            double y1 = coords[0].y;
            for (int i = 0; i < coords.length - 1; ++i) {
                double x2 = coords[i + 1].x;
                double y2 = coords[i + 1].y;
                double curSegmentDistance = SphericalDistanceLibrary.distance(y1, x1, y2, x2);
                edgeLenM += curSegmentDistance;
                while (edgeLenM > sampleDistance) {
                    double pctAlongSeg = (sampleDistance - previousDistance) / curSegmentDistance;
                    coordList.add(new Coordinate(sampleDistance, this.getElevation(coverage, new Coordinate(x1 + pctAlongSeg * (x2 - x1), y1 + pctAlongSeg * (y2 - y1)))));
                    sampleDistance += this.distanceBetweenSamplesM;
                }
                previousDistance = edgeLenM;
                x1 = x2;
                y1 = y2;
            }
            if (edgeLenM - ((Coordinate)coordList.get((int)(coordList.size() - 1))).x < this.distanceBetweenSamplesM / 2.0) {
                coordList.remove(coordList.size() - 1);
            }
            coordList.add(new Coordinate(edgeLenM, this.getElevation(coverage, coords[coords.length - 1])));
            Coordinate[] coordArr = new Coordinate[coordList.size()];
            PackedCoordinateSequence.Double elevPCS = new PackedCoordinateSequence.Double(coordList.toArray(coordArr));
            this.setEdgeElevationProfile(ee, (PackedCoordinateSequence)elevPCS);
        }
        catch (ElevationLookupException e) {
            this.issueStore.add(new ElevationProfileFailure(ee, e.getMessage()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Coverage getThreadSpecificCoverageInterpolator() {
        if (this.multiThreadElevationCalculations) {
            Coverage coverage = this.coverageInterpolatorThreadLocal.get();
            if (coverage == null) {
                ElevationGridCoverageFactory elevationGridCoverageFactory = this.gridCoverageFactory;
                synchronized (elevationGridCoverageFactory) {
                    coverage = this.gridCoverageFactory.getGridCoverage();
                    try {
                        this.getElevation(coverage, this.examplarCoordinate);
                    }
                    catch (ElevationLookupException e) {
                        LOG.warn("Error processing elevation for coordinate: {} due to error: {}", (Object)this.examplarCoordinate, (Object)e);
                    }
                    this.coverageInterpolatorThreadLocal.set(coverage);
                }
            }
            return coverage;
        }
        if (this.singleThreadedCoverageInterpolator == null) {
            this.singleThreadedCoverageInterpolator = this.gridCoverageFactory.getGridCoverage();
        }
        return this.singleThreadedCoverageInterpolator;
    }

    private void setEdgeElevationProfile(StreetEdge ee, PackedCoordinateSequence elevPCS) {
        try {
            StreetElevationExtension.addToEdge(ee, elevPCS, false);
            if (ee.isElevationFlattened()) {
                this.issueStore.add(new ElevationFlattened(ee));
            }
        }
        catch (Exception e) {
            this.issueStore.add(new ElevationProfileFailure(ee, e.getMessage()));
        }
    }

    private double getElevation(Coverage coverage, Coordinate c) throws ElevationLookupException {
        try {
            return this.getElevation(coverage, c.x, c.y);
        }
        catch (ArrayIndexOutOfBoundsException | PointOutsideCoverageException | TransformException e) {
            throw new ElevationLookupException((Exception)e);
        }
    }

    private double getElevation(Coverage coverage, double x, double y) throws PointOutsideCoverageException, TransformException {
        double[] values = new double[1];
        try {
            coverage.evaluate((DirectPosition)new DirectPosition2D(GeometryUtils.WGS84_XY, x, y), values);
        }
        catch (PointOutsideCoverageException e) {
            this.nPointsOutsideDEM.incrementAndGet();
            throw e;
        }
        double elevation = values[0] * this.gridCoverageFactory.elevationUnitMultiplier() - (this.includeEllipsoidToGeoidDifference ? this.getApproximateEllipsoidToGeoidDifference(y, x) : 0.0);
        this.minElevation = Math.min(this.minElevation, elevation);
        this.maxElevation = Math.max(this.maxElevation, elevation);
        this.nPointsEvaluated.incrementAndGet();
        return elevation;
    }

    private double getApproximateEllipsoidToGeoidDifference(double y, double x) throws TransformException {
        int geoidDifferenceCoordinateValueMultiplier = 100;
        int xVal = (int)Math.round(x * (double)geoidDifferenceCoordinateValueMultiplier);
        int yVal = (int)Math.round(y * (double)geoidDifferenceCoordinateValueMultiplier);
        int hash = yVal * 104729 + xVal;
        Double difference = this.geoidDifferenceCache.get(hash);
        if (difference == null) {
            difference = ElevationUtils.computeEllipsoidToGeoidDifference((double)yVal / (1.0 * (double)geoidDifferenceCoordinateValueMultiplier), (double)xVal / (1.0 * (double)geoidDifferenceCoordinateValueMultiplier));
            this.geoidDifferenceCache.put(hash, difference);
        }
        return difference;
    }

    static class ElevationLookupException
    extends Exception {
        public ElevationLookupException(Exception e) {
            super(e);
        }
    }
}

