/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.updater.stoptime;

import com.google.common.base.Preconditions;
import com.google.transit.realtime.GtfsRealtime;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Calendar;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.locks.ReentrantLock;
import org.opentripplanner.model.Agency;
import org.opentripplanner.model.FeedScopedId;
import org.opentripplanner.model.PickDrop;
import org.opentripplanner.model.Route;
import org.opentripplanner.model.StopLocation;
import org.opentripplanner.model.StopPattern;
import org.opentripplanner.model.StopTime;
import org.opentripplanner.model.Timetable;
import org.opentripplanner.model.TimetableSnapshot;
import org.opentripplanner.model.TimetableSnapshotProvider;
import org.opentripplanner.model.TransitMode;
import org.opentripplanner.model.Trip;
import org.opentripplanner.model.TripPattern;
import org.opentripplanner.model.calendar.ServiceDate;
import org.opentripplanner.routing.RoutingService;
import org.opentripplanner.routing.algorithm.raptoradapter.transit.TransitLayer;
import org.opentripplanner.routing.algorithm.raptoradapter.transit.mappers.TransitLayerUpdater;
import org.opentripplanner.routing.graph.Graph;
import org.opentripplanner.routing.trippattern.RealTimeState;
import org.opentripplanner.routing.trippattern.TripTimes;
import org.opentripplanner.updater.GtfsRealtimeFuzzyTripMatcher;
import org.opentripplanner.updater.stoptime.TripPatternCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimetableSnapshotSource
implements TimetableSnapshotProvider {
    private static final Logger LOG = LoggerFactory.getLogger(TimetableSnapshotSource.class);
    private static final int MILLIS_PER_SECOND = 1000;
    private static final long MAX_ARRIVAL_DEPARTURE_TIME = 172800L;
    public int logFrequency = 2000;
    private int appliedBlockCount = 0;
    public int maxSnapshotFrequency = 1000;
    private volatile TimetableSnapshot snapshot = null;
    private final TimetableSnapshot buffer = new TimetableSnapshot();
    private final ReentrantLock bufferLock = new ReentrantLock(true);
    private final TripPatternCache tripPatternCache = new TripPatternCache();
    public boolean purgeExpiredData = true;
    protected ServiceDate lastPurgeDate = null;
    protected long lastSnapshotTime = -1L;
    private final TimeZone timeZone;
    private final RoutingService routingService;
    public GtfsRealtimeFuzzyTripMatcher fuzzyTripMatcher;
    private TransitLayer realtimeTransitLayer;
    private TransitLayerUpdater transitLayerUpdater;

    public TimetableSnapshotSource(Graph graph) {
        this.timeZone = graph.getTimeZone();
        this.routingService = new RoutingService(graph);
        this.realtimeTransitLayer = graph.getRealtimeTransitLayer();
        this.transitLayerUpdater = graph.transitLayerUpdater;
    }

    @Override
    public TimetableSnapshot getTimetableSnapshot() {
        TimetableSnapshot snapshotToReturn;
        if (this.bufferLock.tryLock()) {
            try {
                snapshotToReturn = this.getTimetableSnapshot(false);
            }
            finally {
                this.bufferLock.unlock();
            }
        } else {
            snapshotToReturn = this.snapshot;
        }
        return snapshotToReturn;
    }

    private TimetableSnapshot getTimetableSnapshot(boolean force) {
        long now = System.currentTimeMillis();
        if (force || now - this.lastSnapshotTime > (long)this.maxSnapshotFrequency) {
            if (force || this.buffer.isDirty()) {
                LOG.debug("Committing {}", (Object)this.buffer.toString());
                this.snapshot = this.buffer.commit(this.transitLayerUpdater, force);
            } else {
                LOG.debug("Buffer was unchanged, keeping old snapshot.");
            }
            this.lastSnapshotTime = System.currentTimeMillis();
        } else {
            LOG.debug("Snapshot frequency exceeded. Reusing snapshot {}", (Object)this.snapshot);
        }
        return this.snapshot;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void applyTripUpdates(Graph graph, boolean fullDataset, List<GtfsRealtime.TripUpdate> updates, String feedId) {
        if (updates == null) {
            LOG.warn("updates is null");
            return;
        }
        this.bufferLock.lock();
        try {
            if (fullDataset) {
                this.buffer.clear(feedId);
            }
            LOG.debug("message contains {} trip updates", (Object)updates.size());
            int uIndex = 0;
            for (GtfsRealtime.TripUpdate tripUpdate : updates) {
                if (this.fuzzyTripMatcher != null && tripUpdate.hasTrip()) {
                    GtfsRealtime.TripDescriptor trip = this.fuzzyTripMatcher.match(feedId, tripUpdate.getTrip());
                    tripUpdate = tripUpdate.toBuilder().setTrip(trip).build();
                }
                if (!tripUpdate.hasTrip()) {
                    LOG.warn("Missing TripDescriptor in gtfs-rt trip update: \n{}", (Object)tripUpdate);
                    continue;
                }
                ServiceDate serviceDate = new ServiceDate();
                GtfsRealtime.TripDescriptor tripDescriptor = tripUpdate.getTrip();
                if (tripDescriptor.hasStartDate()) {
                    try {
                        serviceDate = ServiceDate.parseString(tripDescriptor.getStartDate());
                    }
                    catch (ParseException e) {
                        LOG.warn("Failed to parse start date in gtfs-rt trip update: \n{}", (Object)tripUpdate);
                        continue;
                    }
                }
                LOG.debug("trip update #{} ({} updates) :", (Object)(++uIndex), (Object)tripUpdate.getStopTimeUpdateCount());
                LOG.trace("{}", (Object)tripUpdate);
                boolean applied = false;
                GtfsRealtime.TripDescriptor.ScheduleRelationship tripScheduleRelationship = this.determineTripScheduleRelationship(tripUpdate);
                switch (tripScheduleRelationship) {
                    case SCHEDULED: {
                        applied = this.handleScheduledTrip(tripUpdate, feedId, serviceDate);
                        break;
                    }
                    case ADDED: {
                        applied = this.validateAndHandleAddedTrip(graph, tripUpdate, feedId, serviceDate);
                        break;
                    }
                    case UNSCHEDULED: {
                        applied = this.handleUnscheduledTrip(tripUpdate, feedId, serviceDate);
                        break;
                    }
                    case CANCELED: {
                        applied = this.handleCanceledTrip(tripUpdate, feedId, serviceDate);
                        break;
                    }
                    case MODIFIED: {
                        applied = this.validateAndHandleModifiedTrip(graph, tripUpdate, feedId, serviceDate);
                    }
                }
                if (applied) {
                    ++this.appliedBlockCount;
                } else {
                    LOG.warn("Failed to apply TripUpdate.");
                    LOG.trace(" Contents: {}", (Object)tripUpdate);
                }
                if (this.appliedBlockCount % this.logFrequency != 0) continue;
                LOG.info("Applied {} trip updates.", (Object)this.appliedBlockCount);
            }
            LOG.debug("end of update message");
            if (this.purgeExpiredData) {
                boolean modified = this.purgeExpiredData();
                this.getTimetableSnapshot(modified);
            } else {
                this.getTimetableSnapshot(false);
            }
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    private GtfsRealtime.TripDescriptor.ScheduleRelationship determineTripScheduleRelationship(GtfsRealtime.TripUpdate tripUpdate) {
        GtfsRealtime.TripDescriptor.ScheduleRelationship tripScheduleRelationship = GtfsRealtime.TripDescriptor.ScheduleRelationship.SCHEDULED;
        if (tripUpdate.hasTrip() && tripUpdate.getTrip().hasScheduleRelationship()) {
            tripScheduleRelationship = tripUpdate.getTrip().getScheduleRelationship();
        }
        if (tripScheduleRelationship.equals((Object)GtfsRealtime.TripDescriptor.ScheduleRelationship.SCHEDULED)) {
            boolean hasModifiedStops = false;
            for (GtfsRealtime.TripUpdate.StopTimeUpdate stopTimeUpdate : tripUpdate.getStopTimeUpdateList()) {
                GtfsRealtime.TripUpdate.StopTimeUpdate.ScheduleRelationship stopScheduleRelationship;
                if (!stopTimeUpdate.hasScheduleRelationship() || !(stopScheduleRelationship = stopTimeUpdate.getScheduleRelationship()).equals((Object)GtfsRealtime.TripUpdate.StopTimeUpdate.ScheduleRelationship.SKIPPED)) continue;
                hasModifiedStops = true;
            }
            if (hasModifiedStops) {
                tripScheduleRelationship = GtfsRealtime.TripDescriptor.ScheduleRelationship.MODIFIED;
            }
        }
        return tripScheduleRelationship;
    }

    private boolean handleScheduledTrip(GtfsRealtime.TripUpdate tripUpdate, String feedId, ServiceDate serviceDate) {
        GtfsRealtime.TripDescriptor tripDescriptor = tripUpdate.getTrip();
        String tripId = tripDescriptor.getTripId();
        TripPattern pattern = this.getPatternForTripId(feedId, tripId);
        if (pattern == null) {
            LOG.warn("No pattern found for tripId {}, skipping TripUpdate.", (Object)tripId);
            return false;
        }
        if (tripUpdate.getStopTimeUpdateCount() < 1) {
            LOG.warn("TripUpdate contains no updates, skipping.");
            return false;
        }
        this.cancelPreviouslyAddedTrip(new FeedScopedId(feedId, tripId), serviceDate);
        TripTimes updatedTripTimes = pattern.getScheduledTimetable().createUpdatedTripTimes(tripUpdate, this.timeZone, serviceDate);
        if (updatedTripTimes == null) {
            return false;
        }
        updatedTripTimes.setRealTimeState(RealTimeState.UPDATED);
        boolean success = this.buffer.update(pattern, updatedTripTimes, serviceDate);
        return success;
    }

    private boolean validateAndHandleAddedTrip(Graph graph, GtfsRealtime.TripUpdate tripUpdate, String feedId, ServiceDate serviceDate) {
        Preconditions.checkNotNull((Object)graph);
        Preconditions.checkNotNull((Object)tripUpdate);
        Preconditions.checkNotNull((Object)serviceDate);
        GtfsRealtime.TripDescriptor tripDescriptor = tripUpdate.getTrip();
        if (!tripDescriptor.hasTripId()) {
            LOG.warn("No trip id found for ADDED trip, skipping.");
            return false;
        }
        String tripId = tripDescriptor.getTripId();
        Trip trip = this.getTripForTripId(feedId, tripId);
        if (trip != null) {
            LOG.warn("Graph already contains trip id of ADDED trip, skipping.");
            return false;
        }
        if (!tripDescriptor.hasStartDate()) {
            LOG.warn("ADDED trip doesn't have a start date in TripDescriptor, skipping.");
            return false;
        }
        if (tripUpdate.getStopTimeUpdateCount() < 2) {
            LOG.warn("ADDED trip has less then two stops, skipping.");
            return false;
        }
        List<StopLocation> stops = this.checkNewStopTimeUpdatesAndFindStops(feedId, tripUpdate);
        if (stops == null) {
            return false;
        }
        boolean success = this.handleAddedTrip(graph, tripUpdate, stops, feedId, serviceDate);
        return success;
    }

    /*
     * Enabled aggressive block sorting
     */
    private List<StopLocation> checkNewStopTimeUpdatesAndFindStops(String feedId, GtfsRealtime.TripUpdate tripUpdate) {
        Integer previousStopSequence = null;
        Long previousTime = null;
        List<GtfsRealtime.TripUpdate.StopTimeUpdate> stopTimeUpdates = tripUpdate.getStopTimeUpdateList();
        ArrayList<StopLocation> stops = new ArrayList<StopLocation>(stopTimeUpdates.size());
        int index = 0;
        while (index < stopTimeUpdates.size()) {
            GtfsRealtime.TripUpdate.StopTimeUpdate stopTimeUpdate = stopTimeUpdates.get(index);
            boolean skippedStop = this.isStopSkipped(stopTimeUpdate);
            if (stopTimeUpdate.hasStopSequence()) {
                Integer stopSequence = stopTimeUpdate.getStopSequence();
                if (stopSequence < 0) {
                    LOG.warn("Trip update contains negative stop sequence, skipping.");
                    return null;
                }
                if (previousStopSequence != null && previousStopSequence > stopSequence) {
                    LOG.warn("Trip update contains decreasing stop sequence, skipping.");
                    return null;
                }
                previousStopSequence = stopSequence;
            }
            if (!stopTimeUpdate.hasStopId()) {
                LOG.warn("Trip update misses some stop ids, skipping.");
                return null;
            }
            StopLocation stop = this.getStopForStopId(feedId, stopTimeUpdate.getStopId());
            if (stop != null) {
                stops.add(stop);
            } else {
                if (!skippedStop) {
                    LOG.warn("Graph doesn't contain stop id \"{}\" of trip update, skipping.", (Object)stopTimeUpdate.getStopId());
                    return null;
                }
                stops.add(null);
            }
            if (!skippedStop) {
                if (stopTimeUpdate.hasArrival() && stopTimeUpdate.getArrival().hasTime()) {
                    Long time = stopTimeUpdate.getArrival().getTime();
                    if (previousTime != null && previousTime > time) {
                        LOG.warn("Trip update contains decreasing times, skipping.");
                        return null;
                    }
                    previousTime = time;
                } else {
                    for (int earlierIndex = 0; earlierIndex < index; ++earlierIndex) {
                        GtfsRealtime.TripUpdate.StopTimeUpdate earlierStopTimeUpdate = stopTimeUpdates.get(earlierIndex);
                        boolean earlierSkippedStop = this.isStopSkipped(earlierStopTimeUpdate);
                        if (earlierSkippedStop) continue;
                        LOG.warn("Trip update misses arrival time, skipping.");
                        return null;
                    }
                }
                if (stopTimeUpdate.hasDeparture() && stopTimeUpdate.getDeparture().hasTime()) {
                    Long time = stopTimeUpdate.getDeparture().getTime();
                    if (previousTime != null && previousTime > time) {
                        LOG.warn("Trip update contains decreasing times, skipping.");
                        return null;
                    }
                    previousTime = time;
                } else {
                    for (int laterIndex = stopTimeUpdates.size() - 1; laterIndex > index; --laterIndex) {
                        GtfsRealtime.TripUpdate.StopTimeUpdate laterStopTimeUpdate = stopTimeUpdates.get(laterIndex);
                        boolean laterSkippedStop = this.isStopSkipped(laterStopTimeUpdate);
                        if (laterSkippedStop) continue;
                        LOG.warn("Trip update misses departure time, skipping.");
                        return null;
                    }
                }
            }
            ++index;
        }
        return stops;
    }

    private boolean isStopSkipped(GtfsRealtime.TripUpdate.StopTimeUpdate stopTimeUpdate) {
        boolean isSkipped = stopTimeUpdate.hasScheduleRelationship() && stopTimeUpdate.getScheduleRelationship().equals((Object)GtfsRealtime.TripUpdate.StopTimeUpdate.ScheduleRelationship.SKIPPED);
        return isSkipped;
    }

    private boolean handleAddedTrip(Graph graph, GtfsRealtime.TripUpdate tripUpdate, List<StopLocation> stops, String feedId, ServiceDate serviceDate) {
        Preconditions.checkNotNull(stops);
        Preconditions.checkArgument((tripUpdate.getStopTimeUpdateCount() == stops.size() ? 1 : 0) != 0, (Object)"number of stop should match the number of stop time updates");
        String tripId = tripUpdate.getTrip().getTripId();
        this.cancelPreviouslyAddedTrip(new FeedScopedId(feedId, tripId), serviceDate);
        Route route = null;
        if (tripUpdate.getTrip().hasRouteId()) {
            route = this.getRouteForRouteId(feedId, tripUpdate.getTrip().getRouteId());
        }
        if (route == null) {
            FeedScopedId id = tripUpdate.getTrip().hasRouteId() ? new FeedScopedId(feedId, tripUpdate.getTrip().getRouteId()) : new FeedScopedId(feedId, tripId);
            route = new Route(id);
            Agency dummyAgency = new Agency(new FeedScopedId(feedId, "Dummy"), "Dummy", "Europe/Paris");
            route.setAgency(dummyAgency);
            route.setGtfsType(3);
            route.setMode(TransitMode.BUS);
            route.setLongName(tripId);
        }
        Trip trip = new Trip(new FeedScopedId(feedId, tripUpdate.getTrip().getTripId()));
        trip.setRoute(route);
        Set<FeedScopedId> serviceIds = graph.getCalendarService().getServiceIdsOnDate(serviceDate);
        if (serviceIds.isEmpty()) {
            LOG.warn("ADDED trip has service date for which no service id is available, skipping.");
            return false;
        }
        trip.setServiceId(serviceIds.iterator().next());
        boolean success = this.addTripToGraphAndBuffer(graph, trip, tripUpdate, stops, serviceDate, RealTimeState.ADDED);
        return success;
    }

    private boolean addTripToGraphAndBuffer(Graph graph, Trip trip, GtfsRealtime.TripUpdate tripUpdate, List<StopLocation> stops, ServiceDate serviceDate, RealTimeState realTimeState) {
        Preconditions.checkNotNull(stops);
        Preconditions.checkArgument((tripUpdate.getStopTimeUpdateCount() == stops.size() ? 1 : 0) != 0, (Object)"number of stop should match the number of stop time updates");
        Calendar serviceCalendar = serviceDate.getAsCalendar(this.timeZone);
        long midnightSecondsSinceEpoch = serviceCalendar.getTimeInMillis() / 1000L;
        ArrayList<StopTime> stopTimes = new ArrayList<StopTime>(tripUpdate.getStopTimeUpdateCount());
        for (int index = 0; index < tripUpdate.getStopTimeUpdateCount(); ++index) {
            GtfsRealtime.TripUpdate.StopTimeUpdate stopTimeUpdate = tripUpdate.getStopTimeUpdate(index);
            StopLocation stop = stops.get(index);
            boolean skippedStop = this.isStopSkipped(stopTimeUpdate);
            if (skippedStop) continue;
            StopTime stopTime = new StopTime();
            stopTime.setTrip(trip);
            stopTime.setStop(stop);
            if (stopTimeUpdate.hasArrival() && stopTimeUpdate.getArrival().hasTime()) {
                long arrivalTime = stopTimeUpdate.getArrival().getTime() - midnightSecondsSinceEpoch;
                if (arrivalTime < 0L || arrivalTime > 172800L) {
                    LOG.warn("ADDED trip has invalid arrival time (compared to start date in TripDescriptor), skipping.");
                    return false;
                }
                stopTime.setArrivalTime((int)arrivalTime);
            }
            if (stopTimeUpdate.hasDeparture() && stopTimeUpdate.getDeparture().hasTime()) {
                long departureTime = stopTimeUpdate.getDeparture().getTime() - midnightSecondsSinceEpoch;
                if (departureTime < 0L || departureTime > 172800L) {
                    LOG.warn("ADDED trip has invalid departure time (compared to start date in TripDescriptor), skipping.");
                    return false;
                }
                stopTime.setDepartureTime((int)departureTime);
            }
            stopTime.setTimepoint(1);
            if (stopTimeUpdate.hasStopSequence()) {
                stopTime.setStopSequence(stopTimeUpdate.getStopSequence());
            }
            if (index == tripUpdate.getStopTimeUpdateCount() - 1) {
                stopTime.setPickupType(PickDrop.NONE);
            } else {
                stopTime.setPickupType(PickDrop.SCHEDULED);
            }
            if (index == 0) {
                stopTime.setDropOffType(PickDrop.NONE);
            } else {
                stopTime.setDropOffType(PickDrop.SCHEDULED);
            }
            stopTimes.add(stopTime);
        }
        StopPattern stopPattern = new StopPattern(stopTimes);
        TripPattern pattern = this.tripPatternCache.getOrCreateTripPattern(stopPattern, trip, graph);
        int serviceCode = graph.getServiceCodes().get(trip.getServiceId());
        if (!pattern.getServices().get(serviceCode)) {
            BitSet services = (BitSet)pattern.getServices().clone();
            services.set(serviceCode);
            pattern.setServices(services);
        }
        TripTimes newTripTimes = new TripTimes(trip, stopTimes, graph.deduplicator);
        for (int stopIndex = 0; stopIndex < newTripTimes.getNumStops(); ++stopIndex) {
            newTripTimes.updateArrivalTime(stopIndex, newTripTimes.getScheduledArrivalTime(stopIndex));
            newTripTimes.updateDepartureTime(stopIndex, newTripTimes.getScheduledDepartureTime(stopIndex));
        }
        newTripTimes.setServiceCode(serviceCode);
        newTripTimes.setRealTimeState(realTimeState);
        boolean success = this.buffer.update(pattern, newTripTimes, serviceDate);
        return success;
    }

    private boolean cancelScheduledTrip(String feedId, String tripId, ServiceDate serviceDate) {
        boolean success = false;
        TripPattern pattern = this.getPatternForTripId(feedId, tripId);
        if (pattern != null) {
            Timetable timetable = pattern.getScheduledTimetable();
            int tripIndex = timetable.getTripIndex(tripId);
            if (tripIndex == -1) {
                LOG.warn("Could not cancel scheduled trip {}", (Object)tripId);
            } else {
                TripTimes newTripTimes = new TripTimes(timetable.getTripTimes(tripIndex));
                newTripTimes.cancelTrip();
                this.buffer.update(pattern, newTripTimes, serviceDate);
                success = true;
            }
        }
        return success;
    }

    private boolean cancelPreviouslyAddedTrip(FeedScopedId tripId, ServiceDate serviceDate) {
        boolean success = false;
        TripPattern pattern = this.buffer.getLastAddedTripPattern(tripId, serviceDate);
        if (pattern != null) {
            Timetable timetable = this.buffer.resolve(pattern, serviceDate);
            int tripIndex = timetable.getTripIndex(tripId);
            if (tripIndex == -1) {
                LOG.warn("Could not cancel previously added trip {}", (Object)tripId);
            } else {
                TripTimes newTripTimes = new TripTimes(timetable.getTripTimes(tripIndex));
                newTripTimes.cancelTrip();
                this.buffer.update(pattern, newTripTimes, serviceDate);
                success = true;
            }
        }
        return success;
    }

    private boolean handleUnscheduledTrip(GtfsRealtime.TripUpdate tripUpdate, String feedId, ServiceDate serviceDate) {
        LOG.warn("Unscheduled trips are currently unsupported. Skipping TripUpdate.");
        return false;
    }

    private boolean validateAndHandleModifiedTrip(Graph graph, GtfsRealtime.TripUpdate tripUpdate, String feedId, ServiceDate serviceDate) {
        Preconditions.checkNotNull((Object)graph);
        Preconditions.checkNotNull((Object)tripUpdate);
        Preconditions.checkNotNull((Object)serviceDate);
        GtfsRealtime.TripDescriptor tripDescriptor = tripUpdate.getTrip();
        if (!tripDescriptor.hasTripId()) {
            LOG.warn("No trip id found for MODIFIED trip, skipping.");
            return false;
        }
        String tripId = tripDescriptor.getTripId();
        Trip trip = this.getTripForTripId(feedId, tripId);
        if (trip == null) {
            LOG.warn("Graph does not contain trip id of MODIFIED trip, skipping.");
            return false;
        }
        if (!tripDescriptor.hasStartDate()) {
            LOG.warn("MODIFIED trip doesn't have a start date in TripDescriptor, skipping.");
            return false;
        }
        Set<FeedScopedId> serviceIds = graph.getCalendarService().getServiceIdsOnDate(serviceDate);
        if (!serviceIds.contains(trip.getServiceId())) {
            LOG.warn("MODIFIED trip has a service date that is not served by trip, skipping.");
            return false;
        }
        if (tripUpdate.getStopTimeUpdateCount() < 2) {
            LOG.warn("MODIFIED trip has less then two stops, skipping.");
            return false;
        }
        List<StopLocation> stops = this.checkNewStopTimeUpdatesAndFindStops(feedId, tripUpdate);
        if (stops == null) {
            return false;
        }
        boolean success = this.handleModifiedTrip(graph, trip, tripUpdate, stops, feedId, serviceDate);
        return success;
    }

    private boolean handleModifiedTrip(Graph graph, Trip trip, GtfsRealtime.TripUpdate tripUpdate, List<StopLocation> stops, String feedId, ServiceDate serviceDate) {
        Preconditions.checkNotNull(stops);
        Preconditions.checkArgument((tripUpdate.getStopTimeUpdateCount() == stops.size() ? 1 : 0) != 0, (Object)"number of stop should match the number of stop time updates");
        String tripId = tripUpdate.getTrip().getTripId();
        this.cancelScheduledTrip(feedId, tripId, serviceDate);
        this.cancelPreviouslyAddedTrip(new FeedScopedId(feedId, tripId), serviceDate);
        boolean success = this.addTripToGraphAndBuffer(graph, trip, tripUpdate, stops, serviceDate, RealTimeState.MODIFIED);
        return success;
    }

    private boolean handleCanceledTrip(GtfsRealtime.TripUpdate tripUpdate, String feedId, ServiceDate serviceDate) {
        boolean success = false;
        if (tripUpdate.getTrip().hasTripId()) {
            String tripId = tripUpdate.getTrip().getTripId();
            boolean cancelScheduledSuccess = this.cancelScheduledTrip(feedId, tripId, serviceDate);
            boolean cancelPreviouslyAddedSuccess = this.cancelPreviouslyAddedTrip(new FeedScopedId(feedId, tripId), serviceDate);
            if (cancelScheduledSuccess || cancelPreviouslyAddedSuccess) {
                success = true;
            } else {
                LOG.warn("No pattern found for tripId {}, skipping TripUpdate.", (Object)tripId);
            }
        } else {
            LOG.warn("No trip id in CANCELED trip update, skipping TripUpdate.");
        }
        return success;
    }

    private boolean purgeExpiredData() {
        ServiceDate today = new ServiceDate();
        ServiceDate previously = today.previous().previous();
        if (this.lastPurgeDate != null && this.lastPurgeDate.compareTo(previously) >= 0) {
            return false;
        }
        LOG.debug("purging expired realtime data");
        this.lastPurgeDate = previously;
        return this.buffer.purgeExpiredData(previously);
    }

    private TripPattern getPatternForTripId(String feedId, String tripId) {
        Trip trip = this.routingService.getTripForId().get(new FeedScopedId(feedId, tripId));
        TripPattern pattern = this.routingService.getPatternForTrip().get(trip);
        return pattern;
    }

    private Route getRouteForRouteId(String feedId, String routeId) {
        return this.routingService.getRouteForId(new FeedScopedId(feedId, routeId));
    }

    private Trip getTripForTripId(String feedId, String tripId) {
        return this.routingService.getTripForId().get(new FeedScopedId(feedId, tripId));
    }

    private StopLocation getStopForStopId(String feedId, String stopId) {
        return this.routingService.getStopForId(new FeedScopedId(feedId, stopId));
    }
}

