/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.standalone.config;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.MissingNode;
import com.fasterxml.jackson.databind.node.TextNode;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.commons.io.IOUtils;
import org.opentripplanner.standalone.config.BuildConfig;
import org.opentripplanner.standalone.config.EnvironmentVariableReplacer;
import org.opentripplanner.standalone.config.IncludeFileDirective;
import org.opentripplanner.standalone.config.OtpConfig;
import org.opentripplanner.standalone.config.RouterConfig;
import org.opentripplanner.util.OtpAppException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigLoader {
    private static final Logger LOG = LoggerFactory.getLogger(ConfigLoader.class);
    private static final String OTP_CONFIG_FILENAME = "otp-config.json";
    private static final String BUILD_CONFIG_FILENAME = "build-config.json";
    private static final String ROUTER_CONFIG_FILENAME = "router-config.json";
    private static final Set<String> REDACT_KEYS = Set.of("secretKey", "accessKey", "gsCredentials");
    private final ObjectMapper mapper = new ObjectMapper();
    @Nullable
    private final File configDir;
    private final String jsonFallback;

    private ConfigLoader(File configDir, String jsonFallback) {
        this.configDir = configDir;
        this.jsonFallback = jsonFallback;
        this.assertConfigDirIsADirectory();
        this.mapper.configure(JsonParser.Feature.ALLOW_COMMENTS, true);
        this.mapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
    }

    public ConfigLoader(File configDir) {
        this(configDir, null);
    }

    public static ConfigLoader fromString(String json) {
        return new ConfigLoader(null, json);
    }

    public static JsonNode nodeFromString(String json, String source) {
        return new ConfigLoader(null, null).stringToJsonNode(json, source);
    }

    public static boolean isConfigFile(String filename) {
        return OTP_CONFIG_FILENAME.equals(filename) || BUILD_CONFIG_FILENAME.equals(filename) || ROUTER_CONFIG_FILENAME.equals(filename);
    }

    public OtpConfig loadOtpConfig() {
        return new OtpConfig(this.loadJsonByFilename(OTP_CONFIG_FILENAME), OTP_CONFIG_FILENAME, true);
    }

    public BuildConfig loadBuildConfig() {
        JsonNode node = this.loadJsonByFilename(BUILD_CONFIG_FILENAME);
        if (node.isMissingNode()) {
            return BuildConfig.DEFAULT;
        }
        return new BuildConfig(node, BUILD_CONFIG_FILENAME, true);
    }

    public RouterConfig loadRouterConfig() {
        JsonNode node = this.loadJsonByFilename(ROUTER_CONFIG_FILENAME);
        if (node.isMissingNode()) {
            return RouterConfig.DEFAULT;
        }
        return new RouterConfig(node, ROUTER_CONFIG_FILENAME, true);
    }

    public static void logConfigVersion(String otpConfigVersion, String buildConfigVersion, String routerConfigVersion) {
        ConfigLoader.logConfigVersion(otpConfigVersion, OTP_CONFIG_FILENAME);
        ConfigLoader.logConfigVersion(buildConfigVersion, BUILD_CONFIG_FILENAME);
        ConfigLoader.logConfigVersion(routerConfigVersion, ROUTER_CONFIG_FILENAME);
    }

    public JsonNode loadJsonByFilename(String filename) {
        if (this.configDir == null) {
            if (this.jsonFallback != null) {
                return this.stringToJsonNode(this.jsonFallback, filename);
            }
            LOG.warn("Config '{}' not loaded, using defaults. Config directory not set.", (Object)BUILD_CONFIG_FILENAME);
            return MissingNode.getInstance();
        }
        return this.loadJsonFile(new File(this.configDir, filename));
    }

    private JsonNode loadJsonFile(File file) {
        try {
            String configString = IOUtils.toString((InputStream)new FileInputStream(file), (Charset)StandardCharsets.UTF_8);
            JsonNode node = this.stringToJsonNode(configString, file.toString());
            LOG.info("Load JSON configuration file '{}'", (Object)file.getPath());
            LOG.info("Summarizing '{}': {}", (Object)file.getPath(), (Object)ConfigLoader.toRedactedString(node));
            return node;
        }
        catch (FileNotFoundException ex) {
            LOG.info("File '{}' is not present. Using default configuration.", (Object)file);
            return MissingNode.getInstance();
        }
        catch (IOException e) {
            LOG.error("Error while parsing JSON config file '{}': {}", (Object)file, (Object)e.getMessage());
            throw new RuntimeException(e.getLocalizedMessage(), e);
        }
    }

    private JsonNode stringToJsonNode(String jsonAsString, String source) {
        try {
            if (jsonAsString == null || jsonAsString.isBlank()) {
                return MissingNode.getInstance();
            }
            jsonAsString = IncludeFileDirective.includeFileDirective(this.configDir, jsonAsString, source);
            jsonAsString = EnvironmentVariableReplacer.insertEnvironmentVariables(jsonAsString, source);
            return this.mapper.readTree(jsonAsString);
        }
        catch (IOException ie) {
            LOG.error("Error while parsing config '{}'.", (Object)source, (Object)ie);
            throw new OtpAppException("Failed to load config: " + source);
        }
    }

    private void assertConfigDirIsADirectory() {
        if (this.configDir == null) {
            return;
        }
        if (!this.configDir.isDirectory()) {
            throw new IllegalArgumentException(this.configDir + " is not a readable configuration directory.");
        }
    }

    private static String toRedactedString(JsonNode node) {
        JsonNode redactedNode = node.deepCopy();
        ConfigLoader.redactSecretsRecursive(redactedNode);
        return redactedNode.toPrettyString();
    }

    private static void redactSecretsRecursive(JsonNode node) {
        if (node.isObject()) {
            node.fields().forEachRemaining(entry -> {
                if (((JsonNode)entry.getValue()).isObject()) {
                    ConfigLoader.redactSecretsRecursive((JsonNode)entry.getValue());
                } else if (REDACT_KEYS.contains(entry.getKey())) {
                    entry.setValue(new TextNode("********"));
                }
            });
        }
    }

    private static void logConfigVersion(String configVersion, String filename) {
        if (configVersion != null) {
            LOG.info("{} config-version is {}.", (Object)filename, (Object)configVersion);
        }
    }
}

