/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.snapshot;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpSerializable;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: snapshot.create.Response

@JsonpDeserializable
public class CreateSnapshotResponse implements JsonpSerializable {
    @Nullable
    private final Boolean accepted;

    private final SnapshotInfo snapshot;

    // ---------------------------------------------------------------------------------------------

    private CreateSnapshotResponse(Builder builder) {

        this.accepted = builder.accepted;
        this.snapshot = ApiTypeHelper.requireNonNull(builder.snapshot, this, "snapshot");

    }

    public static CreateSnapshotResponse of(Function<Builder, ObjectBuilder<CreateSnapshotResponse>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * API name: {@code accepted}
     */
    @Nullable
    public final Boolean accepted() {
        return this.accepted;
    }

    /**
     * Required - API name: {@code snapshot}
     */
    public final SnapshotInfo snapshot() {
        return this.snapshot;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        if (this.accepted != null) {
            generator.writeKey("accepted");
            generator.write(this.accepted);

        }
        generator.writeKey("snapshot");
        this.snapshot.serialize(generator, mapper);

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link CreateSnapshotResponse}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<CreateSnapshotResponse> {
        @Nullable
        private Boolean accepted;

        private SnapshotInfo snapshot;

        /**
         * API name: {@code accepted}
         */
        public final Builder accepted(@Nullable Boolean value) {
            this.accepted = value;
            return this;
        }

        /**
         * Required - API name: {@code snapshot}
         */
        public final Builder snapshot(SnapshotInfo value) {
            this.snapshot = value;
            return this;
        }

        /**
         * Required - API name: {@code snapshot}
         */
        public final Builder snapshot(Function<SnapshotInfo.Builder, ObjectBuilder<SnapshotInfo>> fn) {
            return this.snapshot(fn.apply(new SnapshotInfo.Builder()).build());
        }

        /**
         * Builds a {@link CreateSnapshotResponse}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public CreateSnapshotResponse build() {
            _checkSingleUse();

            return new CreateSnapshotResponse(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link CreateSnapshotResponse}
     */
    public static final JsonpDeserializer<CreateSnapshotResponse> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        CreateSnapshotResponse::setupCreateSnapshotResponseDeserializer
    );

    protected static void setupCreateSnapshotResponseDeserializer(ObjectDeserializer<CreateSnapshotResponse.Builder> op) {

        op.add(Builder::accepted, JsonpDeserializer.booleanDeserializer(), "accepted");
        op.add(Builder::snapshot, SnapshotInfo._DESERIALIZER, "snapshot");

    }

}
