/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.shutdown;

import java.util.Collections;
import java.util.function.Function;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: shutdown.put_node.Request

/**
 * Adds a node to be shut down. Designed for indirect use by ECE/ESS and ECK.
 * Direct use is not supported.
 *
 */

public class PutNodeRequest extends RequestBase {
    private final String nodeId;

    // ---------------------------------------------------------------------------------------------

    private PutNodeRequest(Builder builder) {

        this.nodeId = ApiTypeHelper.requireNonNull(builder.nodeId, this, "nodeId");

    }

    public static PutNodeRequest of(Function<Builder, ObjectBuilder<PutNodeRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Required - The node id of node to be shut down
     * <p>
     * API name: {@code node_id}
     */
    public final String nodeId() {
        return this.nodeId;
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link PutNodeRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<PutNodeRequest> {
        private String nodeId;

        /**
         * Required - The node id of node to be shut down
         * <p>
         * API name: {@code node_id}
         */
        public final Builder nodeId(String value) {
            this.nodeId = value;
            return this;
        }

        /**
         * Builds a {@link PutNodeRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public PutNodeRequest build() {
            _checkSingleUse();

            return new PutNodeRequest(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code shutdown.put_node}".
     */
    public static final Endpoint<PutNodeRequest, PutNodeResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "PUT";

        },

        // Request path
        request -> {
            final int _nodeId = 1 << 0;

            int propsSet = 0;

            propsSet |= _nodeId;

            if (propsSet == (_nodeId)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_nodes");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.nodeId, buf);
                buf.append("/shutdown");
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            return Collections.emptyMap();

        },
        SimpleEndpoint.emptyMap(),
        false,
        PutNodeResponse._DESERIALIZER
    );
}
