/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.nodes;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: nodes.info.Request

/**
 * Returns information about nodes in the cluster.
 *
 */

public class NodesInfoRequest extends RequestBase {
    @Nullable
    private final Boolean flatSettings;

    @Deprecated
    @Nullable
    private final Time masterTimeout;

    @Nullable
    private final Time clusterManagerTimeout;

    private final List<String> metric;

    private final List<String> nodeId;

    @Nullable
    private final Time timeout;

    // ---------------------------------------------------------------------------------------------

    private NodesInfoRequest(Builder builder) {

        this.flatSettings = builder.flatSettings;
        this.masterTimeout = builder.masterTimeout;
        this.clusterManagerTimeout = builder.clusterManagerTimeout;
        this.metric = ApiTypeHelper.unmodifiable(builder.metric);
        this.nodeId = ApiTypeHelper.unmodifiable(builder.nodeId);
        this.timeout = builder.timeout;

    }

    public static NodesInfoRequest of(Function<Builder, ObjectBuilder<NodesInfoRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * If true, returns settings in flat format.
     * <p>
     * API name: {@code flat_settings}
     */
    @Nullable
    public final Boolean flatSettings() {
        return this.flatSettings;
    }

    /**
     * Period to wait for a connection to the master node. If no response is
     * received before the timeout expires, the request fails and returns an error.
     * <p>
     * API name: {@code master_timeout}
     */
    @Deprecated
    @Nullable
    public final Time masterTimeout() {
        return this.masterTimeout;
    }

    /**
     * Period to wait for a connection to the cluster-manager node. If no response is
     * received before the timeout expires, the request fails and returns an error.
     * <p>
     * API name: {@code cluster_manager_timeout}
     */
    @Nullable
    public final Time clusterManagerTimeout() {
        return this.clusterManagerTimeout;
    }

    /**
     * Limits the information returned to the specific metrics. Supports a
     * comma-separated list, such as http,ingest.
     * <p>
     * API name: {@code metric}
     */
    public final List<String> metric() {
        return this.metric;
    }

    /**
     * Comma-separated list of node IDs or names used to limit returned information.
     * <p>
     * API name: {@code node_id}
     */
    public final List<String> nodeId() {
        return this.nodeId;
    }

    /**
     * Period to wait for a response. If no response is received before the timeout
     * expires, the request fails and returns an error.
     * <p>
     * API name: {@code timeout}
     */
    @Nullable
    public final Time timeout() {
        return this.timeout;
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link NodesInfoRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<NodesInfoRequest> {
        @Nullable
        private Boolean flatSettings;

        @Deprecated
        @Nullable
        private Time masterTimeout;

        @Nullable
        private Time clusterManagerTimeout;

        @Nullable
        private List<String> metric;

        @Nullable
        private List<String> nodeId;

        @Nullable
        private Time timeout;

        /**
         * If true, returns settings in flat format.
         * <p>
         * API name: {@code flat_settings}
         */
        public final Builder flatSettings(@Nullable Boolean value) {
            this.flatSettings = value;
            return this;
        }

        /**
         * Period to wait for a connection to the master node. If no response is
         * received before the timeout expires, the request fails and returns an error.
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(@Nullable Time value) {
            this.masterTimeout = value;
            return this;
        }

        /**
         * Period to wait for a connection to the master node. If no response is
         * received before the timeout expires, the request fails and returns an error.
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.masterTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Period to wait for a connection to the cluster-manager node. If no response is
         * received before the timeout expires, the request fails and returns an error.
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(@Nullable Time value) {
            this.clusterManagerTimeout = value;
            return this;
        }

        /**
         * Period to wait for a connection to the cluster-manager node. If no response is
         * received before the timeout expires, the request fails and returns an error.
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.clusterManagerTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Limits the information returned to the specific metrics. Supports a
         * comma-separated list, such as http,ingest.
         * <p>
         * API name: {@code metric}
         * <p>
         * Adds all elements of <code>list</code> to <code>metric</code>.
         */
        public final Builder metric(List<String> list) {
            this.metric = _listAddAll(this.metric, list);
            return this;
        }

        /**
         * Limits the information returned to the specific metrics. Supports a
         * comma-separated list, such as http,ingest.
         * <p>
         * API name: {@code metric}
         * <p>
         * Adds one or more values to <code>metric</code>.
         */
        public final Builder metric(String value, String... values) {
            this.metric = _listAdd(this.metric, value, values);
            return this;
        }

        /**
         * Comma-separated list of node IDs or names used to limit returned information.
         * <p>
         * API name: {@code node_id}
         * <p>
         * Adds all elements of <code>list</code> to <code>nodeId</code>.
         */
        public final Builder nodeId(List<String> list) {
            this.nodeId = _listAddAll(this.nodeId, list);
            return this;
        }

        /**
         * Comma-separated list of node IDs or names used to limit returned information.
         * <p>
         * API name: {@code node_id}
         * <p>
         * Adds one or more values to <code>nodeId</code>.
         */
        public final Builder nodeId(String value, String... values) {
            this.nodeId = _listAdd(this.nodeId, value, values);
            return this;
        }

        /**
         * Period to wait for a response. If no response is received before the timeout
         * expires, the request fails and returns an error.
         * <p>
         * API name: {@code timeout}
         */
        public final Builder timeout(@Nullable Time value) {
            this.timeout = value;
            return this;
        }

        /**
         * Period to wait for a response. If no response is received before the timeout
         * expires, the request fails and returns an error.
         * <p>
         * API name: {@code timeout}
         */
        public final Builder timeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.timeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Builds a {@link NodesInfoRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public NodesInfoRequest build() {
            _checkSingleUse();

            return new NodesInfoRequest(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code nodes.info}".
     */
    public static final Endpoint<NodesInfoRequest, NodesInfoResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "GET";

        },

        // Request path
        request -> {
            final int _metric = 1 << 0;
            final int _nodeId = 1 << 1;

            int propsSet = 0;

            if (ApiTypeHelper.isDefined(request.metric())) propsSet |= _metric;
            if (ApiTypeHelper.isDefined(request.nodeId())) propsSet |= _nodeId;

            if (propsSet == 0) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_nodes");
                return buf.toString();
            }
            if (propsSet == (_nodeId)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_nodes");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.nodeId.stream().map(v -> v).collect(Collectors.joining(",")), buf);
                return buf.toString();
            }
            if (propsSet == (_metric)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_nodes");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.metric.stream().map(v -> v).collect(Collectors.joining(",")), buf);
                return buf.toString();
            }
            if (propsSet == (_nodeId | _metric)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_nodes");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.nodeId.stream().map(v -> v).collect(Collectors.joining(",")), buf);
                buf.append("/");
                SimpleEndpoint.pathEncode(request.metric.stream().map(v -> v).collect(Collectors.joining(",")), buf);
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            Map<String, String> params = new HashMap<>();
            if (request.masterTimeout != null) {
                params.put("master_timeout", request.masterTimeout._toJsonString());
            }
            if (request.clusterManagerTimeout != null) {
                params.put("cluster_manager_timeout", request.clusterManagerTimeout._toJsonString());
            }
            if (request.flatSettings != null) {
                params.put("flat_settings", String.valueOf(request.flatSettings));
            }
            if (request.timeout != null) {
                params.put("timeout", request.timeout._toJsonString());
            }
            return params;

        },
        SimpleEndpoint.emptyMap(),
        false,
        NodesInfoResponse._DESERIALIZER
    );
}
