/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.indices.get_index_template;

import jakarta.json.stream.JsonGenerator;
import java.util.Map;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonData;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.json.PlainJsonSerializable;
import org.opensearch.client.opensearch._types.mapping.TypeMapping;
import org.opensearch.client.opensearch.indices.Alias;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: indices.get_index_template.IndexTemplateSummary

@JsonpDeserializable
public class IndexTemplateSummary implements PlainJsonSerializable {
    private final Map<String, Alias> aliases;

    @Nullable
    private final TypeMapping mappings;

    private final Map<String, JsonData> settings;

    // ---------------------------------------------------------------------------------------------

    private IndexTemplateSummary(Builder builder) {

        this.aliases = ApiTypeHelper.unmodifiable(builder.aliases);
        this.mappings = builder.mappings;
        this.settings = ApiTypeHelper.unmodifiable(builder.settings);

    }

    public static IndexTemplateSummary of(Function<Builder, ObjectBuilder<IndexTemplateSummary>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * API name: {@code aliases}
     */
    public final Map<String, Alias> aliases() {
        return this.aliases;
    }

    /**
     * API name: {@code mappings}
     */
    @Nullable
    public final TypeMapping mappings() {
        return this.mappings;
    }

    /**
     * API name: {@code settings}
     */
    public final Map<String, JsonData> settings() {
        return this.settings;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        if (ApiTypeHelper.isDefined(this.aliases)) {
            generator.writeKey("aliases");
            generator.writeStartObject();
            for (Map.Entry<String, Alias> item0 : this.aliases.entrySet()) {
                generator.writeKey(item0.getKey());
                item0.getValue().serialize(generator, mapper);

            }
            generator.writeEnd();

        }
        if (this.mappings != null) {
            generator.writeKey("mappings");
            this.mappings.serialize(generator, mapper);

        }
        if (ApiTypeHelper.isDefined(this.settings)) {
            generator.writeKey("settings");
            generator.writeStartObject();
            for (Map.Entry<String, JsonData> item0 : this.settings.entrySet()) {
                generator.writeKey(item0.getKey());
                item0.getValue().serialize(generator, mapper);

            }
            generator.writeEnd();

        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link IndexTemplateSummary}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<IndexTemplateSummary> {
        @Nullable
        private Map<String, Alias> aliases;

        @Nullable
        private TypeMapping mappings;

        @Nullable
        private Map<String, JsonData> settings;

        /**
         * API name: {@code aliases}
         * <p>
         * Adds all entries of <code>map</code> to <code>aliases</code>.
         */
        public final Builder aliases(Map<String, Alias> map) {
            this.aliases = _mapPutAll(this.aliases, map);
            return this;
        }

        /**
         * API name: {@code aliases}
         * <p>
         * Adds an entry to <code>aliases</code>.
         */
        public final Builder aliases(String key, Alias value) {
            this.aliases = _mapPut(this.aliases, key, value);
            return this;
        }

        /**
         * API name: {@code aliases}
         * <p>
         * Adds an entry to <code>aliases</code> using a builder lambda.
         */
        public final Builder aliases(String key, Function<Alias.Builder, ObjectBuilder<Alias>> fn) {
            return aliases(key, fn.apply(new Alias.Builder()).build());
        }

        /**
         * API name: {@code mappings}
         */
        public final Builder mappings(@Nullable TypeMapping value) {
            this.mappings = value;
            return this;
        }

        /**
         * API name: {@code mappings}
         */
        public final Builder mappings(Function<TypeMapping.Builder, ObjectBuilder<TypeMapping>> fn) {
            return this.mappings(fn.apply(new TypeMapping.Builder()).build());
        }

        /**
         * API name: {@code settings}
         * <p>
         * Adds all entries of <code>map</code> to <code>settings</code>.
         */
        public final Builder settings(Map<String, JsonData> map) {
            this.settings = _mapPutAll(this.settings, map);
            return this;
        }

        /**
         * API name: {@code settings}
         * <p>
         * Adds an entry to <code>settings</code>.
         */
        public final Builder settings(String key, JsonData value) {
            this.settings = _mapPut(this.settings, key, value);
            return this;
        }

        /**
         * Builds a {@link IndexTemplateSummary}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public IndexTemplateSummary build() {
            _checkSingleUse();

            return new IndexTemplateSummary(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link IndexTemplateSummary}
     */
    public static final JsonpDeserializer<IndexTemplateSummary> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        IndexTemplateSummary::setupIndexTemplateSummaryDeserializer
    );

    protected static void setupIndexTemplateSummaryDeserializer(ObjectDeserializer<IndexTemplateSummary.Builder> op) {

        op.add(Builder::aliases, JsonpDeserializer.stringMapDeserializer(Alias._DESERIALIZER), "aliases");
        op.add(Builder::mappings, TypeMapping._DESERIALIZER, "mappings");
        op.add(Builder::settings, JsonpDeserializer.stringMapDeserializer(JsonData._DESERIALIZER), "settings");

    }

}
