/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch._types.aggregations;

import jakarta.json.stream.JsonGenerator;
import java.util.List;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;

// typedef: _types.aggregations.MultiTermsBucket

@JsonpDeserializable
public class MultiTermsBucket extends MultiBucketBase {
    private final List<String> key;

    @Nullable
    private final String keyAsString;

    @Nullable
    private final Long docCountErrorUpperBound;

    // ---------------------------------------------------------------------------------------------

    private MultiTermsBucket(Builder builder) {
        super(builder);

        this.key = ApiTypeHelper.unmodifiableRequired(builder.key, this, "key");
        this.keyAsString = builder.keyAsString;
        this.docCountErrorUpperBound = builder.docCountErrorUpperBound;

    }

    public static MultiTermsBucket of(Function<Builder, ObjectBuilder<MultiTermsBucket>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Required - API name: {@code key}
     */
    public final List<String> key() {
        return this.key;
    }

    /**
     * API name: {@code key_as_string}
     */
    @Nullable
    public final String keyAsString() {
        return this.keyAsString;
    }

    /**
     * API name: {@code doc_count_error_upper_bound}
     */
    @Nullable
    public final Long docCountErrorUpperBound() {
        return this.docCountErrorUpperBound;
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        super.serializeInternal(generator, mapper);
        if (ApiTypeHelper.isDefined(this.key)) {
            generator.writeKey("key");
            generator.writeStartArray();
            for (String item0 : this.key) {
                generator.write(item0);

            }
            generator.writeEnd();

        }
        if (this.keyAsString != null) {
            generator.writeKey("key_as_string");
            generator.write(this.keyAsString);

        }
        if (this.docCountErrorUpperBound != null) {
            generator.writeKey("doc_count_error_upper_bound");
            generator.write(this.docCountErrorUpperBound);

        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link MultiTermsBucket}.
     */

    public static class Builder extends MultiBucketBase.AbstractBuilder<Builder> implements ObjectBuilder<MultiTermsBucket> {
        private List<String> key;

        @Nullable
        private String keyAsString;

        @Nullable
        private Long docCountErrorUpperBound;

        /**
         * Required - API name: {@code key}
         * <p>
         * Adds all elements of <code>list</code> to <code>key</code>.
         */
        public final Builder key(List<String> list) {
            this.key = _listAddAll(this.key, list);
            return this;
        }

        /**
         * Required - API name: {@code key}
         * <p>
         * Adds one or more values to <code>key</code>.
         */
        public final Builder key(String value, String... values) {
            this.key = _listAdd(this.key, value, values);
            return this;
        }

        /**
         * API name: {@code key_as_string}
         */
        public final Builder keyAsString(@Nullable String value) {
            this.keyAsString = value;
            return this;
        }

        /**
         * API name: {@code doc_count_error_upper_bound}
         */
        public final Builder docCountErrorUpperBound(@Nullable Long value) {
            this.docCountErrorUpperBound = value;
            return this;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * Builds a {@link MultiTermsBucket}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public MultiTermsBucket build() {
            _checkSingleUse();

            return new MultiTermsBucket(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link MultiTermsBucket}
     */
    public static final JsonpDeserializer<MultiTermsBucket> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        MultiTermsBucket::setupMultiTermsBucketDeserializer
    );

    protected static void setupMultiTermsBucketDeserializer(ObjectDeserializer<MultiTermsBucket.Builder> op) {
        setupMultiBucketBaseDeserializer(op);
        op.add(Builder::key, JsonpDeserializer.arrayDeserializer(JsonpDeserializer.stringDeserializer()), "key");
        op.add(Builder::keyAsString, JsonpDeserializer.stringDeserializer(), "key_as_string");
        op.add(Builder::docCountErrorUpperBound, JsonpDeserializer.longDeserializer(), "doc_count_error_upper_bound");

    }

}
