/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch._types.query_dsl;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;

// typedef: _types.query_dsl.RegexpQuery

@JsonpDeserializable
public class RegexpQuery extends QueryBase implements QueryVariant {
    // Single key dictionary
    private final String field;

    @Nullable
    private final Boolean caseInsensitive;

    @Nullable
    private final String flags;

    @Nullable
    private final Integer maxDeterminizedStates;

    @Nullable
    private final String rewrite;

    private final String value;

    // ---------------------------------------------------------------------------------------------

    private RegexpQuery(Builder builder) {
        super(builder);
        this.field = ApiTypeHelper.requireNonNull(builder.field, this, "field");

        this.caseInsensitive = builder.caseInsensitive;
        this.flags = builder.flags;
        this.maxDeterminizedStates = builder.maxDeterminizedStates;
        this.rewrite = builder.rewrite;
        this.value = ApiTypeHelper.requireNonNull(builder.value, this, "value");

    }

    public static RegexpQuery of(Function<Builder, ObjectBuilder<RegexpQuery>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Query variant kind.
     */
    @Override
    public Query.Kind _queryKind() {
        return Query.Kind.Regexp;
    }

    /**
     * Required - The target field
     */
    public final String field() {
        return this.field;
    }

    /**
     * API name: {@code case_insensitive}
     */
    @Nullable
    public final Boolean caseInsensitive() {
        return this.caseInsensitive;
    }

    /**
     * API name: {@code flags}
     */
    @Nullable
    public final String flags() {
        return this.flags;
    }

    /**
     * API name: {@code max_determinized_states}
     */
    @Nullable
    public final Integer maxDeterminizedStates() {
        return this.maxDeterminizedStates;
    }

    /**
     * API name: {@code rewrite}
     */
    @Nullable
    public final String rewrite() {
        return this.rewrite;
    }

    /**
     * Required - API name: {@code value}
     */
    public final String value() {
        return this.value;
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject(this.field);

        super.serializeInternal(generator, mapper);
        if (this.caseInsensitive != null) {
            generator.writeKey("case_insensitive");
            generator.write(this.caseInsensitive);

        }
        if (this.flags != null) {
            generator.writeKey("flags");
            generator.write(this.flags);

        }
        if (this.maxDeterminizedStates != null) {
            generator.writeKey("max_determinized_states");
            generator.write(this.maxDeterminizedStates);

        }
        if (this.rewrite != null) {
            generator.writeKey("rewrite");
            generator.write(this.rewrite);

        }
        generator.writeKey("value");
        generator.write(this.value);

        generator.writeEnd();

    }

    public Builder toBuilder() {
        return new Builder().field(field)
            .value(value)
            .caseInsensitive(caseInsensitive)
            .flags(flags)
            .maxDeterminizedStates(maxDeterminizedStates)
            .rewrite(rewrite)
            .value(value);
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link RegexpQuery}.
     */

    public static class Builder extends QueryBase.AbstractBuilder<Builder> implements ObjectBuilder<RegexpQuery> {
        private String field;

        /**
         * Required - The target field
         */
        public final Builder field(String value) {
            this.field = value;
            return this;
        }

        @Nullable
        private Boolean caseInsensitive;

        @Nullable
        private String flags;

        @Nullable
        private Integer maxDeterminizedStates;

        @Nullable
        private String rewrite;

        private String value;

        /**
         * API name: {@code case_insensitive}
         */
        public final Builder caseInsensitive(@Nullable Boolean value) {
            this.caseInsensitive = value;
            return this;
        }

        /**
         * API name: {@code flags}
         */
        public final Builder flags(@Nullable String value) {
            this.flags = value;
            return this;
        }

        /**
         * API name: {@code max_determinized_states}
         */
        public final Builder maxDeterminizedStates(@Nullable Integer value) {
            this.maxDeterminizedStates = value;
            return this;
        }

        /**
         * API name: {@code rewrite}
         */
        public final Builder rewrite(@Nullable String value) {
            this.rewrite = value;
            return this;
        }

        /**
         * Required - API name: {@code value}
         */
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * Builds a {@link RegexpQuery}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public RegexpQuery build() {
            _checkSingleUse();

            return new RegexpQuery(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link RegexpQuery}
     */
    public static final JsonpDeserializer<RegexpQuery> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        RegexpQuery::setupRegexpQueryDeserializer
    );

    protected static void setupRegexpQueryDeserializer(ObjectDeserializer<RegexpQuery.Builder> op) {
        QueryBase.setupQueryBaseDeserializer(op);
        op.add(Builder::caseInsensitive, JsonpDeserializer.booleanDeserializer(), "case_insensitive");
        op.add(Builder::flags, JsonpDeserializer.stringDeserializer(), "flags");
        op.add(Builder::maxDeterminizedStates, JsonpDeserializer.integerDeserializer(), "max_determinized_states");
        op.add(Builder::rewrite, JsonpDeserializer.stringDeserializer(), "rewrite");
        op.add(Builder::value, JsonpDeserializer.stringDeserializer(), "value");

        op.setKey(Builder::field, JsonpDeserializer.stringDeserializer());
        op.shortcutProperty("value");

    }

}
