/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.saml.metadata.resolver.impl;

import java.util.Timer;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.hc.client5.http.classic.HttpClient;
import org.opensaml.saml.metadata.resolver.RemoteMetadataResolver;
import org.slf4j.Logger;

import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.resolver.CriteriaSet;

/**
 * Simple implementation of an HTTP-based dynamic metadata resolver which builds the request URL
 * to process based on a {@link Function} instance.
 * 
 * <p>
 * The function defaults to an instance of {@link HTTPEntityIDRequestURLBuilder}, thereby implementing
 * the "well-known location" resolution mechanism defined in the SAML 2 metadata specification if the entity ID
 * is an HTTP or HTTPS URL.
 * </p>
 */
public class FunctionDrivenDynamicHTTPMetadataResolver extends AbstractDynamicHTTPMetadataResolver
        implements RemoteMetadataResolver {
    
    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(FunctionDrivenDynamicHTTPMetadataResolver.class);
    
    /** Function for building the request URL. */
    @Nonnull private Function<CriteriaSet, String> requestURLBuilder;

    /**
     * Constructor.
     *
     * @param client the instance of {@link HttpClient} used to fetch remote metadata
     */
    public FunctionDrivenDynamicHTTPMetadataResolver(@Nonnull final HttpClient client) {
        this(null, client);
    }
    
    /**
     * Constructor.
     *
     * @param backgroundTaskTimer the {@link Timer} instance used to run resolver background managment tasks
     * @param client the instance of {@link HttpClient} used to fetch remote metadata
     */
    public FunctionDrivenDynamicHTTPMetadataResolver(@Nullable final Timer backgroundTaskTimer,
                                                     @Nonnull final HttpClient client) {
        super(backgroundTaskTimer, client);
        requestURLBuilder = new HTTPEntityIDRequestURLBuilder();
    }

    /**
     * Get the function which builds the request URL.
     * 
     * <p>Defaults to an instance of {@link HTTPEntityIDRequestURLBuilder}.</p>
     * 
     * @return the request URL builder function instance
     */
    @Nonnull public Function<CriteriaSet, String> getRequestURLBuilder() {
        return requestURLBuilder;
    }

    /**
     * 
     * Set the function which builds the request URL.
     * 
     * <p>Defaults to an instance of {@link HTTPEntityIDRequestURLBuilder}.</p>
     * 
     * @param builder the request URL builder function instance
     */
    public void setRequestURLBuilder(@Nonnull final Function<CriteriaSet, String> builder) {
        checkSetterPreconditions();
        requestURLBuilder = Constraint.isNotNull(builder, "Request URL builder function cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    @Nullable protected String buildRequestURL(@Nullable final CriteriaSet criteria) {
        final String url = getRequestURLBuilder().apply(criteria);
        
        log.debug("{} URL generated by request builder was: {}", getLogPrefix(), url);
        
        return url;
    }

    /** {@inheritDoc} */
    @Nullable public String getMetadataURI() {
        // This is for metrics exposure.
        if (requestURLBuilder instanceof MetadataQueryProtocolRequestURLBuilder mdq) {
            return mdq.getBaseURL();
        }
        return null;
    }

}