/*
 * Decompiled with CFR 0.152.
 */
package org.opensaml.core.xml.persist;

import com.google.common.collect.Sets;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.xml.namespace.QName;
import net.shibboleth.utilities.java.support.collection.Pair;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.XMLObjectBaseTestCase;
import org.opensaml.core.xml.XMLRuntimeException;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.mock.SimpleXMLObject;
import org.opensaml.core.xml.persist.FilesystemLoadSaveManager;
import org.opensaml.core.xml.util.XMLObjectSource;
import org.opensaml.core.xml.util.XMLObjectSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testng.Assert;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

public class FilesystemLoadSaveManagerTest
extends XMLObjectBaseTestCase {
    private Logger log = LoggerFactory.getLogger(FilesystemLoadSaveManagerTest.class);
    private File baseDir;
    private FilesystemLoadSaveManager<SimpleXMLObject> manager;

    @BeforeMethod
    public void setUp() throws IOException {
        this.baseDir = new File(System.getProperty("java.io.tmpdir"), "load-save-manager-test");
        this.baseDir.deleteOnExit();
        this.log.debug("Using base directory: {}", (Object)this.baseDir.getAbsolutePath());
        this.resetBaseDir();
        Assert.assertTrue((boolean)this.baseDir.mkdirs());
        this.manager = new FilesystemLoadSaveManager(this.baseDir);
    }

    @AfterMethod
    public void tearDown() throws IOException {
        this.resetBaseDir();
    }

    @Test
    public void emptyDir() throws IOException {
        this.testState(Sets.newHashSet());
    }

    @DataProvider
    public Object[][] saveLoadUpdateRemoveParams() {
        return new Object[][]{{Boolean.FALSE}, {Boolean.TRUE}};
    }

    @Test(dataProvider="saveLoadUpdateRemoveParams")
    public void saveLoadUpdateRemove(Boolean buildWithObjectSourceByteArray) throws IOException {
        this.testState(Sets.newHashSet());
        Assert.assertNull((Object)this.manager.load("bogus"));
        this.manager.save("foo", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME, buildWithObjectSourceByteArray))));
        this.testState(Sets.newHashSet((Object[])new String[]{"foo"}));
        this.manager.save("bar", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME, buildWithObjectSourceByteArray))));
        this.manager.save("baz", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME, buildWithObjectSourceByteArray))));
        this.testState(Sets.newHashSet((Object[])new String[]{"foo", "bar", "baz"}));
        this.manager.save("bar", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME, buildWithObjectSourceByteArray))), true);
        this.testState(Sets.newHashSet((Object[])new String[]{"foo", "bar", "baz"}));
        try {
            this.manager.save("bar", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME, buildWithObjectSourceByteArray))), false);
            Assert.fail((String)"Should have failed on duplicate save without overwrite");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.testState(Sets.newHashSet((Object[])new String[]{"foo", "bar", "baz"}));
        this.testState(Sets.newHashSet((Object[])new String[]{"foo", "bar", "baz"}));
        Assert.assertTrue((boolean)this.manager.updateKey("foo", "foo2"));
        this.testState(Sets.newHashSet((Object[])new String[]{"foo2", "bar", "baz"}));
        Assert.assertFalse((boolean)this.manager.updateKey("foo", "foo2"));
        this.testState(Sets.newHashSet((Object[])new String[]{"foo2", "bar", "baz"}));
        try {
            this.manager.updateKey("bar", "baz");
            Assert.fail((String)"updateKey should have failed to due existing new key name");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.testState(Sets.newHashSet((Object[])new String[]{"foo2", "bar", "baz"}));
        Assert.assertFalse((boolean)this.manager.remove("foo"));
        this.testState(Sets.newHashSet((Object[])new String[]{"foo2", "bar", "baz"}));
        Assert.assertTrue((boolean)this.manager.remove("foo2"));
        this.testState(Sets.newHashSet((Object[])new String[]{"bar", "baz"}));
        Assert.assertTrue((boolean)this.manager.remove("bar"));
        Assert.assertTrue((boolean)this.manager.remove("baz"));
        this.testState(Sets.newHashSet());
    }

    @Test
    public void checkCheckModifyTimeTracking() throws IOException {
        this.manager = new FilesystemLoadSaveManager(this.baseDir, true);
        Assert.assertNull((Object)this.manager.load("foo"));
        Assert.assertNull((Object)this.manager.getLoadLastModified("foo"));
        this.manager.save("foo", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME, true))));
        Assert.assertNotNull((Object)this.manager.load("foo"));
        Long initialCachedModified = this.manager.getLoadLastModified("foo");
        Assert.assertNotNull((Object)initialCachedModified);
        Assert.assertNull((Object)this.manager.load("foo"));
        Assert.assertEquals((Object)this.manager.getLoadLastModified("foo"), (Object)initialCachedModified);
        Uninterruptibles.sleepUninterruptibly((long)2L, (TimeUnit)TimeUnit.SECONDS);
        this.manager.save("foo", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME, true))), true);
        Assert.assertNotNull((Object)this.manager.load("foo"));
        Long updatedCachedModified = this.manager.getLoadLastModified("foo");
        Assert.assertNotNull((Object)updatedCachedModified);
        Assert.assertNotEquals((Object)updatedCachedModified, (Object)initialCachedModified);
        Assert.assertNull((Object)this.manager.load("foo"));
        Assert.assertEquals((Object)this.manager.getLoadLastModified("foo"), (Object)updatedCachedModified);
        this.manager.updateKey("foo", "bar");
        Assert.assertNull((Object)this.manager.load("foo"));
        Assert.assertNull((Object)this.manager.load("bar"));
        Assert.assertNull((Object)this.manager.getLoadLastModified("foo"));
        Assert.assertNotNull((Object)this.manager.getLoadLastModified("bar"));
        Assert.assertEquals((Object)this.manager.getLoadLastModified("bar"), (Object)updatedCachedModified);
        this.manager.remove("bar");
        Assert.assertNull((Object)this.manager.getLoadLastModified("bar"));
    }

    @Test
    public void buildTargetFileFromKey() throws IOException {
        File target = this.manager.buildFile("abc");
        Assert.assertEquals((Object)target, (Object)new File(this.baseDir, "abc"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={IOException.class})
    public void targetExistsButIsNotAFile() throws IOException {
        File target = new File(this.baseDir, "abc");
        Assert.assertFalse((boolean)target.exists());
        target.mkdir();
        try {
            this.manager.buildFile("abc");
        }
        finally {
            if (target.exists()) {
                Files.delete(target.toPath());
            }
        }
    }

    @Test(expectedExceptions={ConstraintViolationException.class})
    public void targetKeyIsNull() throws IOException {
        this.manager.buildFile(null);
    }

    @Test(expectedExceptions={ConstraintViolationException.class})
    public void targetKeyIsEmpty() throws IOException {
        this.manager.buildFile("  ");
    }

    @Test
    public void ctorCreateDirectory() throws IOException {
        this.resetBaseDir();
        Assert.assertFalse((boolean)this.baseDir.exists());
        new FilesystemLoadSaveManager(this.baseDir);
        Assert.assertTrue((boolean)this.baseDir.exists());
    }

    @Test
    public void ctorPathTrimming() throws IOException {
        new FilesystemLoadSaveManager(String.format("    %s     ", this.baseDir.getAbsolutePath()));
        File target = this.manager.buildFile("abc");
        Assert.assertEquals((Object)target.getParentFile(), (Object)this.baseDir);
        Assert.assertEquals((String)target.getParent(), (String)this.baseDir.getAbsolutePath());
        Assert.assertFalse((boolean)target.getParent().startsWith(" "));
        Assert.assertFalse((boolean)target.getParent().endsWith(" "));
    }

    @Test(expectedExceptions={ConstraintViolationException.class})
    public void ctorEmptyPathString() {
        new FilesystemLoadSaveManager("  ");
    }

    @Test(expectedExceptions={ConstraintViolationException.class})
    public void ctorNullFile() {
        new FilesystemLoadSaveManager((File)null);
    }

    @Test(expectedExceptions={ConstraintViolationException.class})
    public void ctorRelativeDir() {
        new FilesystemLoadSaveManager("my/relative/dir");
    }

    @Test(expectedExceptions={ConstraintViolationException.class})
    public void ctorBaseDirPathExistsButNotADirectory() throws IOException {
        this.resetBaseDir();
        Files.createFile(this.baseDir.toPath(), new FileAttribute[0]);
        new FilesystemLoadSaveManager(this.baseDir);
    }

    @Test
    public void iterator() throws IOException {
        Iterator iterator = null;
        iterator = this.manager.listAll().iterator();
        Assert.assertFalse((boolean)iterator.hasNext());
        try {
            iterator.next();
            Assert.fail((String)"Should have failed due to no more elements");
        }
        catch (NoSuchElementException noSuchElementException) {
            // empty catch block
        }
        this.manager.save("foo", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME))));
        iterator = this.manager.listAll().iterator();
        Assert.assertTrue((boolean)iterator.hasNext());
        Assert.assertNotNull(iterator.next());
        Assert.assertFalse((boolean)iterator.hasNext());
        try {
            iterator.next();
            Assert.fail((String)"Should have failed due to no more elements");
        }
        catch (NoSuchElementException noSuchElementException) {
            // empty catch block
        }
        this.manager.save("bar", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME))));
        this.manager.save("baz", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME))));
        iterator = this.manager.listAll().iterator();
        Assert.assertTrue((boolean)iterator.hasNext());
        Assert.assertNotNull(iterator.next());
        Assert.assertTrue((boolean)iterator.hasNext());
        Assert.assertNotNull(iterator.next());
        Assert.assertTrue((boolean)iterator.hasNext());
        Assert.assertNotNull(iterator.next());
        Assert.assertFalse((boolean)iterator.hasNext());
        try {
            iterator.next();
            Assert.fail((String)"Should have failed due to no more elements");
        }
        catch (NoSuchElementException noSuchElementException) {
            // empty catch block
        }
        this.manager.remove("foo");
        iterator = this.manager.listAll().iterator();
        Assert.assertTrue((boolean)iterator.hasNext());
        Assert.assertNotNull(iterator.next());
        Assert.assertTrue((boolean)iterator.hasNext());
        Assert.assertNotNull(iterator.next());
        Assert.assertFalse((boolean)iterator.hasNext());
        try {
            iterator.next();
            Assert.fail((String)"Should have failed due to no more elements");
        }
        catch (NoSuchElementException noSuchElementException) {
            // empty catch block
        }
        this.manager.remove("bar");
        this.manager.remove("baz");
        iterator = this.manager.listAll().iterator();
        Assert.assertFalse((boolean)iterator.hasNext());
        try {
            iterator.next();
            Assert.fail((String)"Should have failed due to no more elements");
        }
        catch (NoSuchElementException noSuchElementException) {
            // empty catch block
        }
        this.manager.save("foo", (XMLObject)((SimpleXMLObject)((Object)this.buildXMLObject(SimpleXMLObject.ELEMENT_NAME))));
        Assert.assertTrue((boolean)this.manager.exists("foo"));
        Assert.assertNotNull((Object)this.manager.load("foo"));
        iterator = this.manager.listAll().iterator();
        this.manager.remove("foo");
        Assert.assertFalse((boolean)iterator.hasNext());
    }

    private void testState(Set<String> expectedKeys) throws IOException {
        Assert.assertEquals((boolean)this.manager.listKeys().isEmpty(), (expectedKeys.isEmpty() ? 1 : 0) != 0);
        Assert.assertEquals((Set)this.manager.listKeys(), expectedKeys);
        for (String expectedKey : expectedKeys) {
            Assert.assertTrue((boolean)this.manager.exists(expectedKey));
            SimpleXMLObject sxo = (SimpleXMLObject)this.manager.load(expectedKey);
            Assert.assertNotNull((Object)((Object)sxo));
            Assert.assertEquals((int)sxo.getObjectMetadata().get(XMLObjectSource.class).size(), (int)1);
        }
        Assert.assertEquals((boolean)this.manager.listAll().iterator().hasNext(), (!expectedKeys.isEmpty() ? 1 : 0) != 0);
        int sawCount = 0;
        for (Pair entry : this.manager.listAll()) {
            ++sawCount;
            Assert.assertTrue((boolean)expectedKeys.contains(entry.getFirst()));
            Assert.assertNotNull((Object)entry.getSecond());
        }
        Assert.assertEquals((int)sawCount, (int)expectedKeys.size());
    }

    private void resetBaseDir() throws IOException {
        if (this.baseDir.exists()) {
            if (this.baseDir.isDirectory()) {
                for (File child : this.baseDir.listFiles()) {
                    Files.delete(child.toPath());
                }
            }
            Files.delete(this.baseDir.toPath());
        }
    }

    protected <T extends XMLObject> T buildXMLObject(QName name, boolean withObjectSource) {
        Object xmlObject = super.buildXMLObject(name);
        if (withObjectSource) {
            try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
                XMLObjectSupport.marshallToOutputStream(xmlObject, (OutputStream)baos);
                xmlObject.getObjectMetadata().put((Object)new XMLObjectSource(baos.toByteArray()));
            }
            catch (IOException | MarshallingException e) {
                throw new XMLRuntimeException("Error marshalling XMLObject", (Exception)e);
            }
        }
        return xmlObject;
    }
}

