/*
 * Copyright 2022 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.text;

import lombok.EqualsAndHashCode;
import lombok.Value;
import org.jetbrains.annotations.Nullable;
import org.openrewrite.ExecutionContext;
import org.openrewrite.Option;
import org.openrewrite.Recipe;
import org.openrewrite.SourceFile;
import org.openrewrite.Tree;
import org.openrewrite.internal.ListUtils;

import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.UUID;

@Value
@EqualsAndHashCode(callSuper = true)
public class AppendToTextFile extends Recipe {
    @Option(displayName = "Relative File Name",
            description = "File name, using a relative path. If a non-plaintext file already exists at this location, then this recipe will do nothing.",
            example = "foo/bar/baz.txt")
    String relativeFileName;

    @Option(displayName = "Content",
            description = "Multiline text content to be appended to the file.",
            example = "Some text.")
    String content;

    @Option(displayName = "Preamble",
            description = "If creating this file fresh, then this content will be included at the beginning. Default nothing.",
            example = "# File generated by OpenRewrite #",
            required = false)
    @Nullable String preamble;

    @Option(displayName = "Append newline",
            description = "Print a newline automatically after the content (and preamble). Default true.",
            example = "false",
            required = false)
    @Nullable Boolean appendNewline;

    @Option(displayName = "Existing file strategy",
            description = "Determines behavior if a file exists at this location prior to Rewrite execution.\n\n"
                    + "- `continue`: append new content to existing file contents. If existing file is not plaintext, recipe does nothing.\n"
                    + "- `replace`: remove existing content from file.\n"
                    + "- `leave`: *(default)* do nothing. Existing file is fully preserved.\n\n"
                    + "Note: this only affects the first interaction with the specified file per Rewrite execution.\n"
                    + "Subsequent instances of this recipe in the same Rewrite execution will always append.",
            valid = {"continue", "replace", "leave"},
            required = false)
    @Nullable String existingFileStrategy;
    public enum Strategy { CONTINUE, REPLACE, LEAVE }

    private final static String CREATED_THIS_EXECUTION_MESSAGE_KEY = "AppendToTextFile.CreatedThisExecution";
    String alreadyVisitedMessageKey = "AppendToTextFile.AlreadyVisitedBy." + Tree.randomId();

    public String getDisplayName() {
        return "Append to text file";
    }

    public String getDescription() {
        return "Appends content to a plaintext file. Multiple instances of this recipe in the same execution context can all contribute.";
    }

    @Override
    protected List<SourceFile> visit(List<SourceFile> before, ExecutionContext ctx) {
        String maybeNewline = !Boolean.FALSE.equals(appendNewline) ? "\n" : "";
        return visit(before, ctx,
                relativeFileName,
                content + maybeNewline,
                preamble != null ? preamble + maybeNewline : "",
                existingFileStrategy != null ? Strategy.valueOf(existingFileStrategy.toUpperCase()) : Strategy.LEAVE);
    }

    private List<SourceFile> visit(List<SourceFile> before, ExecutionContext ctx,
            String path, String content, String preamble, Strategy strategy) {
        SourceFile sourceFile = null;
        for (SourceFile it : before) {
            if (it.getSourcePath().toString().equals(Paths.get(path).toString())) {
                sourceFile = it;
                break;
            }
        }
        if (sourceFile == null) {
            return ListUtils.concat(before, createFile(path, preamble + content, ctx, null));
        }
        if (isAlreadyVisited(sourceFile, ctx) || !(sourceFile instanceof PlainText)) {
            return before;
        }
        PlainText existingPlainText = (PlainText) sourceFile;
        if (isCreatedInThisExecution(existingPlainText, ctx)) {
            return replace(before, existingPlainText, withAppend(existingPlainText, content, ctx));
        }
        switch (strategy) {
            case CONTINUE:
                return replace(before, existingPlainText,
                        withAppend(existingPlainText, content, ctx));
            case REPLACE:
                return replace(before, existingPlainText,
                        createFile(path, preamble + content, ctx, existingPlainText.getId()));
            case LEAVE:
                return before;
            default:
                throw new IllegalArgumentException();
        }
    }

    private List<SourceFile> replace(List<SourceFile> before, SourceFile existingFile, PlainText newFile) {
        final ArrayList<SourceFile> after = new ArrayList<>(before);
        after.remove(existingFile);
        after.add(newFile);
        return after;
    }

    private PlainText withAppend(PlainText before, String content, ExecutionContext ctx) {
        final PlainText after = before.withText(before.getText() + content);
        alreadyVisited(after, ctx);
        return after;
    }

    private PlainText createFile(String path, String content, ExecutionContext ctx, @Nullable UUID id) {
        PlainText plainText = new PlainTextParser().parse(content).get(0).withSourcePath(Paths.get(path));
        if (id != null) {
            plainText = plainText.withId(id);
        }
        createdInThisExecution(plainText, ctx);
        return plainText;
    }

    private boolean isCreatedInThisExecution(SourceFile sourceFile, ExecutionContext ctx) {
        Set<UUID> createdSet = ctx.getMessage(CREATED_THIS_EXECUTION_MESSAGE_KEY);
        return createdSet != null && createdSet.contains(sourceFile.getId());
    }

    private void createdInThisExecution(SourceFile sourceFile, ExecutionContext ctx) {
        ctx.putMessageInSet(CREATED_THIS_EXECUTION_MESSAGE_KEY, sourceFile.getId());
        alreadyVisited(sourceFile, ctx);
    }

    private boolean isAlreadyVisited(SourceFile sourceFile, ExecutionContext ctx) {
        Set<UUID> alreadyVisitedSet = ctx.getMessage(this.alreadyVisitedMessageKey);
        return alreadyVisitedSet != null && alreadyVisitedSet.contains(sourceFile.getId());
    }

    private void alreadyVisited(SourceFile sourceFile, ExecutionContext ctx) {
        ctx.putMessageInSet(this.alreadyVisitedMessageKey, sourceFile.getId());
    }
}