"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AstExplorerResolver = void 0;
const fs_1 = require("fs");
const url_1 = require("url");
const NetworkResolver_1 = require("./NetworkResolver");
const EDITOR_HASH_PATTERN = /^#\/gist\/(\w+)(?:\/(\w+))?$/;
/**
 * Resolves plugins from AST Explorer transforms.
 *
 * astexplorer.net uses GitHub gists to save and facilitate sharing. This
 * resolver accepts either the editor URL or the gist API URL.
 */
class AstExplorerResolver extends NetworkResolver_1.NetworkResolver {
    constructor(baseURL = new url_1.URL('https://astexplorer.net/')) {
        super();
        this.baseURL = baseURL;
    }
    canResolve(source) {
        const _super = Object.create(null, {
            canResolve: { get: () => super.canResolve }
        });
        return __awaiter(this, void 0, void 0, function* () {
            if (yield _super.canResolve.call(this, source)) {
                const url = new url_1.URL(yield this.normalize(source));
                const canResolve = this.matchesHost(url) &&
                    /^\/api\/v1\/gist\/[a-f0-9]+(\/(?:[a-f0-9]+|latest))?$/.test(url.pathname);
                return canResolve;
            }
            return false;
        });
    }
    resolve(source) {
        const _super = Object.create(null, {
            resolve: { get: () => super.resolve }
        });
        return __awaiter(this, void 0, void 0, function* () {
            const filename = yield _super.resolve.call(this, yield this.normalize(source));
            const text = yield fs_1.promises.readFile(filename, { encoding: 'utf8' });
            let data;
            try {
                data = JSON.parse(text);
            }
            catch (_a) {
                throw new Error(`data loaded from ${source} is not JSON: ${text.slice(0, 100)}`);
            }
            if (!data ||
                !data.files ||
                !data.files['transform.js'] ||
                !data.files['transform.js'].content) {
                throw new Error("'transform.js' could not be found, perhaps transform is disabled");
            }
            yield fs_1.promises.writeFile(filename, data.files['transform.js'].content, {
                encoding: 'utf8',
            });
            return filename;
        });
    }
    normalize(source) {
        return __awaiter(this, void 0, void 0, function* () {
            const url = new url_1.URL(source);
            if (!this.matchesHost(url)) {
                return source;
            }
            const match = url.hash.match(EDITOR_HASH_PATTERN);
            if (!match) {
                return source;
            }
            let path = `/api/v1/gist/${match[1]}`;
            if (match[2]) {
                path += `/${match[2]}`;
            }
            return new url_1.URL(path, this.baseURL).toString();
        });
    }
    matchesHost(url) {
        if (url.host !== this.baseURL.host) {
            return false;
        }
        // use SSL even if the URL doesn't use it
        return url.protocol === this.baseURL.protocol || url.protocol === 'http:';
    }
}
exports.AstExplorerResolver = AstExplorerResolver;
//# sourceMappingURL=AstExplorerResolver.js.map