import ts from "typescript";
import { ChangeAction } from "../../../utils/textChanges.js";
import PropertyAssignmentBaseFix from "./PropertyAssignmentBaseFix.js";
;
/**
 * Fix a global property access. Requires a module name which will be imported and replaces the defined property access.
 * The property access is in the order of the AST, e.g. ["core", "ui", "sap"]
 */
export default class PropertyAssignmentGeneratorFix extends PropertyAssignmentBaseFix {
    params;
    generatorContext = {};
    propertyName;
    propertyInitializer;
    constructor(params) {
        super();
        this.params = params;
    }
    visitLinterNode(node, sourcePosition, helpers) {
        if (!super.visitLinterNode(node, sourcePosition, helpers)) {
            return false;
        }
        if (!ts.isPropertyAssignment(node) || !ts.isIdentifier(node.name)) {
            return false;
        }
        if (this.params.validatePropertyAssignment) {
            if (!this.params.validatePropertyAssignment(this.generatorContext, helpers, node)) {
                return false;
            }
        }
        this.sourcePosition = sourcePosition;
        return true;
    }
    visitAutofixNode(node, position, sourceFile) {
        if (!super.visitAutofixNode(node, position, sourceFile)) {
            return false;
        }
        if (!ts.isPropertyAssignment(node)) {
            return false;
        }
        this.propertyName = node.name.getFullText();
        this.propertyInitializer = node.initializer.getFullText();
        return true;
    }
    visitAutofixXmlNode(node, toPosition) {
        if (!super.visitAutofixXmlNode(node, toPosition)) {
            return false;
        }
        this.propertyName = node.name.value;
        this.propertyInitializer = node.value.value;
        return true;
    }
    generateChanges() {
        if (this.startPos === undefined || this.endPos === undefined) {
            throw new Error("Start and end position are not defined");
        }
        if (this.propertyName === undefined || this.propertyInitializer === undefined) {
            throw new Error("Property name or initializer is not defined");
        }
        const value = this.params.generator(this.generatorContext, this.propertyName, this.propertyInitializer);
        if (value === undefined) {
            return;
        }
        if (value !== "") {
            return {
                action: ChangeAction.REPLACE,
                start: this.startPos,
                end: this.endPos,
                value,
            };
        }
        else {
            return {
                action: ChangeAction.DELETE,
                start: this.startPos,
                end: this.trailingCommaPos ?? this.endPos,
            };
        }
    }
}
//# sourceMappingURL=PropertyAssignmentGeneratorFix.js.map