// For Library Version: 1.136.12

declare module "sap/fe/test/library" {}

declare module "sap/fe/test/api/BaseAPI" {
  import DialogType from "sap/fe/test/api/DialogType";

  /**
   * An action parameter dialog field identifier
   */
  export type ActionDialogFieldIdentifier = {
    /**
     * The name of the field
     */
    property: string;
  };

  /**
   * An action identifier
   */
  export type ActionIdentifier = {
    /**
     * The name of the service
     */
    service: string;
    /**
     * The name of the action
     */
    action: string;
    /**
     * Defines whether the action is a bound action (default: false)
     */
    unbound?: boolean;
  };

  /**
   * An dialog field identifier
   */
  export type DialogFieldIdentifier = {
    /**
     * The name of the field In case of an action parameter dialog, this is the fieldname only
     * ```javascript
     *
     * {
     *     property:
     * }
     * ```
     *  In case of a create dialog, the concatenation of the entity and fieldname has to be provided
     *
     * ```javascript
     *
     * {
     *     property: ::
     * }
     * ```
     */
    property: string;
  };

  /**
   * A dialog identifier
   */
  export type DialogIdentifier = {
    /**
     * The type of the dialog
     */
    type: DialogType;
  };

  /**
   * A field identifier
   */
  export type FieldIdentifier = {
    /**
     * The name of the field group containing the field
     */
    fieldGroup?: string;
    /**
     * The name of the field
     */
    property: string;
    /**
     * The name of the connected field containing the field
     */
    connectedFields?: string;
  };

  /**
   * A section identifier
   */
  export type SectionIdentifier = {
    /**
     * The name of the section facet
     */
    section: string;
    /**
     * The name of the sub-section facet
     */
    subSection?: string;
  };

  /**
   * A table identifier
   */
  export type TableIdentifier = {
    /**
     * The name of the navigation property used for the table
     */
    property: string;
    /**
     * The name of a qualifier for the table
     */
    qualifier?: string;
  };

  /**
   * A view identifier
   */
  export type ViewIdentifier = {
    /**
     * The name of the view as defined in the manifest file
     */
    key: string;
  };
}

declare module "sap/fe/test/api/TableAPI" {
  /**
   * A column identifier
   */
  export type ColumnIdentifier = {
    /**
     * The technical name of the column
     */
    name: string;
  };

  export default class TableAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Confirms and closes the adaptation dialog of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iConfirmColumnAdaptation(): object;
    /**
     * Confirms and closes the sorting dialog of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iConfirmColumnSorting(): object;
    /**
     * Confirms and closes the filtering dialog of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iConfirmFilterDialog(): object;
    /**
     * Opens the column adaptation dialog of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iOpenColumnAdaptation(): object;
    /**
     * Opens the sorting dialog of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iOpenColumnSorting(): object;
    /**
     * Opens the filtering dialog of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iOpenFilterDialog(): object;
  }
}

declare module "sap/fe/test/api/DialogActions" {
  import { DialogFieldIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class DialogActions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Cancels the dialog by clicking the corresponding button (for example, 'Cancel').
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCancel(): object;
    /**
     * Changes the content of a field in a dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iChangeDialogField(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: DialogFieldIdentifier,
      /**
       * The new target value.
       */
      vValue?: string,
      /**
       * Set to `true` to clear previously set filters, otherwise all previously set values will be kept
       */
      bClearFirst?: boolean
    ): object;
    /**
     * Closes the dialog by pressing the 'Escape' key.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iClose(): object;
    /**
     * Confirms the dialog by clicking the corresponding button (for example, 'OK').
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iConfirm(): object;
  }
}

declare module "sap/fe/test/api/DialogAssertions" {
  import {
    ActionDialogFieldIdentifier,
    DialogFieldIdentifier,
  } from "sap/fe/test/api/BaseAPI";

  export default class DialogAssertions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks the content and state of a field in an action parameter dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckActionParameterDialogField(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: ActionDialogFieldIdentifier,
      /**
       * Expected value(s) of the field. if passed as an object, the following pattern will be considered:
       * ```javascript
       *
       * {
       *     value: , 		// optional
       * }
       * ```
       */
      vValue?: string | any[] | object,
      /**
       * Defines the expected state of the field
       */
      mState?: object
    ): object;
    /**
     * Checks the cancellation button of the dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckCancel(
      /**
       * Defines the expected state of the button
       */
      mButtonState?: object
    ): object;
    /**
     * Checks the confirmation button of the dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckConfirm(
      /**
       * Defines the expected state of the button
       */
      mButtonState?: object
    ): object;
    /**
     * Checks the content and state of a field in a dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckDialogField(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: DialogFieldIdentifier,
      /**
       * Expected value(s) of the field. if passed as an object, the following pattern will be considered:
       * ```javascript
       *
       * {
       *     value: , 		// optional
       * }
       * ```
       */
      vValue?: string | any[] | object,
      /**
       * Defines the expected state of the field
       */
      mState?: object
    ): object;
    /**
     * Checks the dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckState(
      /**
       * Defines the expected state of the dialog
       */
      mDialogState?: object
    ): object;
  }
}

declare module "sap/fe/test/api/DialogCreateActions" {
  import DialogActions from "sap/fe/test/api/DialogActions";

  export default class DialogCreateActions extends DialogActions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Executes the `Create` action on the create dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteCreate(): object;
  }
}

declare module "sap/fe/test/api/DialogCreateAssertions" {
  import DialogAssertions from "sap/fe/test/api/DialogAssertions";

  export default class DialogCreateAssertions extends DialogAssertions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks the `Create` action on the dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckCreate(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
  }
}

declare module "sap/fe/test/api/DialogMessageActions" {
  import DialogActions from "sap/fe/test/api/DialogActions";

  export default class DialogMessageActions extends DialogActions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Executes the `Back` action on the message dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteBack(): object;
    /**
     * Executes the `Refresh` action on the message dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteRefresh(): object;
    /**
     * Selects the specified entry in the draft data loss popup.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSelectDraftDataLossOption(optionKey: any): object;
  }
}

declare module "sap/fe/test/api/DialogMessageAssertions" {
  import DialogAssertions from "sap/fe/test/api/DialogAssertions";

  export default class DialogMessageAssertions extends DialogAssertions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks the `Back` action on the message dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckBack(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks whether a certain message is shown in the dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckMessage(
      /**
       * Defines the expected state of the message, e.g. `{ title: "My message" }`
       */
      oMessage?: object
    ): object;
    /**
     * Checks the `Refresh` action on the dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckRefresh(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
  }
}

declare module "sap/fe/test/api/DialogType" {
  /**
   * Enum for supported dialog types.
   */
  enum DialogType {
    /**
     * A default dialog for action parameters
     */
    Action = "undefined",
    /**
     * A simple dialog supporting base checks and actions such as 'Confirm' and 'Cancel'
     */
    Confirmation = "undefined",
    /**
     * A dialog used for creating a new object
     */
    Create = "undefined",
    /**
     * A dialog used for showing an error message
     */
    Error = "undefined",
    /**
     * A message dialog for showing back-end messages
     */
    Message = "undefined",
    /**
     * A ValueHelp dialog that has a condition definition panel or a filterable selection table
     */
    ValueHelp = "undefined",
  }
  export default DialogType;
}

declare module "sap/fe/test/api/DialogValueHelpActions" {
  import DialogActions from "sap/fe/test/api/DialogActions";

  import { FilterFieldIdentifier } from "sap/fe/test/api/FilterBarAPI";

  export default class DialogValueHelpActions extends DialogActions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Adds a new condition.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iAddCondition(
      /**
       * The condition operator, like EQ, BT, LT, GT (see also {@link sap.ui.model.filter.FilterOperator})
       */
      sOperator: string,
      /**
       * The values to be set. If the operator requires more than one value, like BT (between), an array with
       * the two entries is expected
       */
      vValues: string | string[]
    ): object;
    /**
     * Changes an existing condition.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iChangeCondition(
      /**
       * The condition operator, like EQ, BT, LT, GT (see also {@link sap.ui.model.filter.FilterOperator})
       */
      sOperator: string,
      /**
       * The values to be set. If the operator requires more than one value like BT (between) an array with the
       * two entries is expected
       */
      vValues: string | string[],
      /**
       * The index of the condition to be altered. If not set, the index 0 is used
       */
      iConditionIndex?: number
    ): object;
    /**
     * Changes the value of a filter field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iChangeFilterField(
      /**
       * The identifier of the filter field
       */
      vFieldIdentifier: string | FilterFieldIdentifier,
      /**
       * The new target value.
       */
      vValue?: string,
      /**
       * Set to `true` to clear previously set filters, otherwise all previously set values will be kept
       */
      bClearFirst?: boolean
    ): object;
    /**
     * Changes the value of the search field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iChangeSearchField(
      /**
       * The new search value.
       */
      sSearchText?: string
    ): object;
    /**
     * Starts the search.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iExecuteSearch(): object;
    /**
     * Clicks the `Hide/Show Filters` button.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iExecuteShowHideFilters(): object;
    /**
     * Navigates to the `Define Conditions` tab.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iGoToDefineConditions(): object;
    /**
     * Navigates to the `Search and Select` tab.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iGoToSearchAndSelect(): object;
    /**
     * Removes an existing condition.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iRemoveCondition(
      /**
       * The index of the condition to be removed. If not set, the index 0 is used
       */
      iConditionIndex?: number
    ): object;
    /**
     * Resets the search field value.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iResetSearchField(): object;
    /**
     * Selects the specified rows.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iSelectRows(
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       *  Alternatively, the 0-based row index can be used.
       */
      vRowValues?: object | number,
      /**
       * Defines the expected state of the row
       */
      mState?: object
    ): object;
  }
}

declare module "sap/fe/test/api/DialogValueHelpAssertions" {
  import DialogAssertions from "sap/fe/test/api/DialogAssertions";

  import { FilterFieldIdentifier } from "sap/fe/test/api/FilterBarAPI";

  export default class DialogValueHelpAssertions extends DialogAssertions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks the `Define conditions` tab in a value help dialog. Please be aware that the tab control of the
     * toolbar is checked.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iCheckDefineConditions(
      /**
       * Defines the expected state of the filter bar
       */
      mState?: object
    ): object;
    /**
     * Checks the filter bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iCheckFilterBar(
      /**
       * Defines the expected state of the filter bar
       */
      mState?: object
    ): object;
    /**
     * Checks a filter field. If `vConditionValues` is `undefined`, the current condition values are ignored.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iCheckFilterField(
      /**
       * The identifier of the filter field
       */
      vFieldIdentifier: object | FilterFieldIdentifier,
      /**
       * The expected value(s) of the filter field
       */
      vConditionValues?: string | object | any[],
      /**
       * The expected operator
       */
      sOperator?: string,
      /**
       * Defines the expected state of the filter field
       */
      mState?: object
    ): object;
    /**
     * Checks the rows of a table. If `mRowValues` is provided, only rows with the corresponding values are
     * considered. If `iNumberOfRows` is provided, the number of rows are checked with respect to the provided
     * `mRowValues` (if set) or in total. If `iNumberOfRows` is omitted, it checks for at least one matching
     * row. If `mState` is provided, the row must be in the given state.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iCheckRows(
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       */
      mRowValues?: object,
      /**
       * The expected number of rows considering `mRowValues` and `mRowState`
       */
      iExpectedNumberOfRows?: number,
      /**
       * Defines the expected state of the target row
       */
      mState?: object
    ): object;
    /**
     * Checks the `Search and Select` tab in a value help dialog. Please be aware that the tab control of the
     * toolbar is checked.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iCheckSearchAndSelect(
      /**
       * Defines the expected state of the filter bar
       */
      mState?: object
    ): object;
    /**
     * Checks the search field in the filter bar. If the `sSearchText` parameter is `undefined`, the search
     * text is not validated.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iCheckSearchField(
      /**
       * The expected text in the search field
       */
      sSearchText?: string,
      /**
       * Defines the expected state of the search field
       */
      mState?: object
    ): object;
    /**
     * Checks the dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckState(
      /**
       * Defines the expected state of the dialog
       */
      mDialogState?: object
    ): object;
    /**
     * Checks the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iCheckTable(
      /**
       * Defines the expected state of the search field
       */
      mState?: object
    ): object;
  }
}

declare module "sap/fe/test/api/EditState" {
  /**
   * Enum for supported editing states.
   */
  enum EditState {
    /**
     * All
     */
    All = "undefined",
    /**
     * All (Hiding Drafts)
     */
    AllHidingDrafts = "undefined",
    /**
     * Locked by Another User
     */
    Locked = "undefined",
    /**
     * Own Draft
     */
    OwnDraft = "undefined",
    /**
     * Unchanged
     */
    Unchanged = "undefined",
    /**
     * Unsaved Changes by Another User
     */
    UnsavedChanges = "undefined",
  }
  export default EditState;
}

declare module "sap/fe/test/api/FilterBarActions" {
  import {
    default as FilterBarAPI,
    FilterFieldIdentifier,
  } from "sap/fe/test/api/FilterBarAPI";

  import EditState from "sap/fe/test/api/EditState";

  export default class FilterBarActions extends FilterBarAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Adds a field as a filter field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iAddAdaptationFilterField(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: string | FilterFieldIdentifier
    ): object;
    /**
     * Changes the editing status filter field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeEditingStatus(
      /**
       * Value of an edit state
       */
      sEditState: EditState
    ): object;
    /**
     * Changes the value of the defined filter field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeFilterField(
      /**
       * The identifier for the filter field
       */
      vFieldIdentifier: string | FilterFieldIdentifier,
      /**
       * The new target value
       */
      vValue?: string,
      /**
       * Set to `true` to clear previously set filters, otherwise all previously set values will be kept
       */
      bClearFirst?: boolean
    ): object;
    /**
     * Changes the search field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeSearchField(
      /**
       * The new search text
       */
      sSearchText?: string
    ): object;
    /**
     * Executes a keyboard shortcut.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteKeyboardShortcut(
      /**
       * Pattern for the shortcut
       */
      sShortcut: string,
      /**
       * The identifier of the field
       */
      vFieldIdentifier?: string | FilterFieldIdentifier
    ): object;
    /**
     * Executes the search with the current filters.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteSearch(): object;
    /**
     * Opens the value help of the given field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, that can be used for chaining statements
     */
    iOpenValueHelp(
      /**
       * The identifier of the filter field
       */
      vFieldIdentifier: string | FilterFieldIdentifier
    ): object;
    /**
     * Removes a field as a filter field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iRemoveAdaptationFilterField(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: string | FilterFieldIdentifier
    ): object;
    /**
     * Resets the search field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iResetSearchField(): object;
    /**
     * Saves a variant under the given name, or overwrites the current variant.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSaveVariant(
      /**
       * The name of the new variant. If no new variant name is defined, the current variant will be overwritten.
       */
      sVariantName?: string,
      /**
       * Saves the new variant with option "Set as Default".
       */
      bSetAsDefault?: string,
      /**
       * Saves the new variant with option "Apply Automatically".
       */
      bApplyAutomatically?: string
    ): object;
    /**
     * Selects the chosen variant.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSelectVariant(
      /**
       * The name of the variant to be selected
       */
      sVariantName: string
    ): object;
  }
}

declare module "sap/fe/test/api/FilterBarAPI" {
  export default class FilterBarAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Confirms the filter bar adaptation. It can be used in an action chain as well as in an assertion chain.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iConfirmFilterAdaptation(): object;
    /**
     * Opens the filter bar adaptation. It can be used in an action chain as well as in an assertion chain.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iOpenFilterAdaptation(): object;
  }
  /**
   * A filter field identifier
   */
  export type FilterFieldIdentifier = {
    /**
     * The name of the property
     */
    property: string;
  };
}

declare module "sap/fe/test/api/FilterBarAssertions" {
  import {
    default as FilterBarAPI,
    FilterFieldIdentifier,
  } from "sap/fe/test/api/FilterBarAPI";

  import EditState from "sap/fe/test/api/EditState";

  export default class FilterBarAssertions extends FilterBarAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks a field in the adaptation dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckAdaptationFilterField(
      /**
       * The identifier of the filter field, or its label
       */
      vFieldIdentifier: string | FilterFieldIdentifier,
      /**
       * Defines the expected state of the filter field in the adaptation dialog
       */
      mState?: object
    ): object;
    /**
     * Check the filter field for the editing status.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckEditingStatus(
      /**
       * The expected edit state value
       */
      sEditState?: EditState,
      /**
       * Defines the expected state of the filter field
       */
      mFieldState?: object
    ): object;
    /**
     * Checks whether the filter adaptation dialog is open.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckFilterAdaptation(): object;
    /**
     * Checks a filter field. If `vConditionValues` is `undefined`, the current condition values are ignored.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckFilterField(
      /**
       * The identifier of the filter field
       */
      vFieldIdentifier: object | FilterFieldIdentifier,
      /**
       * The expected value(s) of the filter field
       */
      vConditionValues?: string | object | any[],
      /**
       * The expected operator
       */
      sOperator?: string,
      /**
       * Defines the expected state of the filter field
       */
      mState?: object
    ): object;
    /**
     * Checks the search button.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckSearch(
      /**
       * Defines the expected state of the Go button
       */
      mState?: object
    ): object;
    /**
     * Checks the search field in the filter bar. If the `sSearchText` parameter is `undefined`, the search
     * text is not validated.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckSearchField(
      /**
       * The expected text in the search field
       */
      sSearchText?: string,
      /**
       * Defines the expected state of the search field
       */
      mState?: object
    ): object;
    /**
     * Checks the filter bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckState(
      /**
       * Defines the expected state of the filter bar
       */
      mFilterBarState?: object
    ): object;
    /**
     * Checks if the focus is on the filter field.
     *
     *
     * @returns The result of the {@link sa p.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeeFocusOnFilterField(
      /**
       * The field label
       */
      sFilterFieldLabel?: string
    ): object;
    /**
     * Checks if the focus is on the Go Button of the filter bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeeFocusOnGoButton(): object;
  }
}

declare module "sap/fe/test/api/FooterActionsBase" {
  import { ActionIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class FooterActionsBase {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Executes a footer action.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteAction(
      /**
       * The identifier of the action or its label
       */
      vActionIdentifier?: string | ActionIdentifier
    ): object;
  }
}

declare module "sap/fe/test/api/FooterActionsOP" {
  import FooterActionsBase from "sap/fe/test/api/FooterActionsBase";

  export default class FooterActionsOP extends FooterActionsBase {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Confirms the Cancel action when user clicks `Cancel` in draft mode.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iConfirmCancel(): object;
    /**
     * Executes the Apply action in the footer bar of a sub-object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteApply(): object;
    /**
     * Executes the Cancel action in the footer bar of the object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteCancel(): object;
    /**
     * Executes the Save or Create action in the footer bar of the object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteSave(): object;
  }
}

declare module "sap/fe/test/api/FooterAssertionsBase" {
  import { ActionIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class FooterAssertionsBase {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks the state of an action in the footer bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckAction(
      /**
       * The identifier of an action
       */
      vActionIdentifier: string | ActionIdentifier,
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the state of the footer bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckState(
      /**
       * Defines the expected state of the footer bar
       */
      mState?: object
    ): object;
  }
}

declare module "sap/fe/test/api/FooterAssertionsOP" {
  import FooterAssertionsBase from "sap/fe/test/api/FooterAssertionsBase";

  export default class FooterAssertionsOP extends FooterAssertionsBase {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks the state of the Apply action in the footer bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckApply(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the state of the Cancel action in the footer bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckCancel(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks for draft state 'Clear' in the footer bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckDraftStateClear(): object;
    /**
     * Checks for draft state 'Saved' in the footer bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckDraftStateSaved(): object;
    /**
     * Checks the state of the Save or Create action in the footer bar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckSave(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
  }
}

declare module "sap/fe/test/api/FormActions" {
  import FormAPI from "sap/fe/test/api/FormAPI";

  import { FieldIdentifier, ActionIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class FormActions extends FormAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Changes the value of a field within a form.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeField(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: FieldIdentifier | string,
      /**
       * The value to be set for the field
       */
      sValue?: string
    ): object;
    /**
     * Toggles the value of a checkbox within a form.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iClickCheckBox(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: FieldIdentifier
    ): object;
    /**
     * Clicks a link within a form.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iClickLink(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: FieldIdentifier | string
    ): object;
    /**
     * Executes an action assigned to a form in a subsection.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteAction(
      /**
       * The identifier of the action or its label
       */
      vActionIdentifier: string | ActionIdentifier
    ): object;
    /**
     * Executes the Show Less action of a form in a subsection.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteShowLess(): object;
    /**
     * Executes the Show More action of a form in a subsection.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteShowMore(): object;
    /**
     * Opens the value help of the given field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function that can be used for chaining statements
     */
    iOpenValueHelp(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: string | FieldIdentifier
    ): object;
  }
}

declare module "sap/fe/test/api/FormAPI" {
  export default class FormAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();
  }
  /**
   * A form identifier
   */
  export type FormIdentifier = {
    /**
     * The facet ID
     */
    section: string;
    /**
     * The fieldgroup ID
     */
    fieldGroup: string;
    /**
     * Is it about the editable header facet
     */
    isHeaderFacet?: boolean;
  };
}

declare module "sap/fe/test/api/FormAssertions" {
  import FormAPI from "sap/fe/test/api/FormAPI";

  import { ActionIdentifier, FieldIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class FormAssertions extends FormAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks the state of an action in a subsection.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckAction(
      /**
       * The identifier of an action
       */
      vActionIdentifier: string | ActionIdentifier,
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the content and state of a field within a form.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckField(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: string | FieldIdentifier,
      /**
       * Expected value(s) of the field. if passed as an object, the following pattern will be considered:
       * ```javascript
       *
       * {
       *     value: , 		// optional
       *     description:  	// optional
       * }
       * ```
       */
      vValue?: string | any[] | object,
      /**
       * Defines the expected state of the field
       */
      mState?: object
    ): object;
    /**
     * Checks the field is a link with the given text and state.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckLink(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: string | FieldIdentifier,
      /**
       * The link text
       */
      sText?: string,
      /**
       * Defines the expected state of the field
       */
      mState?: object
    ): object;
    /**
     * Checks the Show Less action of a form in a subsection.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckShowLess(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the Show More action of a form in a subsection.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckShowMore(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the state of the form.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckState(
      /**
       * Defines the expected state of the form
       */
      mState: object
    ): object;
  }
}

declare module "sap/fe/test/api/HeaderActions" {
  import { ActionIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class HeaderActions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Executes an action in the header toolbar of an object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteAction(
      /**
       * The identifier of the action
       */
      vActionIdentifier: string | ActionIdentifier
    ): object;
    /**
     * Executes the Delete action in the header toolbar of an object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteDelete(): object;
    /**
     * Executes the Draft/Saved Version action in the header toolbar of an object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteDraftAndSavedVersionSwitch(): object;
    /**
     * Executes the Edit action in the header toolbar of an object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteEdit(): object;
    /**
     * Executes an action in the drop-down menu that is currently open.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteMenuAction(
      /**
       * The label of the action or its state
       */
      vAction: string | object
    ): object;
    /**
     * Navigates to the next sub-object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecutePaginatorDown(): object;
    /**
     * Navigates to the previous sub-object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecutePaginatorUp(): object;
    /**
     * Executes the Related Apps action in the header toolbar of an object page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteRelatedApps(): object;
    /**
     * Executes an action in the selection list that is currently open.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteSelectListAction(
      /**
       * The label of the action or its state
       */
      vAction: string | object
    ): object;
  }
}

declare module "sap/fe/test/api/HeaderActionsLR" {
  import { ActionIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class HeaderActionsLR {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Executes an action in the header toolbar of a list report.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteAction(
      /**
       * The identifier of the action
       */
      vActionIdentifier?: string | ActionIdentifier
    ): object;
  }
}

declare module "sap/fe/test/api/HeaderAssertions" {
  import { ActionIdentifier, FieldIdentifier } from "sap/fe/test/api/BaseAPI";

  import { HeaderFacetIdentifier } from "sap/fe/test/api/HeaderAPI";

  export default class HeaderAssertions {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks an action in the header toolbar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckAction(
      /**
       * The identifier of the action
       */
      vActionIdentifier: string | ActionIdentifier,
      /**
       * Defines the expected state of the button. Checking for visible=false means to check a hidden action.
       */
      mState?: object
    ): object;
    /**
     * Checks the `Delete` action in the header toolbar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckDelete(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the `Edit` action in the header toolbar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckEdit(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks a field within a field group in the object page header.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckFieldInFieldGroup(
      /**
       * The identifier of the field
       */
      vFieldIdentifier: FieldIdentifier | string,
      /**
       * The value to check. If it is an array, the first entry is considered as the value and the second as the
       * description. If it is an object it must follow this pattern:
       * ```javascript
       *
       * 	{
       * 		value: , 		// optional
       * 		description:  	// optional
       * 	}
       * ```
       */
      vValue?: string | any[] | object,
      /**
       * Defines the expected state of the field
       */
      mState?: object
    ): object;
    /**
     * Checks the custom facet in the object page header.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckHeaderFacet(
      /**
       * The Identifier of the header facet
       */
      vFacetIdentifier: HeaderFacetIdentifier,
      /**
       * Defines the expected state
       */
      mState?: object
    ): object;
    /**
     * Checks an action in the popover that is currently open.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckMenuAction(
      /**
       * The state map or label of the action
       */
      vAction: object | string
    ): object;
    /**
     * Checks the paginator down button.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckPaginatorDown(
      /**
       * Defines the expected state of the button
       */
      mState: object
    ): object;
    /**
     * Checks the paginator up button.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckPaginatorUp(
      /**
       * Defines the expected state of the button
       */
      mState: object
    ): object;
    /**
     * Checks the `Related Apps` action in the header toolbar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckRelatedApps(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the `Save as Tile` action.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckSaveAsTile(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks `Send Email` action.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckSendEmail(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the title and description of the object page.
     *
     * If either title or description is `undefined`, it will not be checked.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckTitle(
      /**
       * Title of the object page header
       */
      sTitle?: string,
      /**
       * Description of the object page header
       */
      sDescription?: string
    ): object;
  }
}

declare module "sap/fe/test/api/HeaderAssertionsLR" {
  import { ActionIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class HeaderAssertionsLR {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks an action of the header toolbar.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckAction(
      /**
       * The identifier of the action, or its label
       */
      vActionIdentifier?: string | ActionIdentifier,
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the `Save as Tile` action.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckSaveAsTile(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the `Send Email` action.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckSendEmail(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
  }
}

declare module "sap/fe/test/api/HeaderAPI" {
  /**
   * An identifier for the header facet
   */
  export type HeaderFacetIdentifier = {
    /**
     * The identifier of the facet
     */
    facetId: string;
    /**
     * Defines whether the facet is a collection facet (default: `false`)
     */
    collection?: boolean;
    /**
     * Defines whether the facet is a custom header facet (default: `false`)
     */
    custom?: boolean;
  };
}

declare module "sap/fe/test/api/TableActions" {
  import {
    default as TableAPI,
    ColumnIdentifier,
  } from "sap/fe/test/api/TableAPI";

  import { SortOrder } from "sap/ui/core/library";

  import { ActionIdentifier } from "sap/fe/test/api/BaseAPI";

  export default class TableActions extends TableAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Adds a field as a column to the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iAddAdaptationColumn(
      /**
       * The identifier of the column field, or its label
       */
      vColumnIdentifier: string | ColumnIdentifier
    ): object;
    /**
     * Aggregates the table entries by the specified column.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iAggregateByColumn(
      /**
       * The identifier of the column field, its label or index
       */
      vColumnIdentifier: string | ColumnIdentifier | number,
      /**
       * The target field to group on. If undefined, parameter vColumnIdentifier is used as label to identify
       * the toggle button.
       */
      sFieldLabel?: string
    ): object;
    /**
     * Adds a filter condition to the filter field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeFilterField(
      /**
       * The identifier of the column
       */
      vColumnIdentifier: string | ColumnIdentifier,
      /**
       * Defines the value of the filter field condition
       */
      vValue: string | object,
      /**
       * Set to `true` to clear previously set filters, otherwise all previously set values will be kept
       */
      bClearFirst?: boolean
    ): object;
    /**
     * Changes the specified row. The given value map must match exactly one row.
     *
     * If only one parameter is provided, it must be the `mTargetValues` and `mRowValues` is considered undefined.
     * If `vRowValues` are not defined, then the targetValues are inserted in the creationRow.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeRow(
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       *  Alternatively, the 0-based row index can be used.
       */
      vRowValues: object | number,
      /**
       * A map of columns (either name or index) to its new value. The columns do not need to match the ones defined
       * in `vRowValues`.
       */
      mTargetValues: object,
      /**
       * If true, we keep the focus on the modified cell and don't press enter to validate the input
       */
      bInputNotFinalized: boolean
    ): object;
    /**
     * Changes the specified row. The given value map must match exactly one row.
     *
     * If only one parameter is provided, it must be the `mTargetValues` and `mRowValues` is considered undefined.
     * If `vRowValues` are not defined, then the targetValues are inserted in the creationRow.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeRow(
      /**
       * A map of columns (either name or index) to its new value. The columns do not need to match the ones defined
       * in `vRowValues`.
       */
      mTargetValues: object,
      /**
       * If true, we keep the focus on the modified cell and don't press enter to validate the input
       */
      bInputNotFinalized: boolean
    ): object;
    /**
     * Changes the search field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeSearchField(
      /**
       * The new search text
       */
      sSearchText?: string
    ): object;
    /**
     * Adds a field to the sorting of the table via the sort dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iChangeSortOrder(
      /**
       * The identifier of the column field, or its label
       */
      vColumnIdentifier: string | ColumnIdentifier,
      /**
       * The sort order, default is {@link sap.ui.core.SortOrder.Ascending}
       */
      sSortOrder?: SortOrder | keyof typeof SortOrder
    ): object;
    /**
     * Collapses a row corresponding to a visual group.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCollapseGroupRow(
      /**
       * The level of the group row to be collapsed (1-based)
       */
      iLevel: number,
      /**
       * The title of the group row to be collapsed
       */
      sTitle: string
    ): object;
    /**
     * Executes an action on the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteAction(
      /**
       * The identifier of the action, or its label
       */
      vActionIdentifier?: string | ActionIdentifier
    ): object;
    /**
     * Executes the `Create` action on the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteCreate(): object;
    /**
     * Executes the `Delete` action on the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteDelete(): object;
    /**
     * Executes the `Fullscreen` action on the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteFullScreen(): object;
    /**
     * Executes an action that is available in a certain column within a table row.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteInlineAction(
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       *  Alternatively, the 0-based row index can be used.
       */
      vRowValues: object | number,
      /**
       * The column name, label or index
       */
      vColumn: string | number
    ): object;
    /**
     * Executes an action that is available in a certain column within a table row.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteInlineAction(
      /**
       * The column name, label or index
       */
      vColumn: string | number
    ): object;
    /**
     * Executes the action to create a row in the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteInlineCreate(): object;
    /**
     * Executes a keyboard shortcut on the table or a cell control. If only `sShortcut` is defined, the shortcut
     * is executed on the table directly. If additionally `vRowValues` and `vColumn` are defined, the shortcut
     * is executed on table cell level.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteKeyboardShortcut(
      /**
       * The shortcut pattern
       */
      sShortcut: string,
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       *  Alternatively, the 0-based row index can be used.
       */
      vRowValues: object | number,
      /**
       * The column name, label or index
       */
      vColumn: string | number
    ): object;
    /**
     * Executes a keyboard shortcut on the table or a cell control. If only `sShortcut` is defined, the shortcut
     * is executed on the table directly. If additionally `vRowValues` and `vColumn` are defined, the shortcut
     * is executed on table cell level.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteKeyboardShortcut(
      /**
       * The shortcut pattern
       */
      sShortcut: string,
      /**
       * The column name, label or index
       */
      vColumn: string | number
    ): object;
    /**
     * Executes an action form the drop-down menu that is currently open.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteMenuAction(
      /**
       * The label of the action or its state
       */
      vAction: string | object
    ): object;
    /**
     * Executes the `Show/Hide details` action on the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExecuteShowHideDetails(
      /**
       * Optional parameter to enforce a certain state (showing details yes/no corresponds to true/false); if
       * not set, state is toggled
       */
      bShowDetails?: boolean
    ): object;
    /**
     * Expands a row corresponding to a visual group.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iExpandGroupRow(
      /**
       * The level of the group row to be expanded (1-based)
       */
      iLevel: number,
      /**
       * The title of the group row to be expanded
       */
      sTitle: string
    ): object;
    /**
     * Groups the table entries by the specified column.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iGroupByColumn(
      /**
       * The identifier of the column field, its label or index
       */
      vColumnIdentifier: string | ColumnIdentifier | number,
      /**
       * The target field for grouping. If undefined, parameter vColumnIdentifier is used as label to identify
       * the toggle button.
       */
      sFieldLabel?: string
    ): object;
    /**
     * Pastes data into the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iPasteData(
      /**
       * The data to be pasted
       */
      aData: string[][]
    ): object;
    /**
     * Presses the control in the table cell.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iPressCell(
      /**
       * Specifies the target row by column-value map, e.g.
       * ```javascript
       *
       * {
       *     0: "Max",
       *     "Last Name": "Mustermann"
       * }
       * ```
       */
      mRowValues: object,
      /**
       * The column name, label or index
       */
      vColumn: string | number
    ): object;
    /**
     * Presses the control in the table cell.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iPressCell(
      /**
       * The column name, label or index
       */
      vColumn: string | number
    ): object;
    /**
     * Clicks the specified row.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iPressRow(
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       *  Alternatively, the 0-based row index can be used.
       */
      vRowValues?: object | number
    ): object;
    /**
     * Removes a field as a column from the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iRemoveAdaptationColumn(
      /**
       * The identifier of the column field, or its label
       */
      vColumnIdentifier: string | ColumnIdentifier
    ): object;
    /**
     * Removes the variant of the given name.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iRemoveVariant(
      /**
       * The name of the variant to be removed
       */
      sVariantName: string
    ): object;
    /**
     * Resets the search field.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iResetSearchField(): object;
    /**
     * Saves a variant under the given name, or overwrites the current one.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSaveVariant(
      /**
       * The name of the new variant. If omitted, the current variant will be overwritten
       */
      sVariantName?: string
    ): object;
    /**
     * Selects all rows in a table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSelectAllRows(): object;
    /**
     * Selects a quick-filter item on the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSelectQuickFilterItem(
      /**
       * If passed as an object, the following pattern will be considered:
       * ```javascript
       *
       * 	{
       * 		:
       *  }
       * ```
       *  If using a plain string as the identifier, it is considered the item label
       */
      vItemIdentifier?: object | string
    ): object;
    /**
     * Selects the specified rows.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSelectRows(
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       *  Alternatively, the 0-based row index can be used.
       */
      vRowValues?: object | number,
      /**
       * Defines the expected state of the row
       */
      mState?: object
    ): object;
    /**
     * Selects the chosen variant.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSelectVariant(
      /**
       * The name of the variant to be selected
       */
      sVariantName: string
    ): object;
    /**
     * Sets the variant as the default.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSetDefaultVariant(
      /**
       * The name of the variant to be set as the default variant. If omitted, the Standard variant is set as
       * the default
       */
      sVariantName: string
    ): object;
    /**
     * Sorts the table entries by the specified column.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSortByColumn(
      /**
       * The identifier of the column field, its label or index
       */
      vColumnIdentifier: string | ColumnIdentifier | number,
      /**
       * The target field for sorting in case the field label differs from the column label or in case of a complex
       * property
       */
      sFieldLabel?: string,
      /**
       * Sorting (true=descending, false=ascending (default))
       */
      bDescending?: boolean
    ): object;
  }
}

declare module "sap/fe/test/api/TableAssertions" {
  import {
    default as TableAPI,
    ColumnIdentifier,
  } from "sap/fe/test/api/TableAPI";

  import { ActionIdentifier } from "sap/fe/test/api/BaseAPI";

  import { SortOrder } from "sap/ui/core/library";

  export default class TableAssertions extends TableAPI {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Checks the state of a table action.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckAction(
      /**
       * The identifier of the action, or its label
       */
      vActionIdentifier: string | ActionIdentifier,
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks a field in the adaptation dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckAdaptationColumn(
      /**
       * The identifier of the column, or its label
       */
      vColumnIdentifier: string | ColumnIdentifier,
      /**
       * Defines the expected state of the field control in the adaptation dialog
       */
      mState?: object
    ): object;
    /**
     * Checks the state of the cells of a table. This function can only be used for simple content within a
     * table cell. Checking complex content like combined fields within a cell is not supported.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckCells(
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       *  Alternatively, the 0-based row index can be used.
       */
      vRowValues: object | number,
      /**
       * A map of columns and their state. The map looks like
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: {
       * 			header: "My header"
       * 		}
       * 	}
       * ```
       */
      mColumnStateMap: object
    ): object;
    /**
     * Checks the state of the cells of a table. This function can only be used for simple content within a
     * table cell. Checking complex content like combined fields within a cell is not supported.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckCells(
      /**
       * A map of columns and their state. The map looks like
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: {
       * 			header: "My header"
       * 		}
       * 	}
       * ```
       */
      mColumnStateMap: object
    ): object;
    /**
     * Checks whether the adaptation button is available for the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckColumnAdaptation(): object;
    /**
     * Checks whether the column adaptation dialog is open.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckColumnAdaptationDialog(): object;
    /**
     * Checks whether the filter button is available for the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckColumnFiltering(): object;
    /**
     * Checks the state of the columns of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckColumns(
      /**
       * The expected number of columns
       */
      iExpectedNumberOfColumns?: number,
      /**
       * A map of columns and their expected state. The map looks like
       * ```javascript
       *
       * 	{
       * 		<columnName | columnLabel | columnIndex>: {
       * 			header: "My header"
       * 		}
       * 	}
       * ```
       */
      mColumnStateMap?: object
    ): object;
    /**
     * Checks whether the sort button is available for the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckColumnSorting(): object;
    /**
     * Checks the `Create` action of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckCreate(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks the state of the CreationRow button in the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckCreationRow(
      /**
       * Defines the expected row values. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       */
      mRowValues?: object,
      /**
       * Defines the expected state of the target row
       */
      mState?: object
    ): object;
    /**
     * Checks the `Delete` action of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckDelete(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks whether the export button is available for the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckExport(): object;
    /**
     * Checks, if a filter field is available in the filter dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckFilterField(
      /**
       * The identifier of the field, or its label
       */
      vColumnIdentifier: string | ColumnIdentifier,
      /**
       * Defines the expected state of the field control in the filter dialog
       */
      mState?: object
    ): object;
    /**
     * Checks the `Fullscreen` action of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckFullScreen(
      /**
       * Defines the expected state of the button
       */
      mState?: object
    ): object;
    /**
     * Checks an action in the drop-down menu that is currently open.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckMenuAction(
      /**
       * The label of the action, or its state
       */
      vAction: object | string
    ): object;
    /**
     * Checks a messageStrip on a table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckMessageStrip(
      /**
       * Defines the expected properties of the messageStrip in the table
       */
      mProperty?: object
    ): object;
    /**
     * Checks whether the paste button is available for the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckPaste(): object;
    /**
     * Checks the quick filter action of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckQuickFilter(
      /**
       * Defines the expected state of the control
       */
      mState?: object
    ): object;
    /**
     * Checks the number of items in the quick-filter menu.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckQuickFilterItems(
      /**
       * The expected number of quick-filter items
       */
      iExpectedNumberOfItems: number
    ): object;
    /**
     * Checks the rows of a table. If `mRowValues` is provided, only rows with the corresponding values are
     * considered. If `iNumberOfRows` is provided, the number of rows are checked with respect to the provided
     * `mRowValues` (if set) or in total. If `iNumberOfRows` is omitted, it checks for at least one matching
     * row. If `mState` is provided, the row must be in the given state.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckRows(
      /**
       * Defines the row values of the target row. The pattern is:
       * ```javascript
       *
       * 	{
       * 		<column-name-or-index>: <expected-value>
       *  }
       * ```
       */
      mRowValues?: object,
      /**
       * The expected number of rows considering `mRowValues` and `mRowState`
       */
      iExpectedNumberOfRows?: number,
      /**
       * Defines the expected state of the target row
       */
      mState?: object,
      /**
       * Defines the cells of the identified rows which are expected to the hidden (visible=false). You can also
       * use test function iCheckCells to explicitly check a specific cell of a row.
       */
      vHiddenCells?: any[]
    ): object;
    /**
     * Checks the search field in the table toolbar. If the `sSearchText` parameter is `undefined`, the search
     * text is not validated.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function that can be used for chaining statements
     */
    iCheckSearchField(
      /**
       * The expected text in the search field
       */
      sSearchText?: string,
      /**
       * Defines the expected state of the search field
       */
      mState?: object
    ): object;
    /**
     * Checks a field in the sorting dialog.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckSortOrder(
      /**
       * The identifier of the column, or its label
       */
      vColumnIdentifier: string | ColumnIdentifier,
      /**
       * The sort order of the column, default is {@link sap.ui.core.SortOrder.Ascending}
       */
      sSortOrder?: SortOrder | keyof typeof SortOrder,
      /**
       * Defines if the order is checked via sorting dialog, or via the column itself
       */
      bCheckPersonalization?: boolean
    ): object;
    /**
     * Checks the state of the table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckState(
      /**
       * Defines the expected state of the table
       */
      mState?: object
    ): object;
    /**
     * Checks if the focus is on a table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeeFocusOnHeader(): object;
    /**
     * Checks if the focus is on a table row.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeeFocusOnRow(
      /**
       * The expected focused row
       */
      iRowIndex: number
    ): object;
  }
}

declare module "sap/fe/test/BaseActions" {
  import Opa5 from "sap/ui/test/Opa5";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * All common actions (`When`) for all Opa tests are defined here.
   */
  export default class BaseActions extends Opa5 {
    /**
     * Constructs a new {@link sap.fe.test.Opa5} instance.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.fe.test.BaseActions with name `sClassName` and enriches it with the
     * information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.test.Opa5.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, BaseActions>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.fe.test.BaseActions.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Closes the open popover via function.
     *
     * NOTE: This function ensures that a certain UI state is maintained in some exceptional cases. This function
     * isn't usually called directly in a journey.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iClosePopover(): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Simulates the pressing of the Esc key for the element in focus.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iPressEscape(): /* was: sap.ui.test.OpaBuilder */ any;
  }
}

declare module "sap/fe/test/BaseArrangements" {
  import Opa5 from "sap/ui/test/Opa5";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * All common arrangements (`Given`) for all Opa tests are defined here.
   */
  export default class BaseArrangements extends Opa5 {
    /**
     * Constructs a new {@link sap.fe.test.Opa5} instance.
     */
    constructor(
      /**
       * The settings object required for launching the application
       */
      mSettings: {
        /**
         * The URL to the launching page (usually a FLP.html)
         */
        launchUrl: string;
        /**
         * The URL launch parameters
         */
        launchParameters: Record<string, string | boolean>;
      }
    );

    /**
     * Creates a new subclass of class sap.fe.test.BaseArrangements with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.test.Opa5.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, BaseArrangements>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.fe.test.BaseArrangements.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Resets the mock data.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iResetMockData(
      /**
       * The application id or a uri pointing to the service. One of the following properties needs to be provided
       * within the parameter:
       * ```javascript
       *
       * 	{
       * 		:
       * 		:
       *  }
       * ```
       *  NOTE: When passing the AppId, the application needs to be running already. Calling the function
       * before the application is started can be done by passing the ServiceUri.
       */
      oAppInfo?: object
    ): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Resets the test data.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iResetTestData(): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Simulates a refresh of the page by tearing it down and then restarting it with the very same hash.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iRestartMyApp(): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Starts the app in an IFrame, using the `launchUrl` and `launchParameters` provided via the settings object
     * of the {@link sap.fe.test.BaseArrangements#constructor}.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iStartMyApp(
      /**
       * The app hash
       */
      sAppHash?: string,
      /**
       * A map with additional URL parameters
       */
      mInUrlParameters?: object
    ): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Tears down the current application.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iTearDownMyApp(): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Clears the browser's local storage and session storage.
     *
     * NOTE: The function itself is not meant to be called directly within a journey. Instead, it can be overwritten
     * to add custom clean-up functionality when calling {@link sap.fe.test.BaseArrangements#iResetTestData}.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns A `Promise` object
     */
    resetTestData(): Promise<any>;
  }
}

declare module "sap/fe/test/BaseAssertions" {
  import Opa5 from "sap/ui/test/Opa5";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * All common assertions (`Then`) for all Opa tests are defined here.
   */
  export default class BaseAssertions extends Opa5 {
    /**
     * Constructs a new {@link sap.fe.test.Opa5} instance.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.fe.test.BaseAssertions with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.test.Opa5.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, BaseAssertions>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.fe.test.BaseAssertions.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Checks whether a {@link sap.m.IllustratedMessage} with given message is shown.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeeIllustratedMessage(
      /**
       * The message shown on the page
       */
      sMessage: string
    ): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Checks whether a {@link sap.m.MessagePage} with given message is shown.
     *
     * @deprecated As of version 1.118. use iSeePage() && iSeeIllustratedMessage(sMessage) instead
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeeMessagePage(
      /**
       * The message shown on the message page
       */
      sMessage: string
    ): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Checks whether a {@link sap.m.MessageToast} with the given text is shown.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeeMessageToast(
      /**
       * The text shown in the MessageToast
       */
      sText: string
    ): /* was: sap.ui.test.OpaBuilder */ any;
    /**
     * Checks whether a {@link sap.m.Page} is shown.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeePage(): /* was: sap.ui.test.OpaBuilder */ any;
  }
}

declare module "sap/fe/test/JourneyRunner" {
  import BaseObject from "sap/ui/base/Object";

  import ListReport from "sap/fe/test/ListReport";

  import ObjectPage from "sap/fe/test/ObjectPage";

  import { Config } from "sap/ui/test/Opa5";

  import Opa from "sap/ui/test/Opa";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * A JourneyRunner for executing integration tests with given settings.
   */
  export default class JourneyRunner extends BaseObject {
    /**
     * Constructs a new JourneyRunner instance.
     */
    constructor(
      /**
       * The settings object
       */
      mSettings?: {
        /**
         * The available Opa pages
         */
        pages?: Record<string, ListReport | ObjectPage>;
        /**
         * The Opa configuration applied via {@link sap.ui.test.Opa5#sap.ui.test.Opa5.extendConfig}
         */
        opaConfig?: Config;
        /**
         * The URL to the launching page (usually a FLP.html)
         */
        launchUrl?: string;
        /**
         * The URL launch parameters
         */
        launchParameters?: Record<string, string | boolean>;
        /**
         * If false (default), only one JourneyRunner is executed at a time
         */
        async?: boolean;
      }
    );

    /**
     * Creates a new subclass of class sap.fe.test.JourneyRunner with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, JourneyRunner>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Gets the global journey runner instance.
     *
     *
     * @returns The global default {@link sap.fe.test.JourneyRunner} instance
     */
    static getDefaultRunner(): object;
    /**
     * Returns a metadata object for class sap.fe.test.JourneyRunner.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Static function to run the default runner with given parameters. Shortcut for
     * ```javascript
     * JourneyRunner.getDefaultRunner().run(mSettings, Journey1, Journey2, ...)```
     *  See {@link sap.fe.test.JourneyRunner#run} for parameter details.
     */
    static run(): void;
    /**
     * Sets the global journey runner instance.
     */
    static setDefaultRunner(
      /**
       * Defines the global default {@link sap.fe.test.JourneyRunner} instance
       */
      oDefaultRunner: JourneyRunner
    ): void;
    /**
     * Returns the base action instance to be used for {@link sap.ui.test.Opa5#sap.ui.test.Opa5.extendConfig }
     * `actions` setting.
     *
     * This function is only used if `actions` is not defined via the runner settings. It is meant to be overridden
     * by a custom JourneyRunner implementation to provide custom base actions.
     *
     * By default, an instance of {@link sap.fe.test.BaseActions} will be returned.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns An Opa instance for the base actions
     */
    getBaseActions(): Opa;
    /**
     * Returns the base arrangements instance to be used for {@link sap.ui.test.Opa5#sap.ui.test.Opa5.extendConfig }
     * `arrangements` setting.
     *
     * This function is only used if `arrangements` is not defined via the runner settings. It is meant to be
     * overridden by a custom JourneyRunner implementation to provide custom base assertions.
     *
     * By default, an instance of {@link sap.fe.test.BaseArrangements} will be returned.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns An Opa instance for the base arrangements
     */
    getBaseArrangements(
      /**
       * The settings object of the runner instance
       */
      mSettings: object
    ): Opa;
    /**
     * Returns the base assertions instance to be used for {@link sap.ui.test.Opa5#sap.ui.test.Opa5.extendConfig }
     * `assertions` setting.
     *
     * This function is only used if `assertions` is not defined via the runner settings. It is meant to be
     * overridden by a custom JourneyRunner implementation to provide custom base assertions.
     *
     * By default, an instance of {@link sap.fe.test.BaseAssertions} will be returned.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns An Opa instance for the base assertions
     */
    getBaseAssertions(): Opa;
    /**
     * Executes the journeys in the given order.
     *
     * The settings provided as first parameter are merged into the base settings of the JourneyRunner instance.
     *
     *
     * @returns A `Promise` that is resolved after all tests have been executed
     */
    run(
      /**
       * The settings object for the tests to run. Overrides instance settings
       */
      mSettings: {
        /**
         * The available Opa pages
         */
        pages?: Record<string, ListReport | ObjectPage>;
        /**
         * The Opa configuration applied via {@link sap.ui.test.Opa5#sap.ui.test.Opa5.extendConfig}
         */
        opaConfig?: Config;
        /**
         * The URL to the launching page (usually a FLP.html)
         */
        launchUrl?: string;
        /**
         * The URL launch parameters
         */
        launchParameters?: Record<string, string | boolean>;
        /**
         * If false (default), only one JourneyRunner is executed at a time
         */
        async?: boolean;
      },
      /**
       * The journeys to be executed. If a journey is represented as a string, it will be interpreted as a module
       * path to the file that should be loaded. Else it is expected to be a function. Alternatively, instead
       * of being wrapped in an array, the journeys can also be given as separate parameters:
       * ```javascript
       * .run(mSettings, Journey1, Journey2, ...)```
       */
      _vJourneys: Array<Function> | string[]
    ): object;
    /**
     * Executes the journeys in the given order.
     *
     * The settings provided as first parameter are merged into the base settings of the JourneyRunner instance.
     *
     *
     * @returns A `Promise` that is resolved after all tests have been executed
     */
    run(
      /**
       * The journeys to be executed. If a journey is represented as a string, it will be interpreted as a module
       * path to the file that should be loaded. Else it is expected to be a function. Alternatively, instead
       * of being wrapped in an array, the journeys can also be given as separate parameters:
       * ```javascript
       * .run(mSettings, Journey1, Journey2, ...)```
       */
      _vJourneys: Array<Function> | string[]
    ): object;
  }
}

declare module "sap/fe/test/ListReport" {
  import {
    default as TemplatePage,
    actions as actions1,
    assertions as assertions1,
  } from "sap/fe/test/TemplatePage";

  import { ViewIdentifier, TableIdentifier } from "sap/fe/test/api/BaseAPI";

  import FilterBarActions from "sap/fe/test/api/FilterBarActions";

  import HeaderActionsLR from "sap/fe/test/api/HeaderActionsLR";

  import TableActions from "sap/fe/test/api/TableActions";

  import FilterBarAssertions from "sap/fe/test/api/FilterBarAssertions";

  import HeaderAssertionsLR from "sap/fe/test/api/HeaderAssertionsLR";

  import TableAssertions from "sap/fe/test/api/TableAssertions";

  /**
   * Provides a test page definition for a list report page with the corresponding parameters.
   *
   * Sample usage:
   * ```javascript
   *
   * var oListReportDefinition = new ListReport({ appId: "MyApp", componentId: "MyListReportId", entitySet: "MyEntitySet" });
   * ```
   */
  export default class ListReport extends TemplatePage {
    /**
     * Constructs a new ListReport definition.
     */
    constructor(
      /**
       * The required parameters
       */
      oPageDefinition: {
        /**
         * The app id (defined in the manifest root)
         */
        appId: string;
        /**
         * The component id (defined in the target section for the list report within the manifest)
         */
        componentId: string;
        /**
         * The entitySet (optional)(defined in the settings of the corresponding target component within the manifest)
         */
        entitySet: string;
        /**
         * The contextPath (optional)(defined in the settings of the corresponding target component within the manifest)
         */
        contextPath: string;
      },
      /**
       * Additional custom page functions, provided in an object containing `actions` and `assertions`
       */
      ..._aInAdditionalPageDefinitions: object[]
    );
    /**
     * ListReport actions
     */
    static actions: typeof TemplatePage.actions & actions;

    /**
     * ListReport assertions
     */
    static assertions: typeof TemplatePage.assertions & assertions;
  }
  /**
   * ListReport actions
   */
  interface actions extends actions1 {
    /**
     * Collapses or expands the page header.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCollapseExpandPageHeader(
      /**
       * Defines whether the header should be collapsed, else it is expanded (default)
       */
      bCollapse?: boolean
    ): object;
    /**
     * Navigates to or focuses on the defined view of a Multiple Views List Report table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iGoToView(
      /**
       * The identifier of a view as defined in the manifest file, or its label passed as a string if passed as
       * an object, the following pattern will be considered:
       * ```javascript
       *
       * {
       *     key: ,
       * }
       * ```
       *  Depending on property 'keepPreviousPersonalization' in the manifest the key could be set different
       * within the id of the table. If necessary please check the UI control tree within the debugger.
       */
      vViewIdentifier: string | ViewIdentifier
    ): object;
    /**
     * Returns a {@link sap.fe.test.api.FilterBarActions} instance.
     *
     *
     * @returns The available filter bar actions
     */
    onFilterBar(): FilterBarActions;
    /**
     * Returns a {@link sap.fe.test.api.HeaderActionsLR} instance.
     *
     *
     * @returns The available header actions
     */
    onHeader(): HeaderActionsLR;
    /**
     * Returns a {@link sap.fe.test.api.TableActions} instance for the specified table.
     *
     *
     * @returns The available table actions
     */
    onTable(
      /**
       * The identifier of the table, or its header title
       */
      vTableIdentifier: string | TableIdentifier
    ): TableActions;
  }
  export const actions: actions;

  /**
   * ListReport assertions
   */
  interface assertions extends assertions1 {
    /**
     * Checks the view of a Multiple View List Report table.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckView(
      /**
       * The identifier of a view as defined in the manifest file, or its label passed as a string if passed as
       * an object, the following pattern will be considered:
       * ```javascript
       *
       * {
       *     key: ,
       * }
       * ```
       *  Depending on property 'keepPreviousPersonalization' in the manifest, the key could be set differently
       * within the id of the table. If necessary please check the UI control tree within the debugger.
       */
      vViewIdentifier: string | ViewIdentifier,
      /**
       * An object containing properties of the view to be checked Example:
       * ```javascript
       *
       * {
       *     count: ,
       * }
       * ```
       */
      mState: object
    ): object;
    /**
     * Returns a {@link sap.fe.test.api.FilterBarAssertions} instance.
     *
     *
     * @returns The available filter bar assertions
     */
    onFilterBar(): FilterBarAssertions;
    /**
     * Returns a {@link sap.fe.test.api.HeaderAssertionsLR} instance.
     *
     *
     * @returns The available header assertions
     */
    onHeader(): HeaderAssertionsLR;
    /**
     * Returns a {@link sap.fe.test.api.TableAssertions} instance for the specified table.
     *
     *
     * @returns The available table assertions
     */
    onTable(
      /**
       * The identifier of the table, or its header title
       */
      vTableIdentifier: string | TableIdentifier
    ): TableAssertions;
  }
  export const assertions: assertions;
}

declare module "sap/fe/test/ObjectPage" {
  import {
    default as TemplatePage,
    actions as actions1,
    assertions as assertions1,
  } from "sap/fe/test/TemplatePage";

  import { SectionIdentifier, TableIdentifier } from "sap/fe/test/api/BaseAPI";

  import FilterBarActions from "sap/fe/test/api/FilterBarActions";

  import FooterActionsOP from "sap/fe/test/api/FooterActionsOP";

  import { FormIdentifier } from "sap/fe/test/api/FormAPI";

  import FormActions from "sap/fe/test/api/FormActions";

  import HeaderActions from "sap/fe/test/api/HeaderActions";

  import TableActions from "sap/fe/test/api/TableActions";

  import FilterBarAssertions from "sap/fe/test/api/FilterBarAssertions";

  import FooterAssertionsOP from "sap/fe/test/api/FooterAssertionsOP";

  import FormAssertions from "sap/fe/test/api/FormAssertions";

  import HeaderAssertions from "sap/fe/test/api/HeaderAssertions";

  import TableAssertions from "sap/fe/test/api/TableAssertions";

  /**
   * Provides a test page definition for an object page with the corresponding parameters.
   *
   * Sample usage:
   * ```javascript
   *
   * var oObjectPageDefinition = new ObjectPage({ appId: "MyApp", componentId: "MyObjectPageId", entitySet: "MyEntitySet" });
   * ```
   */
  export default class ObjectPage extends TemplatePage {
    /**
     * Constructs a new ObjectPage instance.
     */
    constructor(
      /**
       * The required parameters
       */
      oPageDefinition: {
        /**
         * The app id (defined in the manifest root)
         */
        appId: string;
        /**
         * The component id (defined in the target section for the list report within the manifest)
         */
        componentId: string;
        /**
         * The entitySet (defined in the settings of the corresponding target component within the manifest)
         */
        entitySet: string;
        /**
         * The contextPath (optional)(defined in the settings of the corresponding target component within the manifest)
         */
        contextPath: string;
      },
      /**
       * Additional custom page functions, provided in an object containing `actions` and `assertions`
       */
      ..._aAdditionalPageDefinitions: object[]
    );
    /**
     * ObjectPage actions
     */
    static actions: typeof TemplatePage.actions & actions;

    /**
     * ObjectPage assertions
     */
    static assertions: typeof TemplatePage.assertions & assertions;
  }
  /**
   * ObjectPage actions
   */
  interface actions extends actions1 {
    /**
     * Collapses or expands the page header.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCollapseExpandPageHeader(
      /**
       * Defines whether header should be collapsed, else it gets expanded (default)
       */
      bCollapse?: boolean
    ): object;
    /**
     * Navigates to or focuses on the defined section.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iGoToSection(
      /**
       * The identifier of a section, or its label if passed as an object, the following pattern will be considered
       * (please use to open sub-sections via drop-down):
       * ```javascript
       *
       * {
       *     section: ,
       *     subSection:
       * }
       * ```
       *  to open the editable header section use pattern:
       * ```javascript
       *
       * {
       *     section: "EditableHeaderSection"
       * }
       * ```
       */
      vSectionIdentifier: string | SectionIdentifier
    ): object;
    /**
     * Returns a {@link sap.fe.test.api.FilterBarActions} instance.
     *
     *
     * @returns The available filter bar actions
     */
    onFilterBar(
      /**
       * The identifier of the filterbar
       */
      vFilterBarIdentifier: /* was: sap.fe.test.api.FilterBarIdentifier */
        | any
        | string
    ): FilterBarActions;
    /**
     * Returns a {@link sap.fe.test.api.FooterActionsOP} instance.
     *
     *
     * @returns The available footer actions
     */
    onFooter(): FooterActionsOP;
    /**
     * Returns a {@link sap.fe.test.api.FormActions} instance.
     *
     *
     * @returns The available form actions
     */
    onForm(
      /**
       * The identifier of the form, or its title
       */
      vFormIdentifier: FormIdentifier | string
    ): FormActions;
    /**
     * Returns a {@link sap.fe.test.api.HeaderActions} instance.
     *
     *
     * @returns The available header actions
     */
    onHeader(): HeaderActions;
    /**
     * Returns a {@link sap.fe.test.api.TableActions} instance for the specified table.
     *
     *
     * @returns The available table actions
     */
    onTable(
      /**
       * The identifier of the table, or its header title
       */
      vTableIdentifier: string | TableIdentifier
    ): TableActions;
  }
  export const actions: actions;

  /**
   * ObjectPage assertions
   */
  interface assertions extends assertions1 {
    /**
     * Returns a {@link sap.fe.test.api.FilterBarAssertions} instance.
     *
     *
     * @returns The available filter bar assertions
     */
    onFilterBar(
      /**
       * The identifier of the filterbar
       */
      vFilterBarIdentifier: /* was: sap.fe.test.api.FilterBarIdentifier */
        | any
        | string
    ): FilterBarAssertions;
    /**
     * Returns a {@link sap.fe.test.api.FooterAssertionsOP} instance.
     *
     *
     * @returns The available footer assertions
     */
    onFooter(): FooterAssertionsOP;
    /**
     * Returns a {@link sap.fe.test.api.FormAssertions} instance.
     *
     *
     * @returns The available form actions
     */
    onForm(
      /**
       * The identifier of the form, or its title
       */
      vFormIdentifier: FormIdentifier | string
    ): FormAssertions;
    /**
     * Returns a {@link sap.fe.test.api.HeaderAssertions} instance.
     *
     *
     * @returns The available header assertions
     */
    onHeader(): HeaderAssertions;
    /**
     * Returns a {@link sap.fe.test.api.TableAssertions} instance for the specified table.
     *
     *
     * @returns The available table assertions
     */
    onTable(
      /**
       * The identifier of the table, or its header title
       */
      vTableIdentifier: string | TableIdentifier
    ): TableAssertions;
  }
  export const assertions: assertions;
}

declare module "sap/fe/test/Shell" {
  /**
   * Provides a test page definition for the shell.
   *
   * When using {@link sap.fe.test.JourneyRunner}, this page is made available by default via `onTheShell`.
   */
  export default class Shell {
    /**
     * Constructs a test page definition for the shell.
     */
    constructor(
      /**
       * Additional custom page functions, provided in an object containing `actions` and `assertions`
       */
      ...aAdditionalPageDefinitions: object[]
    );

    /**
     * Check an intent-based navigation. The function checks the semantic object and the action within the URL
     * of an application. Optionally, further URL parameters can be checked.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iCheckIntentBasedNavigation(
      /**
       * The semantic object of the application
       */
      sSemanticObject: string,
      /**
       * The action of the application
       */
      sAction: string,
      /**
       * More URL parameters to be checked. The pattern is:
       * ```javascript
       *
       * 	[{
       * 		property: <expected name of URL parameter>,
       * 		value: <expected value of URL parameter>
       *  }]
       * ```
       */
      aURLParams?: any[]
    ): object;
    /**
     * Navigates back via shell back button.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iNavigateBack(): object;
    /**
     * Navigates to the launch pad via the home button.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iNavigateHome(): object;
    /**
     * Navigates via a navigation item in the shell's navigation menu.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iNavigateViaMenu(
      /**
       * The label of the navigation item
       */
      sItem: string
    ): object;
    /**
     * Opens the navigation menu in the shell header.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iOpenNavigationMenu(
      /**
       * The title of the navigation menu to be clicked. If undefined the menu is identified by the internal id
       * only.
       */
      NavigationTitle?: string
    ): object;
    /**
     * Selecting a tile in the launchpad by its target app, for example `iPressTile("SalesOrder-manage")`.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iPressTile(
      /**
       * The target application (hash)
       */
      sTarget: string
    ): object;
  }
}

declare module "sap/fe/test/TemplatePage" {
  import DialogActions from "sap/fe/test/api/DialogActions";

  import DialogCreateActions from "sap/fe/test/api/DialogCreateActions";

  import { DialogIdentifier } from "sap/fe/test/api/BaseAPI";

  import DialogMessageActions from "sap/fe/test/api/DialogMessageActions";

  import DialogValueHelpActions from "sap/fe/test/api/DialogValueHelpActions";

  import DialogAssertions from "sap/fe/test/api/DialogAssertions";

  import DialogCreateAssertions from "sap/fe/test/api/DialogCreateAssertions";

  import DialogMessageAssertions from "sap/fe/test/api/DialogMessageAssertions";

  import DialogValueHelpAssertions from "sap/fe/test/api/DialogValueHelpAssertions";

  /**
   * Provides a test page definition for a template page with the corresponding parameters.
   */
  export default class TemplatePage {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();
    /**
     * TemplatePage actions
     */
    static actions: actions;

    /**
     * Assertions that are available to all template pages used in SAP Fiori elements.
     */
    static assertions: assertions;
  }
  /**
   * TemplatePage actions
   */
  interface actions {
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Action })`.
     *
     *
     * @returns The available dialog actions
     */
    onActionDialog(): DialogActions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Confirmation })`.
     *
     *
     * @returns The available dialog actions
     */
    onConfirmationDialog(): DialogActions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Create })`.
     *
     *
     * @returns The available dialog actions
     */
    onCreateDialog(): DialogCreateActions;
    /**
     * Returns a {@link sap.fe.test.api.DialogActions} instance.
     *
     *
     * @returns The available dialog actions
     */
    onDialog(
      /**
       * The identifier of the dialog, or its title
       */
      vDialogIdentifier: string | DialogIdentifier
    ): DialogActions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Error })`.
     *
     *
     * @returns The available dialog actions
     */
    onErrorDialog(): DialogActions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Message })`.
     *
     *
     * @returns The available dialog actions
     */
    onMessageDialog(): DialogMessageActions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.ValueHelp })`.
     *
     *
     * @returns The available dialog actions
     */
    onValueHelpDialog(): DialogValueHelpActions;
  }
  export const actions: actions;

  /**
   * Assertions that are available to all template pages used in SAP Fiori elements.
   */
  interface assertions {
    /**
     * Confirms the visibility of the current page.
     *
     *
     * @returns The result of the {@link sap.ui.test.Opa5#waitFor} function, to be used for chained statements
     */
    iSeeThisPage(): object;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Action })`.
     *
     *
     * @returns The available dialog assertions
     */
    onActionDialog(): DialogAssertions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Confirmation })`.
     *
     *
     * @returns The available dialog assertions
     */
    onConfirmationDialog(): DialogAssertions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Create })`.
     *
     *
     * @returns The available dialog assertions
     */
    onCreateDialog(): DialogCreateAssertions;
    /**
     * Returns a {@link sap.fe.test.api.DialogAssertions} instance.
     *
     *
     * @returns The available dialog actions
     */
    onDialog(
      /**
       * The identifier of the dialog, or its title
       */
      vDialogIdentifier: string | DialogIdentifier
    ): DialogAssertions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Error })`.
     *
     *
     * @returns The available dialog assertions
     */
    onErrorDialog(): DialogAssertions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.Message })`.
     *
     *
     * @returns The available dialog assertions
     */
    onMessageDialog(): DialogMessageAssertions;
    /**
     * Shortcut for `onDialog({ type: sap.fe.test.api.DialogType.ValueHelp })`.
     *
     *
     * @returns The available dialog assertions
     */
    onValueHelpDialog(): DialogValueHelpAssertions;
  }
  export const assertions: assertions;
}

declare namespace sap {
  interface IUI5DefineDependencyNames {
    "sap/fe/test/api/BaseAPI": undefined;

    "sap/fe/test/api/DialogActions": undefined;

    "sap/fe/test/api/DialogAssertions": undefined;

    "sap/fe/test/api/DialogCreateActions": undefined;

    "sap/fe/test/api/DialogCreateAssertions": undefined;

    "sap/fe/test/api/DialogMessageActions": undefined;

    "sap/fe/test/api/DialogMessageAssertions": undefined;

    "sap/fe/test/api/DialogType": undefined;

    "sap/fe/test/api/DialogValueHelpActions": undefined;

    "sap/fe/test/api/DialogValueHelpAssertions": undefined;

    "sap/fe/test/api/EditState": undefined;

    "sap/fe/test/api/FilterBarActions": undefined;

    "sap/fe/test/api/FilterBarAPI": undefined;

    "sap/fe/test/api/FilterBarAssertions": undefined;

    "sap/fe/test/api/FooterActionsBase": undefined;

    "sap/fe/test/api/FooterActionsOP": undefined;

    "sap/fe/test/api/FooterAssertionsBase": undefined;

    "sap/fe/test/api/FooterAssertionsOP": undefined;

    "sap/fe/test/api/FormActions": undefined;

    "sap/fe/test/api/FormAPI": undefined;

    "sap/fe/test/api/FormAssertions": undefined;

    "sap/fe/test/api/HeaderActions": undefined;

    "sap/fe/test/api/HeaderActionsLR": undefined;

    "sap/fe/test/api/HeaderAPI": undefined;

    "sap/fe/test/api/HeaderAssertions": undefined;

    "sap/fe/test/api/HeaderAssertionsLR": undefined;

    "sap/fe/test/api/TableActions": undefined;

    "sap/fe/test/api/TableAPI": undefined;

    "sap/fe/test/api/TableAssertions": undefined;

    "sap/fe/test/BaseActions": undefined;

    "sap/fe/test/BaseArrangements": undefined;

    "sap/fe/test/BaseAssertions": undefined;

    "sap/fe/test/JourneyRunner": undefined;

    "sap/fe/test/library": undefined;

    "sap/fe/test/ListReport": undefined;

    "sap/fe/test/ObjectPage": undefined;

    "sap/fe/test/Shell": undefined;

    "sap/fe/test/TemplatePage": undefined;
  }
}
