import { SaxEventType, Tag as SaxTag, Text as SaxText } from "sax-wasm";
import { taskStart } from "../../utils/perf.js";
import Parser from "./Parser.js";
import { getLogger } from "@ui5/logger";
import { MESSAGE } from "../messages.js";
import { loadApiExtract } from "../../utils/ApiExtract.js";
import { parseXml, initSaxWasm } from "../../utils/xmlParser.js";
const log = getLogger("linter:xmlTemplate:transpiler");
let apiExtract;
export default async function transpileXml(resourcePath, contentStream, context, controllerByIdInfo) {
    await init();
    try {
        const taskEnd = taskStart("Transpile XML", resourcePath, true);
        const res = await transpileXmlToJs(resourcePath, contentStream, context, controllerByIdInfo);
        taskEnd();
        if (!res.source) {
            log.verbose(`XML transpiler returned no result for ${resourcePath}`);
            return res;
        }
        return res;
    }
    catch (err) {
        const message = err instanceof Error ? err.message : String(err);
        context.addLintingMessage(resourcePath, { id: MESSAGE.PARSING_ERROR, args: { message } });
    }
}
let initializing;
async function init() {
    // eslint-disable-next-line @typescript-eslint/no-misused-promises
    if (initializing) {
        return initializing;
    }
    const taskEnd = taskStart("XML Transpiler initialization");
    return initializing = Promise.all([
        loadApiExtract(),
        initSaxWasm(),
    ]).then(([apiExtractRes]) => {
        apiExtract = apiExtractRes;
        taskEnd();
    });
}
async function transpileXmlToJs(resourcePath, contentStream, context, controllerByIdInfo) {
    const parser = new Parser(resourcePath, apiExtract, context, controllerByIdInfo);
    await parseXml(contentStream, (event, data) => {
        if (data instanceof SaxTag) {
            if (event === SaxEventType.OpenTag) {
                parser.pushTag(data.toJSON());
            }
            else if (event === SaxEventType.CloseTag) {
                parser.popTag(data.toJSON());
            }
        }
        else if (data instanceof SaxText && event === SaxEventType.Comment) {
            parser.parseComment(data.toJSON());
        }
    });
    return parser.generate();
}
//# sourceMappingURL=transpiler.js.map